
package com.crawler.excel;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import javax.servlet.http.HttpServletResponse;

import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFCellStyle;
import org.apache.poi.hssf.usermodel.HSSFClientAnchor;
import org.apache.poi.hssf.usermodel.HSSFFont;
import org.apache.poi.hssf.usermodel.HSSFPalette;
import org.apache.poi.hssf.usermodel.HSSFPatriarch;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.hssf.util.HSSFColor;
import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.apache.poi.poifs.filesystem.POIFSFileSystem;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.xssf.usermodel.XSSFCell;
import org.apache.poi.xssf.usermodel.XSSFRow;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.multipart.MultipartFile;

import com.crawler.exception.ExcelException;

import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

/**
 * excel 工具
 * 
 * @author rubekid
 *
 */
public class ExcelUtil {

	/**
	 * LOGGER
	 */
	private static final Logger LOGGER = LoggerFactory.getLogger(ExcelUtil.class);

	/**
	 * 解析excel
	 * 
	 * @param <T>
	 * @param multipartFile
	 * @param clazz
	 * @param keyConfig
	 *            key和中文名称映射
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public static <T> List<T> parse(MultipartFile multipartFile, Class<T> clazz, Map<String, String> keyConfig) {
		List<T> list = new ArrayList<T>();
		try {
			JSONArray array = parse(multipartFile);
			JSONArray labels = array.getJSONArray(0);

			for (int i = 1; i < array.size(); i++) {
				JSONArray data = array.getJSONArray(i);
				JSONObject item = new JSONObject();
				for (int j = 0; j < data.size(); j++) {
					String label = labels.getString(j);
					if (keyConfig.containsKey(label)) {
						String key = keyConfig.get(label);
						item.put(key, data.getString(j));
					}
				}
				list.add((T) JSONObject.toBean(item, clazz));
			}
		} catch (IllegalStateException | InvalidFormatException | IOException e) {
			throw new ExcelException("Excel解析失败");
		}
		return list;
	}

	/**
	 * 解析 excel
	 * 
	 * @param multipartFile
	 * @return
	 * @throws FileNotFoundException
	 * @throws IOException
	 * @throws IllegalStateException
	 * @throws InvalidFormatException
	 * @throws Exception
	 */
	public static JSONArray parse(MultipartFile multipartFile)
			throws FileNotFoundException, IOException, InvalidFormatException {
		String name = multipartFile.getOriginalFilename();
		String ext = name.substring(name.lastIndexOf(".") + 1);
		if ("xls".equalsIgnoreCase(ext)) {
			return parseXLS(multipartFile.getInputStream());
		} else if ("csv".equalsIgnoreCase(ext)) {
			return parseCsv(multipartFile.getInputStream());
		} else if ("xlsx".equalsIgnoreCase(ext)) {
			return parseXLSX(multipartFile.getInputStream());
		} else {
			throw new ExcelException("不是有效的excel文件");
		}
	}

	/**
	 * 解析 excel
	 * 
	 * @param file
	 * @return
	 * @throws IOException
	 * @throws InvalidFormatException
	 * @throws Exception
	 */
	public static JSONArray parse(File file) throws InvalidFormatException, IOException {
		String name = file.getName();
		String ext = name.substring(name.lastIndexOf(".") + 1);
		if ("xls".equalsIgnoreCase(ext)) {
			return parseXLS(file);
		} else if ("csv".equalsIgnoreCase(ext)) {
			return parseCsv(file);
		} else if ("xlsx".equalsIgnoreCase(ext)) {
			return parseXLSX(file);
		} else {
			throw new ExcelException("不是有效的excel文件");
		}
	}

	/**
	 * 解析 xls
	 * 
	 * @param file
	 * @return
	 * @throws FileNotFoundException
	 * @throws IOException
	 */
	public static JSONArray parseXLS(File file) throws FileNotFoundException, IOException {
		return parseXLS(new FileInputStream(file));
	}

	/**
	 * 解析 csv
	 * 
	 * @param file
	 * @return
	 * @throws FileNotFoundException
	 * @throws IOException
	 */
	public static JSONArray parseCsv(File file) throws FileNotFoundException, IOException {
		return parseCsv(new FileInputStream(file));
	}

	/**
	 * 解析 xlsx
	 * 
	 * @param file
	 * @return
	 * @throws InvalidFormatException
	 * @throws IOException
	 */
	public static JSONArray parseXLSX(File file) throws InvalidFormatException, IOException {
		return parseXLSX(new FileInputStream(file));
	}

	/**
	 * 解析 xls
	 * 
	 * @param inputStream
	 * @return
	 * @throws FileNotFoundException
	 * @throws IOException
	 */
	public static JSONArray parseXLS(InputStream inputStream) throws FileNotFoundException, IOException {
		JSONArray array = new JSONArray();
		POIFSFileSystem poifsFileSystem = new POIFSFileSystem(inputStream);
		HSSFWorkbook hssfWorkbook = new HSSFWorkbook(poifsFileSystem);
		HSSFSheet hssfSheet = hssfWorkbook.getSheetAt(0);

		int rowstart = hssfSheet.getFirstRowNum();
		int rowEnd = hssfSheet.getLastRowNum();
		for (int i = rowstart; i <= rowEnd; i++) {
			HSSFRow row = hssfSheet.getRow(i);
			if (null == row)
				continue;
			int cellStart = row.getFirstCellNum();
			int cellEnd = row.getLastCellNum();
			JSONArray jsonRow = new JSONArray();
			for (int k = cellStart; k <= cellEnd; k++) {
				HSSFCell cell = row.getCell(k);
				if (null == cell)
					continue;
				jsonRow.add(getValue(cell));
			}
			array.add(jsonRow);
		}
		hssfWorkbook.close();
		return array;
	}

	/**
	 * 解析csv
	 * 
	 * @param inputStream
	 * @return
	 * @throws FileNotFoundException
	 * @throws IOException
	 */
	public static JSONArray parseCsv(InputStream inputStream) throws IOException {
		JSONArray array = new JSONArray();
		BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream, "GBK"));
		
		String line = null;
		while ((line = reader.readLine()) != null) {
			String[] items = line.split(",");// CSV格式文件为逗号分隔符文件，这里根据逗号切分
			
			JSONArray jsonRow = new JSONArray();
			for (String str : items) {
				jsonRow.add(str);
			}
			array.add(jsonRow);
		}

		
		return array;
	}

	/**
	 * 解析XLSX
	 * 
	 * @param inputStream
	 * @return
	 * @throws InvalidFormatException
	 * @throws IOException
	 */
	public static JSONArray parseXLSX(InputStream inputStream) throws InvalidFormatException, IOException {
		JSONArray array = new JSONArray();
		XSSFWorkbook xssfWorkbook = new XSSFWorkbook(inputStream);
		XSSFSheet xssfSheet = xssfWorkbook.getSheetAt(0);

		int rowstart = xssfSheet.getFirstRowNum();
		int rowEnd = xssfSheet.getLastRowNum();
		for (int i = rowstart; i <= rowEnd; i++) {
			XSSFRow row = xssfSheet.getRow(i);
			if (null == row)
				continue;
			int cellStart = row.getFirstCellNum();
			int cellEnd = row.getLastCellNum();
			JSONArray jsonRow = new JSONArray();
			for (int k = cellStart; k <= cellEnd; k++) {
				XSSFCell cell = row.getCell(k);
				if (null == cell)
					continue;
				jsonRow.add(getValue(cell));
			}
			array.add(jsonRow);
		}
		xssfWorkbook.close();
		return array;
	}

	/**
	 * 
	 * <p>
	 * Description: 导出excel
	 * </p>
	 * <p>
	 * Create Time: 2016年2月29日
	 * </p>
	 * <p>
	 * Create author: Administrator
	 * </p>
	 * 
	 * @param excelBean
	 * @param response
	 * @throws IOException
	 * @throws Exception
	 */
	public static void export(ExcelBean excelBean, HttpServletResponse response) {
		try {
			String filename = excelBean.getName();
			filename = new String(filename.replaceAll("\\s|;", "").getBytes("gbk"), "ISO8859-1");

			response.setContentType("application/octet-stream;charset=utf-8");
			response.setCharacterEncoding("utf-8");
			response.setContentType("application/vnd.ms-excel");
			response.setHeader("Content-disposition", "attachment;filename=" + filename);
			OutputStream outputStream = response.getOutputStream();
			createExcel(excelBean, outputStream);
			outputStream.flush();
			outputStream.close();
		} catch (IOException ex) {
			LOGGER.error(ex.getMessage(), ex);
			throw new ExcelException("导出excel失败");
		}
	}

	/**
	 * 
	 * <p>
	 * Description: 创建excel
	 * </p>
	 * <p>
	 * Create Time: 2016年2月29日
	 * </p>
	 * <p>
	 * Create author: Administrator
	 * </p>
	 * 
	 * @param excelBean
	 * @param outputStream
	 * @throws IOException
	 */
	public static void createExcel(ExcelBean excelBean, OutputStream outputStream) throws IOException {
		HSSFWorkbook wb = new HSSFWorkbook();
		HSSFSheet sheet = wb.createSheet(excelBean.getSheetName());
		HSSFRow row = sheet.createRow(0);

		// 设置样式
		HSSFFont titleFont = wb.createFont();
		titleFont.setColor(HSSFColor.GREEN.index);
		titleFont.setBoldweight(HSSFFont.BOLDWEIGHT_BOLD);
		titleFont.setFontHeight(getRowHeight(14));
		HSSFCellStyle titleStyle = wb.createCellStyle();
		titleStyle.setFont(titleFont);
		setTitleStyle(wb, titleStyle);
		addBorder(titleStyle);

		HSSFCellStyle cellStyle = wb.createCellStyle();
		setAlignCenter(cellStyle);
		setVerticalCenter(cellStyle);
		cellStyle.setWrapText(true);
		HSSFFont font = wb.createFont();
		font.setFontHeight(getRowHeight(14));
		cellStyle.setFont(font);

		HSSFCell cell;
		ExcelTitle[] titles = excelBean.getTitles();
		for (int i = 0; i < titles.length; i++) {
			ExcelTitle title = titles[i];
			cell = row.createCell(i);
			cell.setCellValue(title.getValue());
			cell.setCellStyle(titleStyle);
			int columnWidth = title.getWidth() > 0 ? title.getWidth() : excelBean.getColumnWidth();
			sheet.setColumnWidth(i, getColWidth(columnWidth));
		}

		int rowIndex = 1;
		int rowHeight = excelBean.getRowHeight();
		HSSFPatriarch patriarch = sheet.createDrawingPatriarch();

		for (String[] data : excelBean.getDataList()) {
			row = sheet.createRow(rowIndex++);
			if (rowHeight > 0) {
				row.setHeight((short) getRowHeight(rowHeight));
			} else {
				rowHeight = 18;
			}
			for (int j = 0; j < data.length; j++) {
				String value = data[j];
				if (value == null || "null".equalsIgnoreCase(value)) {
					value = "";
				}
				cell = row.createCell(j);

				if (isUrl(value)) {
					if (isImage(value)) {
						int columnWidth = titles[j].getWidth() > 0 ? titles[j].getWidth() : excelBean.getColumnWidth();
						insertImage(wb, patriarch, value, rowHeight, columnWidth, (short) (j), rowIndex);
					} else {
						setLink(wb, cell, value);
					}
				} else {
					cell.setCellValue(value);
					cellStyle.setVerticalAlignment(HSSFCellStyle.VERTICAL_CENTER);
					cell.setCellStyle(cellStyle);
				}
			}
		}

		wb.write(outputStream);
		wb.close();
	}

	/**
	 * 
	 * <p>
	 * Description: 设置连接
	 * </p>
	 * <p>
	 * Create Time: 2016年2月29日
	 * </p>
	 * <p>
	 * Create author: Administrator
	 * </p>
	 * 
	 * @param wb
	 * @param cell
	 * @param value
	 */
	public static void setLink(HSSFWorkbook wb, HSSFCell cell, String value) {
		cell.setCellValue(value);
		cell.setCellFormula("HYPERLINK(\"" + value + "\",\"" + value + "\")");
		HSSFCellStyle linkStyle = wb.createCellStyle();
		HSSFFont cellFont = wb.createFont();
		cellFont.setUnderline((byte) 1);
		cellFont.setColor(HSSFColor.BLUE.index);
		linkStyle.setFont(cellFont);
		linkStyle.setVerticalAlignment(HSSFCellStyle.VERTICAL_CENTER);
		cell.setCellStyle(linkStyle);
	}

	/**
	 * 
	 * <p>
	 * Description: 插入图片
	 * </p>
	 * <p>
	 * Create Time: 2016年2月29日
	 * </p>
	 * <p>
	 * Create author: Administrator
	 * </p>
	 * 
	 * @param patriarch
	 * @param value
	 * @param rowHeight
	 * @param columnWidth
	 * @param col
	 * @param rowIndex
	 * @throws IOException
	 */
	public static void insertImage(HSSFWorkbook wb, HSSFPatriarch patriarch, String value, int rowHeight,
			int columnWidth, short col, int rowIndex) throws IOException {
		ExcelImage excelImage = new ExcelImage(value);
		ExcelBox excelBox = new ExcelBox(excelImage.getWidth(), excelImage.getHeight(), rowHeight, columnWidth, 10);
		HSSFClientAnchor anchor = new HSSFClientAnchor();
		int cw = getColWidth(columnWidth);
		int rh = getRowHeight(rowHeight);
		int rowNum = rowIndex - 1;
		anchor.setDx1(getAnchorX(excelBox.getX1(), cw));
		anchor.setDy1(getAnchorY(excelBox.getY1(), rh));
		anchor.setDx2(getAnchorX(excelBox.getX2(), cw));
		anchor.setDy2(getAnchorY(excelBox.getY2(), rh));
		anchor.setCol1(col);
		anchor.setRow1(rowNum);
		anchor.setCol2(col);
		anchor.setRow2(rowNum);

		anchor.setAnchorType(0);
		patriarch.createPicture(anchor, wb.addPicture(excelImage.getByteArray(), HSSFWorkbook.PICTURE_TYPE_JPEG));
	}

	/**
	 * <p>
	 * Description:设置标题栏样式
	 * </p>
	 * <p>
	 * Create Time: 2016年2月27日
	 * </p>
	 * <p>
	 * Create author: Administrator
	 * </p>
	 * 
	 * @param style
	 */
	public static void setTitleStyle(HSSFWorkbook workbook, HSSFCellStyle style) {
		style.setAlignment(HSSFCellStyle.ALIGN_CENTER);
		style.setVerticalAlignment(HSSFCellStyle.VERTICAL_CENTER);
		style.setFillPattern(HSSFCellStyle.SOLID_FOREGROUND); // 填充单元格
		HSSFPalette palette = workbook.getCustomPalette();
		palette.setColorAtIndex(HSSFColor.LIGHT_GREEN.index, (byte) 226, (byte) 239, (byte) 218);
		style.setFillForegroundColor(HSSFColor.LIGHT_GREEN.index);
	}

	/**
	 * 
	 * <p>
	 * Description: 设置边框
	 * </p>
	 * <p>
	 * Create Time: 2016年2月27日
	 * </p>
	 * <p>
	 * Create author: Administrator
	 * </p>
	 * 
	 * @param style
	 */
	public static void addBorder(HSSFCellStyle style) {
		style.setBorderBottom(HSSFCellStyle.BORDER_THIN); // 下边框
		style.setBorderLeft(HSSFCellStyle.BORDER_THIN);// 左边框
		style.setBorderTop(HSSFCellStyle.BORDER_THIN);// 上边框
		style.setBorderRight(HSSFCellStyle.BORDER_THIN);// 右边框
	}

	/**
	 * 
	 * <p>
	 * Description: 设置水平居中样式
	 * </p>
	 * <p>
	 * Create Time: 2016年2月29日
	 * </p>
	 * <p>
	 * Create author: Administrator
	 * </p>
	 * 
	 * @param style
	 */
	public static void setAlignCenter(HSSFCellStyle style) {
		style.setAlignment(HSSFCellStyle.ALIGN_CENTER);
	}

	/**
	 * 
	 * <p>
	 * Description: 设置水平居左样式
	 * </p>
	 * <p>
	 * Create Time: 2016年2月29日
	 * </p>
	 * <p>
	 * Create author: Administrator
	 * </p>
	 * 
	 * @param style
	 */
	public static void setAlignLeft(HSSFCellStyle style) {
		style.setAlignment(HSSFCellStyle.ALIGN_LEFT);
	}

	/**
	 * 
	 * <p>
	 * Description: 设置水平居右样式
	 * </p>
	 * <p>
	 * Create Time: 2016年2月29日
	 * </p>
	 * <p>
	 * Create author: Administrator
	 * </p>
	 * 
	 * @param style
	 */
	public static void setAlignRight(HSSFCellStyle style) {
		style.setAlignment(HSSFCellStyle.ALIGN_RIGHT);
	}

	/**
	 * 
	 * <p>
	 * Description: 垂直居中
	 * </p>
	 * <p>
	 * Create Time: 2016年2月27日
	 * </p>
	 * <p>
	 * Create author: Administrator
	 * </p>
	 * 
	 * @param style
	 */
	public static void setVerticalCenter(HSSFCellStyle style) {
		style.setVerticalAlignment(HSSFCellStyle.VERTICAL_CENTER);
	}

	/**
	 * 
	 * <p>
	 * Description: 垂直置顶
	 * </p>
	 * <p>
	 * Create Time: 2016年2月27日
	 * </p>
	 * <p>
	 * Create author: Administrator
	 * </p>
	 * 
	 * @param style
	 */
	public static void setVerticalTop(HSSFCellStyle style) {
		style.setVerticalAlignment(HSSFCellStyle.VERTICAL_TOP);
	}

	/**
	 * 
	 * <p>
	 * Description: 垂直置底
	 * </p>
	 * <p>
	 * Create Time: 2016年2月27日
	 * </p>
	 * <p>
	 * Create author: Administrator
	 * </p>
	 * 
	 * @param style
	 */
	public static void setVerticalBottom(HSSFCellStyle style) {
		style.setVerticalAlignment(HSSFCellStyle.VERTICAL_BOTTOM);
	}

	/**
	 * 获取图片x方向长度坐标转换
	 * 
	 * @param px
	 * @param colWidth
	 * @return
	 */
	public static int getAnchorX(int px, int colWidth) {
		return (int) Math.round(((double) 701 * 16000.0 / 301) * ((double) 1 / colWidth) * px);
	}

	/**
	 * 获取图片y方向长度坐标转换
	 * 
	 * @param px
	 * @param rowHeight
	 * @return
	 */
	public static int getAnchorY(int px, int rowHeight) {
		return (int) Math.round(((double) 144 * 8000 / 301) * ((double) 1 / rowHeight) * px);
	}

	/**
	 * 行高转换
	 * 
	 * @param px
	 * @return
	 */
	public static short getRowHeight(int px) {
		return (short) Math.round(((double) 4480 / 300) * px);
	}

	/**
	 * 列宽转换
	 * 
	 * @param px
	 * @return
	 */
	public static int getColWidth(int px) {
		return (int) Math.round(((double) 10971 / 300) * px);
	}

	/**
	 * 判断是否为链接地址
	 */
	public static boolean isUrl(String string) {
		Pattern pattern = Pattern.compile(
				"^((http|https):\\/\\/([\\w\\-]+\\.)+[\\w\\-]+(\\/[\\w\\u4e00-\\u9fa5\\-\\.\\/?\\@\\%\\!\\&=\\+\\~\\:\\#\\;\\,]*)?)",
				Pattern.CASE_INSENSITIVE);
		return pattern.matcher(string).matches();
	}

	/**
	 * 判断是否为图片
	 */
	public static boolean isImage(String string) {
		Pattern pattern = Pattern.compile("\\S+\\.(jpg|jpeg|png|gif|bmp)(\\?\\S+)?$", Pattern.CASE_INSENSITIVE);
		return isUrl(string) && pattern.matcher(string).matches();
	}
	
	/**
	 * 获取excel 单元值
	 * @param cell
	 * @return
	 */
	private static Object getValue(Cell cell) {
		Object value;
		switch (cell.getCellType()) {
		case HSSFCell.CELL_TYPE_NUMERIC:
			try {
				value = new DecimalFormat("#").format(cell.getNumericCellValue());
			} catch (ExcelException ex) {
				value = cell.getNumericCellValue();
			}
			break;
		case HSSFCell.CELL_TYPE_STRING:
			value = cell.getStringCellValue();
			break;
		case HSSFCell.CELL_TYPE_BOOLEAN:
			value = cell.getBooleanCellValue();
			break;
		case HSSFCell.CELL_TYPE_FORMULA:
			value = cell.getCellFormula();
			break;
		case HSSFCell.CELL_TYPE_BLANK:
		case HSSFCell.CELL_TYPE_ERROR:
		default:
			value = "";
			break;
		}
		return value;
	}
}
