package com.crawler.utils;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Enumeration;
import java.util.List;

import org.apache.tools.zip.ZipEntry;
import org.apache.tools.zip.ZipFile;
import org.apache.tools.zip.ZipOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.multipart.MultipartFile;

import com.crawler.beans.ZipItem;


/**
 * zip 工具
 * @author rubekid
 *
 */
public class ZipUtils {
	
	private static Logger logger = LoggerFactory.getLogger(ZipUtils.class);
	
	/**使用GBK编码可以避免压缩中文文件名乱码*/
    private static final String CHINESE_CHARSET = "GBK";
    
    /**文件读取缓冲区大小*/
    private static final int CACHE_SIZE = 1024;
    
    /**
     * 解压
     * @param multipartFile
     * @param basePath
     * @return
     * @throws IllegalStateException
     * @throws IOException
     */
    public static File unzip(MultipartFile multipartFile, String basePath) throws IllegalStateException, IOException{
    	String path = basePath + "/" + System.currentTimeMillis() + StringUtils.getRandomString(8);
		String fileName =  path + ".zip";
		File file = new File(fileName);
		file.createNewFile();
		multipartFile.transferTo(file);
		File directory = new File(path);
		directory.mkdirs();
		ZipUtils.unzip(file.getAbsolutePath(), path + "/");
		file.delete();
		return directory;
    }
    
    /**
     * 解压
     * @param zipFilePath
     * @param path
     */
	public static void unzip(String zipFilePath, String path){
		if(!path.endsWith(File.separator)){
			path += File.separator;
		}
		ZipFile zipFile = null;
        try {
            BufferedInputStream bis = null;
            FileOutputStream fos = null;
            BufferedOutputStream bos = null;
            zipFile = new ZipFile(zipFilePath, CHINESE_CHARSET);
            Enumeration<ZipEntry> zipEntries = zipFile.getEntries();
            File file, parentFile;
            ZipEntry entry;
            byte[] cache = new byte[CACHE_SIZE];
            while (zipEntries.hasMoreElements()) {
                entry = (ZipEntry) zipEntries.nextElement();
                if (entry.isDirectory()) {
                    new File(path + entry.getName()).mkdirs();
                    continue;
                }
                try {
                	 bis = new BufferedInputStream(zipFile.getInputStream(entry));
                     file = new File(path + entry.getName());
                     parentFile = file.getParentFile();
                     if (parentFile != null && (!parentFile.exists())) {
                         parentFile.mkdirs();
                     }
                     fos = new FileOutputStream(file);
                     bos = new BufferedOutputStream(fos, CACHE_SIZE);
                     int readIndex = 0;
                     while ((readIndex = bis.read(cache, 0, CACHE_SIZE)) != -1) {
                         fos.write(cache, 0, readIndex);
                     }
                     bos.flush();
                }
                finally {
                	if (bos != null) {
                		try {
                			bos.close();
                		} catch (IOException e) {
                			logger.error(e.getMessage(), e);
                		}
                	}
                	if (fos != null) {
                		try {
                			fos.close();
                		} catch (IOException e) {
                			logger.error(e.getMessage(), e);
                		}
                	}
                	if (bis != null) {
                		try {
                			bis.close();
                		} catch (IOException e) {
                			logger.error(e.getMessage(), e);
                		}
                	}
				}
               
                
            }
        } catch (IOException e) {
            logger.error(e.getMessage(), e);
        }finally{
            if(zipFile != null){
                try {
                    zipFile.close();
                } catch (IOException e) {
                    logger.error(e.getMessage(), e);
                }
            }
        }
	}
	
	
	
	/**
	 * <p>Description: 常见压缩包 </p>
	 * <p>Create Time: 2016年2月26日   </p>
	 * <p>Create author: Administrator   </p>
	 * @param resourcesPath
	 * @param targetPath
	 * @throws Exception
	 */
    public static ByteArrayOutputStream compress(String resourcesPath) throws Exception{
        File resourcesFile = new File(resourcesPath);
        
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        ZipOutputStream out = null;
        try {
        	out = new ZipOutputStream(new BufferedOutputStream(byteArrayOutputStream));
        	createCompressedFile(out, resourcesFile, "");        	
        }
        finally {
        	if(out != null) {
        		try {
                	out.close(); 
				} catch (IOException e) {
					logger.error(e.getMessage(), e);
				}
        	}
			
		}
        return byteArrayOutputStream;
    }

    /**
     * <p>Description:生成压缩文件 </p>
     * <p>Create Time: 2016年2月26日   </p>
     * <p>Create author: Administrator   </p>
     * @param out
     * @param file
     * @param dir
     * @throws Exception
     */
    public static void createCompressedFile(ZipOutputStream out, File file, String dir) throws Exception{
        if(dir == null){
            dir = "";
        }
        if(file.isDirectory()){
            File[] files = file.listFiles();
            if(files.length > 0){
                dir = "".equals(dir) ? "" : dir +"/";
                for(int i = 0 ; i < files.length ; i++){
                    createCompressedFile(out, files[i], dir + files[i].getName());
                }
            }
            else{
                if(!"".equals(dir)){
                    out.putNextEntry(new ZipEntry(dir + "/"));
                }
            }
            
            
        }
        else{
            FileInputStream fis = null;
            try {
            	fis = new FileInputStream(file);
            	out.putNextEntry(new ZipEntry(dir));
                int count =  0;
                byte[] buffer = new byte[1024];
                while((count = fis.read(buffer)) > 0){                    
                    out.write(buffer, 0, count);
                }
            }
            finally {
            	if(fis != null) {
            		try {
            			fis.close();
    				} catch (IOException e) {
    					logger.error(e.getMessage(), e);
    				}
            	}
    			
    		}
        }
    }
    
    /**
     * 生成压缩文件字节数组流
     * @param zipItems
     * @return
     * @throws IOException
     */
    public static ByteArrayOutputStream createCompressedFile(List<ZipItem> zipItems) throws IOException{
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        ZipOutputStream out = null;
        try {
        	out = new ZipOutputStream(new BufferedOutputStream(outputStream));
            for(ZipItem zipItem : zipItems){
                if(zipItem.isFile()){
                    out.putNextEntry(new ZipEntry(zipItem.getPath()));
                    int count =  0;
                    byte[] buffer = new byte[1024];
                    while((count = zipItem.getInputStream().read(buffer)) > 0){                    
                        out.write(buffer,0, count);
                    }
                }
                else{
                    out.putNextEntry(new ZipEntry(zipItem.getPath() + "/"));
                }
            }
            out.flush();
            
        }
        finally {
        	if(out != null) {
        		try {
        			out.close();
				} catch (IOException e) {
					logger.error(e.getMessage(), e);
				}
        	}
		}
        return outputStream;
    }

}
