package com.crawler.http;

import java.io.File;
import java.io.IOException;
import java.net.SocketTimeoutException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.http.Consts;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.ParseException;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.CookieStore;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.conn.ConnectTimeoutException;
import org.apache.http.cookie.Cookie;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.FileBody;
import org.apache.http.impl.client.BasicCookieStore;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.cookie.BasicClientCookie;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;

import com.crawler.config.ConfigProperties;
import com.crawler.config.JsonMapper;
import com.crawler.exception.SimpleRuntimeException;
import com.crawler.utils.StringUtils;

import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

/**
 * 简易http请求
 * @author rubekid
 * @date 2016年10月11日
 */
public class SimpleHttpClient {
	
	private static final Logger LOGGER = LoggerFactory.getLogger(SimpleHttpClient.class);
	
	private static final Integer TIMEOUT = ConfigProperties.getPropertyForInteger("httpclient.timeout", "10000");

	private static CloseableHttpClient httpClient;
	
	private static CookieStore cookieStore;

	static {
		cookieStore  = new BasicCookieStore();
		RequestConfig requestConfig = RequestConfig.custom().setSocketTimeout(TIMEOUT).setConnectTimeout(TIMEOUT).build();	
		// 将CookieStore设置到httpClient中
		httpClient = HttpClients.custom().setDefaultRequestConfig(requestConfig).setDefaultCookieStore(cookieStore).build();
	}
	
	/**
	 * 获取cookie
	 * @param name
	 * @return
	 */
	public static String getCookie(String name){
		List<Cookie> cookies =  cookieStore.getCookies();
		for(Cookie cookie : cookies){
			if(cookie.getName().equalsIgnoreCase(name)){
				return cookie.getValue();
			}
		}
		return null;
		
	}
	
	
	/**
	 * 设置cookie
	 * @param name
	 * @param value
	 */
	public static void setCookie(String name, String value){
		BasicClientCookie cookie = new BasicClientCookie(name, value);
		cookieStore.getCookies().add(cookie);
	}
	
	/**
	 * GET 请求
	 * @param url
	 * @return
	 * @throws ClientProtocolException
	 * @throws IOException
	 */
	public static String get(String url){
		return get(url, null);
	}

	/**
	 * GET 请求
	 * @param url
	 * @return
	 * @throws ClientProtocolException
	 * @throws IOException
	 */
	public static String get(String url, Map<String, String> headers){
		HttpResponse httpResponse = null;
		HttpGet method = null;
		try {
			method = new HttpGet(url);
			if(headers == null || !headers.containsKey("User-Agent")){
				method.setHeader("User-Agent", "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/39.0.2171.95 Safari/537.36 MicroMessenger/6.5.2.501 NetType/WIFI WindowsWechat QBCore/3.43.691.400 QQBrowser/9.0.2524.400");
			}
			if(headers != null){
				addHeaders(method, headers);
			}
			httpResponse = httpClient.execute(method);
			String content = EntityUtils.toString(httpResponse.getEntity(), "UTF-8");
			return content;
		} catch (ParseException | IOException ex) {
			if(method != null) method.releaseConnection();
			if(ex instanceof SocketTimeoutException){
				LOGGER.error("请求超时");
			}
			else{
				LOGGER.error(ex.getMessage(), ex);				
			}
		}
		return "";
	}

	/**
	 * POST 请求
	 * @param url
	 * @param params
	 * @param headers
	 * @return
	 */
	public static String post(String url, Map<String, ?> params, Map<String, String> headers){
		try {
			HttpPost httpPost = new HttpPost(url);
			UrlEncodedFormEntity postEntity = new UrlEncodedFormEntity(getParam(params), "UTF-8");
			httpPost.setEntity(postEntity);
			httpPost.setHeader("User-Agent", "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/70.0.3538.77 Safari/537.36");
			if(headers != null){
				addHeaders(httpPost, headers);
			}
			HttpResponse httpResponse = httpClient.execute(httpPost);
			if(httpResponse.getStatusLine().getStatusCode() == 302) {
				String locationUrl = httpResponse.getLastHeader("Location").getValue();
				httpResponse.getEntity().getContent().close();
				return get(locationUrl);
			}
			return EntityUtils.toString(httpResponse.getEntity(), "UTF-8");
		} catch (ParseException | IOException ex) {
			if(ex instanceof SocketTimeoutException){
				LOGGER.error("请求超时");
			}
			else{
				LOGGER.error(ex.getMessage(), ex);				
			}
		}
		return "";
		
	}
	
	/**
	 * POST 支付宝请求
	 * @param url
	 * @param params
	 * @param headers
	 * @return
	 */
	public static String postForAlipay(String url, Map<String, ?> params, Map<String, String> headers){
		try {
			HttpPost httpPost = new HttpPost(url);
			UrlEncodedFormEntity postEntity = new UrlEncodedFormEntity(getParam(params), "UTF-8");
			httpPost.setEntity(postEntity);
			httpPost.setHeader("User-Agent", "mozilla/5.0 (linux; u; android 6.0.1; zh-cn; mi 5 build/mxb48t) applewebkit/537.36 (khtml, like gecko) version/4.0 chrome/40.0.2214.89 ucbrowser/11.6.4.950 uws/2.11.0.39 mobile safari/537.36 ucbs/2.11.0.39 nebula nebulasdk/1.7.0 alipaydefined(nt:wifi,ws:360|0|3.0) aliapp(ap/10.1.10.1226101) alipayclient/10.1.10.1226101 language/zh-hans usestatusbar/true");
			if(headers != null){
				addHeaders(httpPost, headers);
			}
			HttpResponse httpResponse = httpClient.execute(httpPost);
			if(httpResponse.getStatusLine().getStatusCode() == 302) {
				String locationUrl = httpResponse.getLastHeader("Location").getValue();
				return locationUrl;
			}
			return EntityUtils.toString(httpResponse.getEntity(), "UTF-8");
		} catch (ParseException | IOException ex) {
			if(ex instanceof SocketTimeoutException){
				LOGGER.error("请求超时");
			}
			else{
				LOGGER.error(ex.getMessage(), ex);				
			}
		}
		return "";
		
	}
	
	/**
	 * POST 支付宝请求
	 * @param url
	 * @return
	 */
	public static String postForAlipay(String url) {
		return postForAlipay(url, null, null);
	}
	
	/**
	 * POST 请求
	 * @param url
	 * @param params
	 * @return
	 */
	public static String postForm(String url, Map<String, ?> params){
		try {
			HttpPost httpPost = new HttpPost(url);
			UrlEncodedFormEntity postEntity = new UrlEncodedFormEntity(getParam(params), "UTF-8");
			httpPost.setEntity(postEntity);
			httpPost.setHeader("User-Agent", "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/60.0.3112.90 Safari/537.36");
			HttpResponse httpResponse = httpClient.execute(httpPost);
			return EntityUtils.toString(httpResponse.getEntity(), "UTF-8");
		} catch (ParseException | IOException ex) {
			if(ex instanceof SocketTimeoutException){
				throw new SimpleRuntimeException("请求超时");
			}
			else{
				LOGGER.error(ex.getMessage(), ex);				
			}
		}
		return "";
		
	}
	
	/**
	 * POST 请求
	 * @param url
	 * @param params
	 * @param headers
	 * @return
	 */
	public static String post(String url, JSONObject body){
		try {
			HttpPost httpPost = new HttpPost(url);
			StringEntity entity = new StringEntity(body.toString(),"utf-8");
			entity.setContentEncoding("UTF-8");    
			entity.setContentType("application/json");    
			httpPost.setEntity(entity);
			httpPost.setHeader("User-Agent", "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/60.0.3112.90 Safari/537.36");

			HttpResponse httpResponse = httpClient.execute(httpPost);
			return EntityUtils.toString(httpResponse.getEntity(), "UTF-8");
		} catch (ParseException | IOException ex) {
			if(ex instanceof SocketTimeoutException
					|| ex instanceof ConnectTimeoutException){
				LOGGER.error("请求超时");
			}
			else{
				LOGGER.error(ex.getMessage(), ex);				
			}
		}
		return "";
	}
	
	/**
	 * POST 请求
	 * @param url
	 * @param body
	 * @return
	 */
	public static HttpResponse postForResponse(String url, JSONObject body){
		try {
			HttpPost httpPost = new HttpPost(url);
			StringEntity entity = new StringEntity(body.toString(),"utf-8");
			entity.setContentEncoding("UTF-8");    
			entity.setContentType("application/json");    
			httpPost.setEntity(entity);
			httpPost.setHeader("User-Agent", "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/60.0.3112.90 Safari/537.36");

			return httpClient.execute(httpPost);
		} catch (ParseException | IOException ex) {
			if(ex instanceof SocketTimeoutException){
				LOGGER.error("请求超时");
			}
			else{
				LOGGER.error(ex.getMessage(), ex);				
			}
		}
		return null;
	}
	
	/**
	 * POST 请求
	 * @param url
	 * @return
	 * @throws ParseException
	 * @throws IOException
	 */
	public static String post(String url){
		Map<String, Object> params = new HashMap<String, Object>();
		return post(url, params, null);
	}
	
	/**
	 * POST 请求
	 * @param url
	 * @param params
	 * @param headers
	 * @return
	 */
	public static String postMultipartForm(String url, Map<String, ?> params, Map<String, String> headers){
		try {
			HttpPost httpPost = new HttpPost(url);
			
			MultipartEntityBuilder builder = MultipartEntityBuilder.create();
			
			for(Map.Entry<String, ?> entry : params.entrySet()) {
				Object value = entry.getValue();
				String key = entry.getKey();
				if(value instanceof File) {
					builder.addBinaryBody(key, (File) value);
				}
				else if(value instanceof FileBody) {
					builder.addPart(key, (FileBody) value);
				}
				else {
					builder.addTextBody(key, String.valueOf(value), ContentType.create("text/plain", Consts.UTF_8));
				}
			}

			HttpEntity postEntity = builder.build();
			httpPost.setEntity(postEntity);
			httpPost.setHeader("User-Agent", "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/60.0.3112.90 Safari/537.36");

			if(headers != null) {
				for(Map.Entry<String, String> entry : headers.entrySet()) {
					httpPost.setHeader(entry.getKey(), entry.getValue());
				}
			}

			HttpResponse httpResponse = httpClient.execute(httpPost);
			return EntityUtils.toString(httpResponse.getEntity(), "UTF-8");
		} catch (ParseException | IOException ex) {
			if(ex instanceof SocketTimeoutException
					|| ex instanceof ConnectTimeoutException){
				LOGGER.error("请求超时");
			}
			else{
				LOGGER.error(ex.getMessage(), ex);				
			}
		}
		return "";
		
	}
	
	/**
	 * 获取数据并转成对象
	 * @param url
	 * @param clazz
	 * @return
	 */
	public static <T> T getForObject(String url, Class<T> clazz){
		try{
			return toBean(get(url), clazz);
		}
		catch(SimpleRuntimeException ex){
			LOGGER.error(ex.getMessage(), ex);
		}
		return null;
	}
	
	/**
	 * 获取数据并转成对象
	 * @param url
	 * @param clazz
	 * @param headers
	 * @return
	 */
	public static <T> T getForObject(String url, Class<T> clazz, Map<String, String> headers){
		try{
			return toBean(get(url, headers), clazz);
		}
		catch(SimpleRuntimeException ex){
			LOGGER.error(ex.getMessage(), ex);
		}
		return null;
	}
	
	/**
	 * 获取数据并转成对象
	 * @param url
	 * @param clazz
	 * @return
	 */
	public static <T> T postForObject(String url, Class<T> clazz){
		try{
			return toBean(post(url), clazz);
		}
		catch(SimpleRuntimeException ex){
			LOGGER.error(ex.getMessage(), ex);
		}
		return null;
	}
	
	/**
	 * 获取数据并转成对象
	 * @param url
	 * @param clazz
	 * @return
	 */
	public static <T> T postForObject(String url, JSONObject body,  Class<T> clazz){
		try{
			return toBean(post(url, body), clazz);
		}
		catch(SimpleRuntimeException ex){
			LOGGER.error(ex.getMessage(), ex);
		}
		return null;
	}

	/**
	 * 请求参数
	 * @param parameterMap
	 * @return
	 */
	private static List<NameValuePair> getParam(Map<String, ?> parameterMap) {
		if(parameterMap == null) {
			parameterMap = new HashMap<String, String>();
		}
		List<NameValuePair> param = new ArrayList<NameValuePair>();
		for(Map.Entry<String, ?> entry : parameterMap.entrySet()){
			param.add(new BasicNameValuePair(entry.getKey(), String.valueOf(entry.getValue())));
		}
		return param;
	}
	
	/**
	 * 添加请求头部
	 * @param httpRequest
	 * @param headerMap
	 */
	private static void addHeaders(HttpRequestBase httpRequest, Map<String, String> headerMap){
		for(Map.Entry<String, String> entry : headerMap.entrySet()){
			httpRequest.addHeader(entry.getKey(), entry.getValue());
		}
	}
	
	/**
	 * 文本转换成BEAN
	 * @param response
	 * @param clazz
	 * @return
	 */
	private static <T> T toBean(String response, Class<T> clazz){
		LOGGER.info("响应文本内容：{}", response);
		MappingJackson2HttpMessageConverter messageConverter = new MappingJackson2HttpMessageConverter();
        messageConverter.setObjectMapper(JsonMapper.getMapper());
        if(List.class.isAssignableFrom(clazz)){
        	return messageConverter.getObjectMapper().convertValue(JSONArray.fromObject(StringUtils.UrlDecode(response)), clazz);
        }
		return messageConverter.getObjectMapper().convertValue(JSONObject.fromObject(StringUtils.UrlDecode(response)), clazz);
	}
	
	
	/**
	 * 获取httpClient
	 * @return
	 */
	public CloseableHttpClient getHttpClient() {
		return httpClient;
	}
}
