package com.crawler.utils;

import java.io.IOException;
import java.net.URLEncoder;

import javax.servlet.http.HttpServletRequest;

import org.apache.http.client.ClientProtocolException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.crawler.beans.Coords;
import com.crawler.beans.Location;
import com.crawler.config.ConfigProperties;
import com.crawler.http.SimpleHttpClient;

import net.sf.json.JSONObject;

/**
 * 百度地图工具
 * @author rubekid
 * @date 2016年9月10日
 */
public class BaiduMapUtils {
	
	private static final Logger LOGGER = LoggerFactory.getLogger(BaiduMapUtils.class);
	
	private static final String KEY = ConfigProperties.getProperty("map.baidu.key", "XoZYVVPR337Ot2AOBfUqLEum");

	private static final String GEO_CODER_URL = "http://api.map.baidu.com/geocoding/v3/?ak="+ KEY +"&output=json&address=%s";
	
	/**
	 * 通过ip地址获取地理位置
	 */
	private static final String IP_URL = "http://api.map.baidu.com/location/ip?ak="+ KEY +"&coor=bd09ll&ip=%s";

	/**
	 * 通过定位获取地理位置
	 */
	private static final String COORDS_URL = "http://api.map.baidu.com/reverse_geocoding/v3/?ak="+ KEY +"&output=json&coordtype=wgs84ll&location=";
	
	
	public static Coords getCoords(String address) {
		return getCoords(null, address);
	}
	
	/**
	 * 根据地址获取坐标
	 * @param city 城市
	 * @param address 详细地址
	 * @return
	 */
	public static Coords getCoords(String city, String address){
		Coords coords = null;
		if(address == null || "".equals(address)){
			address = city;
		}
		if(StringUtils.isNullOrEmpty(city) && StringUtils.isNullOrEmpty(address)) {
			return coords;
		}
		
		if(city != null && !"".equals(city) && city.indexOf("市") == -1){
			city = city + "市";
		}
		
		try{
			String url = String.format(GEO_CODER_URL, URLEncoder.encode(address, "utf-8"));
			if(!StringUtils.isNullOrEmpty(city)) {
				url += "&city=" + city;
			}
			
			JSONObject response = JSONObject.fromObject(SimpleHttpClient.get(url));
			LOGGER.info("\n请求地址：{}\n响应内容：{}", url, response);
			if(response.containsKey("status") && response.getInt("status") ==0){
				JSONObject result = response.getJSONObject("result");
				JSONObject location = result.getJSONObject("location");
				coords = new Coords();
				coords.setLongitude(location.getDouble("lng"));
				coords.setLatitude(location.getDouble("lat"));
			}
		}
		catch(Exception ex){
			LOGGER.error(ex.getMessage(), ex);
		}
		return coords;
	}
	
	/**
	 * 通过ip获取定位信息
	 * @param request
	 * @return
	 * @throws IOException 
	 * @throws ClientProtocolException 
	 */
	public static Location getLocation(HttpServletRequest request) {
		return getLocation(IpAddressUtils.getIpAddress(request));
	}
	
	/**
	 * 通过ip获取定位信息
	 * @param request
	 * @return
	 * @throws IOException 
	 * @throws ClientProtocolException 
	 */
	public static Location getLocation(String ip) {
		try{
			String url = String.format(IP_URL, ip);
			JSONObject response = JSONObject.fromObject(SimpleHttpClient.get(url));
			LOGGER.info("\n请求地址：{}\n响应内容：{}", url, response);
			if(response.containsKey("status") && response.getInt("status") ==0){
				JSONObject content = response.getJSONObject("content");
				Location location = new Location();
				Coords coords = new Coords(content.getJSONObject("point").getDouble("x"), content.getJSONObject("point").getDouble("y"));
				JSONObject addressDetail = content.getJSONObject("address_detail");
				location.setProvince(addressDetail.getString("province"));
				location.setCity(addressDetail.getString("city"));
				location.setDistrict(addressDetail.getString("district"));
				location.setCoords(coords);
				return location;
			}
		}
		catch(Exception ex){
			LOGGER.error(ex.getMessage(), ex);
		}
		return null;
	}
	
	/**
	 * 通过坐标获取定位信息
	 * @param request
	 * @return
	 * @throws IOException 
	 * @throws ClientProtocolException 
	 */
	public static Location getLocation(Coords coords){
		try{
			String url = COORDS_URL + coords.getLatitude() + "," + coords.getLongitude();
			JSONObject response = JSONObject.fromObject(SimpleHttpClient.get(url));
			LOGGER.info("\n请求地址：{}\n响应内容：{}", url, response);
			if(response.containsKey("status") && response.getInt("status") ==0){
				JSONObject result = response.getJSONObject("result");
				Location location = new Location();
				location.setCoords(coords);
				JSONObject addressDetail = result.getJSONObject("addressComponent");
				location.setProvince(addressDetail.getString("province"));
				location.setCity(addressDetail.getString("city"));
				location.setDistrict(addressDetail.getString("district"));
				location.setStreet(addressDetail.getString("street"));
				if(result.containsKey("formatted_address")) {
					location.setAddress(result.getString("formatted_address"));					
				}
				return location;
			}
		}
		catch(Exception ex){
			LOGGER.error(ex.getMessage(), ex);
		}
		return null;
	}
}
