package com.crawler.utils;

import java.io.IOException;
import java.net.URLEncoder;

import javax.servlet.http.HttpServletRequest;

import org.apache.http.client.ClientProtocolException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.crawler.beans.Coords;
import com.crawler.beans.Location;
import com.crawler.config.ConfigProperties;
import com.crawler.http.SimpleHttpClient;

import net.sf.json.JSONObject;

/**
 * 腾讯地图工具
 * @author rubekid
 * @date 2020年03月22日
 */
public class TencentMapUtils {
	
	private static final Logger LOGGER = LoggerFactory.getLogger(TencentMapUtils.class);
	
	private static final String KEY = ConfigProperties.getProperty("map.tencent.key", "JKGBZ-GJXCO-FMPWM-SD7GJ-JEGGV-TABXR");
	
	/**
	 * 定位
	 */
	private static final String GEO_CODER_URL = "https://apis.map.qq.com/ws/geocoder/v1/?key="+ KEY +"&address=";
	
	/**
	 * 通过ip地址获取地理位置
	 */
	private static final String IP_URL = "https://apis.map.qq.com/ws/location/v1/ip?key="+ KEY +"&ip=";

	/**
	 * 通过定位获取地理位置
	 */
	private static final String COORDS_URL = "https://apis.map.qq.com/ws/geocoder/v1/?key="+ KEY +"&location=";
	
	/**
	 * 根据地址获取坐标
	 * @param address
	 * @return
	 */
	public static Coords getCoords(String address) {
		return getCoords(null, address);
	}
	
	/**
	 * 根据地址获取坐标
	 * @param city 城市
	 * @param address 详细地址
	 * @return
	 */
	public static Coords getCoords(String city, String address){
		Coords coords = null;
		try{
			String url = GEO_CODER_URL + URLEncoder.encode(address, "utf-8");
			if(city != null && !"".equals(city)) {
				url += "&region=" + city;
			}
				
			JSONObject response = JSONObject.fromObject(SimpleHttpClient.get(url));
			LOGGER.info("\n请求地址：{}\n响应内容：{}", url, response);
			if(response.containsKey("status") && response.getInt("status") ==0){
				JSONObject result = response.getJSONObject("result");
				JSONObject location = result.getJSONObject("location");
				coords = new Coords();
				coords.setLongitude(location.getDouble("lng"));
				coords.setLatitude(location.getDouble("lat"));
			}
		}
		catch(Exception ex){
			LOGGER.error(ex.getMessage(), ex);
		}
		return coords;
	}
	
	/**
	 * 通过ip获取定位信息
	 * @param request
	 * @return
	 * @throws IOException 
	 * @throws ClientProtocolException 
	 */
	public static Location getLocation(HttpServletRequest request) {
		return getLocation(IpAddressUtils.getIpAddress(request));
	}
	
	/**
	 * 通过ip获取定位信息
	 * @param request
	 * @return
	 * @throws IOException 
	 * @throws ClientProtocolException 
	 */
	public static Location getLocation(String ip) {
		try{
			String url = IP_URL + ip;
			JSONObject response = JSONObject.fromObject(SimpleHttpClient.get(url));
			LOGGER.info("\n请求地址：{}\n响应内容：{}", url, response);
			if(response.containsKey("status") && response.getInt("status") ==0){
				JSONObject content = response.getJSONObject("result");
				Location location = new Location();
				Coords coords = new Coords(content.getJSONObject("location").getDouble("lng"), content.getJSONObject("location").getDouble("lat"));
				JSONObject addressDetail = content.getJSONObject("ad_info");
				location.setProvince(addressDetail.getString("province"));
				location.setCity(addressDetail.getString("city"));
				location.setDistrict(addressDetail.getString("district"));
				location.setCoords(coords);
				return location;
			}
		}
		catch(Exception ex){
			LOGGER.error(ex.getMessage(), ex);
		}
		return null;
	}
	
	/**
	 * 通过坐标获取定位信息
	 * @param request
	 * @return
	 * @throws IOException 
	 * @throws ClientProtocolException 
	 */
	public static Location getLocation(Coords coords){
		try{
			String url = COORDS_URL + coords.getLatitude() + "," + coords.getLongitude();
			JSONObject response = JSONObject.fromObject(SimpleHttpClient.get(url));
			LOGGER.info("\n请求地址：{}\n响应内容：{}", url, response);
			if(response.containsKey("status") && response.getInt("status") ==0){
				JSONObject result = response.getJSONObject("result");
				Location location = new Location();
				location.setCoords(coords);
				JSONObject addressDetail = result.getJSONObject("address_component");
				location.setProvince(addressDetail.getString("province"));
				location.setCity(addressDetail.getString("city"));
				location.setDistrict(addressDetail.getString("district"));
				location.setStreet(addressDetail.getString("street"));
				location.setAddress(result.getString("address"));
				return location;
			}
		}
		catch(Exception ex){
			LOGGER.error(ex.getMessage(), ex);
		}
		return null;
	}
}
