package com.crawler.pay.wxpay;

import com.alibaba.fastjson.JSONObject;
import com.crawler.pay.alipay.exception.PayException;
import com.crawler.pay.common.MD5;
import com.crawler.pay.common.StringUtils;
import com.crawler.pay.http.ClientCustomSSL;
import com.crawler.pay.http.HttpUtils;
import com.crawler.pay.wxpay.model.Hbinfo;
import com.crawler.pay.wxpay.model.RedPackageQueryResult;
import com.crawler.pay.wxpay.model.RedPackageResult;
import com.crawler.pay.utils.XmlUtils;
import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.JDOMException;
import org.jdom2.input.SAXBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.net.URLEncoder;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.util.*;
import java.util.Map.Entry;


/**
 * 微信
 * 
 * @author rubekid
 * @date 2016年9月11日
 */
public class WxpayUtils {
	
	private static final Logger logger = LoggerFactory.getLogger(WxpayUtils.class);
	
	/**
	 * 统一下单接口
	 */
	public static final String WX_UNIFIED_ORDER = "https://api.mch.weixin.qq.com/pay/unifiedorder";
	
	/**
	 * 订单查询接口
	 */
	public static final String WX_ORDER_QUERY = "https://api.mch.weixin.qq.com/pay/orderquery";
	
	/**
	 * 退款接口
	 */
	public static final String WX_REFUND = "https://api.mch.weixin.qq.com/secapi/pay/refund";
	
	/**
	 * 企业付款到余额接口
	 */
	public static final String WX_TRANSFERS = "https://api.mch.weixin.qq.com/mmpaymkttransfers/promotion/transfers";
	
	/**
	 * 发送普通红包
	 */
	public static final String WX_SEND_RED_PACKAGE = "https://api.mch.weixin.qq.com/mmpaymkttransfers/sendredpack";

	/**
	 * 查询红包记录
	 */
	public static final String WX_QUERY_RED_PACKAGE = "https://api.mch.weixin.qq.com/mmpaymkttransfers/gethbinfo";

	/**
	 * 发送小程序红包
	 */
	public static final String WX_WEAPP_RED_PACKAGE = "https://api.mch.weixin.qq.com/mmpaymkttransfers/sendminiprogramhb";

	
	/**
	 *  创建sign规则
	 * @param parameters
	 * @param apiKey
	 * @return
	 */
	public static String createSign(SortedMap<String, Object> parameters, String apiKey){
		StringBuffer sb = new StringBuffer();
		for (Map.Entry<String, Object> entry : parameters.entrySet()) {
			String key = entry.getKey();
			String value = String.valueOf(entry.getValue());
			if (!"sign".equalsIgnoreCase(key) && !"key".equalsIgnoreCase(key) && !StringUtils.isNullOrEmpty(value)) {
				sb.append(key + "=" + value + "&");
			}
		}
		sb.append("key=" + apiKey);
		return MD5.encode(sb.toString()).toUpperCase();
	}


	/**
	 * 构建请求对象为xml字符串
	 * 
	 * @param parameters
	 * @return
	 */
	public static String toXml(SortedMap<String, Object> parameters) {
		StringBuffer sb = new StringBuffer();
		sb.append("<xml>");
		for (Map.Entry<String, Object> entry : parameters.entrySet()) {
			String key = entry.getKey();
			String value = String.valueOf(entry.getValue());
			if ("attach".equalsIgnoreCase(key) || "body".equalsIgnoreCase(key) || "sign".equalsIgnoreCase(key)) {
				sb.append("<" + key + ">" + "<![CDATA[" + value + "]]></" + key + ">");
			} else {
				sb.append("<" + key + ">" + value + "</" + key + ">");
			}
		}
		sb.append("</xml>");
		return sb.toString();
	}

	/**
	 * 解析xml
	 * 
	 * @param xml
	 * @return
	 * @throws IOException
	 */
	public static SortedMap<String, String> doXMLParse(String xml){
		if(xml == null){
			return null;
		}
		xml = xml.replaceFirst("encoding=\".*\"", "encoding=\"UTF-8\"");
		if (null == xml || "".equals(xml)) {
			return null;
		}
		
		SortedMap<String, String> map = new TreeMap<String, String>();
		
		try{
			InputStream in = new ByteArrayInputStream(xml.getBytes("UTF-8"));
			SAXBuilder builder = new SAXBuilder();
			Document doc = builder.build(in);
			Element root = doc.getRootElement();
			List<Element> list = root.getChildren();
			Iterator<Element> it = list.iterator();
			while (it.hasNext()) {
				Element e = (Element) it.next();
				String key = e.getName();
				String value = "";
				List<Element> children = e.getChildren();
				if (children.isEmpty()) {
					value = e.getTextNormalize();
				} else {
					value = getChildrenText(children);
				}
				map.put(key, value);
			}
			in.close();
		}
		catch( JDOMException | IOException ex){
			logger.error("XML解析失败" + ex.getMessage());
			throw new PayException("XML解析失败");
		}
		
		return map;
	}

	/**
	 * 获取子结点的xml
	 * 
	 * @param children
	 * @return String
	 */
	public static String getChildrenText(List<Element> children) {
		StringBuffer sb = new StringBuffer();
		if (!children.isEmpty()) {
			Iterator<Element> it = children.iterator();
			while (it.hasNext()) {
				Element e = (Element) it.next();
				String name = e.getName();
				String value = e.getTextNormalize();
				List<Element> list = e.getChildren();
				sb.append("<" + name + ">");
				if (!list.isEmpty()) {
					sb.append(getChildrenText(list));
				}
				sb.append(value);
				sb.append("</" + name + ">");
			}
		}
		return sb.toString();
	}

	/**
	 * 设置xml
	 * 
	 * @param return_code
	 * @param return_msg
	 * @return
	 */
	public static String setXML(String return_code, String return_msg) {
		return "<xml><return_code><![CDATA[" + return_code + "]]></return_code><return_msg><![CDATA[" + return_msg
				+ "]]></return_msg></xml>";
	}
	
	
	/**
	 * 签名校验
	 * @param map
	 * @return
	 */
	public static boolean checkSign(SortedMap<String, String> map, String apiKey){
		String wxsign = String.valueOf(map.get("sign"));
		StringBuffer sb = new StringBuffer();
		for(Entry<String, String> entry : map.entrySet()){
			String k = entry.getKey();
			String v = entry.getValue();
			if(!"sign".equals(k) && null != v && !"".equals(v)) {
				sb.append(k + "=" + v + "&");
			}
		}
		//算出摘要
		String sign = MD5.encode(sb.toString() + "key=" + apiKey).toLowerCase();
		
		return wxsign.equalsIgnoreCase(sign);
	}
	
	/**
	 * 随机
	 * @return
	 */
	public static String  nonceStr(){
		return MD5.encode(getTimestamp());
	}
	
	/**
	 * 成功返回
	 * @return
	 */
	public static String successResponse(){
		SortedMap<String, Object> response = new TreeMap<String, Object>();
		response.put("return_code", "SUCCESS");
		response.put("return_msg", "OK");
    	return toXml(response);
	}
	
	/**
	 * 微信统一下单
	 * @param orderNo 订单号
	 * @param detail 订单详情
	 * @param totalFee 总金额
	 * @param openId openID
	 * @param wxpayConfig 配置
	 * @return
	 */
	public static SortedMap<String,Object>  wechatUnifiedOrder(String orderNo, String detail, BigDecimal totalFee, String openId, WxpayConfig wxpayConfig){

		//****************************** 统一下单 start****************************//*
		SortedMap<String,Object> parameters = new TreeMap<String, Object>();
		parameters.put("appid", wxpayConfig.getAppId());
		parameters.put("mch_id", wxpayConfig.getMchId());
		if(openId!=null){
			 parameters.put("openid", openId);
		}
		parameters.put("nonce_str", WxpayUtils.nonceStr());
		parameters.put("body",detail);
		parameters.put("out_trade_no",orderNo);
		parameters.put("total_fee", totalFee.multiply(BigDecimal.valueOf(100)).intValue());
		parameters.put("spbill_create_ip", wxpayConfig.getIp());
		parameters.put("notify_url", wxpayConfig.getNotifyUrl());
		parameters.put("trade_type", "JSAPI");//调用APP支付
		String sign = WxpayUtils.createSign(parameters, wxpayConfig.getApiKey());
		parameters.put("sign", sign);
		String requestXML = WxpayUtils.toXml(parameters);
		System.out.println("==================统一下单=========================");
		System.out.println(requestXML);
		System.out.println("==================统一下单=========================");
		//*******************************统一下单 end
		
		
		//调用统一下单接口接收返回数据
		String response = HttpUtils.httpsRequest(WX_UNIFIED_ORDER, HttpUtils.METHOD_POST, requestXML);
		//微信返回的信息
		Map<String, String> map = WxpayUtils.doXMLParse(response);
		SortedMap<String,Object> params = null;
		if("SUCCESS".equals(map.get("result_code"))){
			params=new TreeMap<String, Object>();
			params.put("appId",wxpayConfig.getAppId());
			params.put("timeStamp",getTimestamp());
			params.put("nonceStr", WxpayUtils.createSign(params, wxpayConfig.getApiKey()));
			params.put("package", "prepay_id=" + map.get("prepay_id"));
			params.put("signType", "MD5");
			params.put("paySign",WxpayUtils.createSign(params, wxpayConfig.getApiKey()));
		}
		else{
			logger.error("统一下单失败" + response);
			throw new PayException("微信统一下单失败");
		}
		
		return params;
	}
	
	/**
	 * 微信统一下单
	 * @param orderNo 订单号
	 * @param detail 订单详情
	 * @param totalFee 总价格
	 * @param wxpayConfig 配置
	 * @return
	 */
	public static SortedMap<String,Object>  h5UnifiedOrder(String orderNo, String detail, BigDecimal totalFee, WxpayConfig wxpayConfig){

		//****************************** 统一下单 start****************************//*
		SortedMap<String,Object> parameters = new TreeMap<String, Object>();
		parameters.put("appid", wxpayConfig.getAppId());
		parameters.put("mch_id", wxpayConfig.getMchId());
		parameters.put("nonce_str", WxpayUtils.nonceStr());
		parameters.put("body",detail);
		parameters.put("out_trade_no",orderNo);
		parameters.put("total_fee", totalFee.multiply(BigDecimal.valueOf(100)).intValue());
		parameters.put("spbill_create_ip", wxpayConfig.getIp());
		parameters.put("notify_url", wxpayConfig.getNotifyUrl());
		parameters.put("trade_type", "MWEB");//调用APP支付
		String sign = WxpayUtils.createSign(parameters, wxpayConfig.getApiKey());
		parameters.put("sign", sign);
		String requestXML = WxpayUtils.toXml(parameters);
		System.out.println("==================统一下单=========================");
		System.out.println(requestXML);
		System.out.println("==================统一下单=========================");
		//*******************************统一下单 end
		
		
		//调用统一下单接口接收返回数据
		String response = HttpUtils.httpsRequest(WX_UNIFIED_ORDER, HttpUtils.METHOD_POST, requestXML);
		//微信返回的信息
		Map<String, String> map = WxpayUtils.doXMLParse(response);
		SortedMap<String,Object> params = null;
		if("SUCCESS".equals(map.get("result_code"))){
			params=new TreeMap<String, Object>();
			params.put("appId",wxpayConfig.getAppId());
			params.put("timeStamp",getTimestamp());
			params.put("nonceStr", WxpayUtils.createSign(params, wxpayConfig.getApiKey()));
			params.put("package", "prepay_id=" + map.get("prepay_id"));
			params.put("signType", "MD5");
			params.put("paySign",WxpayUtils.createSign(params, wxpayConfig.getApiKey()));
			params.put("mwebUrl", map.get("mweb_url"));
		}
		else{
			logger.error("统一下单失败" + response);
			throw new PayException("微信统一下单失败");
		}
		
		return params;
	}
	
	
	/**
	 * 微信统一下单
	 * @param orderNo 订单号
	 * @param detail 订单详情
	 * @param totalFee 总金额
	 * @param wxpayConfig 配置
	 * @return
	 */
	public static SortedMap<String,Object>  appUnifiedOrder(String orderNo, String detail, BigDecimal totalFee, WxpayConfig wxpayConfig){
		//****************************** 统一下单 start****************************//*
		SortedMap<String,Object> parameters = new TreeMap<String, Object>();
		parameters.put("appid", wxpayConfig.getAppId());
		parameters.put("mch_id", wxpayConfig.getMchId());
		parameters.put("nonce_str", WxpayUtils.nonceStr());
		parameters.put("body",detail);
		parameters.put("out_trade_no",orderNo);
		parameters.put("total_fee",totalFee.multiply(BigDecimal.valueOf(100)).intValue());
		parameters.put("spbill_create_ip", wxpayConfig.getIp());
		parameters.put("notify_url", wxpayConfig.getNotifyUrl());
		parameters.put("trade_type", "APP");//调用APP支付
		String sign = WxpayUtils.createSign(parameters, wxpayConfig.getApiKey());
		parameters.put("sign", sign);
		String requestXML = WxpayUtils.toXml(parameters);
		//*******************************统一下单 end
		
		
		//调用统一下单接口接收返回数据
		String response = HttpUtils.httpsRequest(WX_UNIFIED_ORDER, HttpUtils.METHOD_POST, requestXML);
		//微信返回的信息
		Map<String, String> map = WxpayUtils.doXMLParse(response);
		SortedMap<String,Object> params = null;
		if("SUCCESS".equals(map.get("result_code"))){
			params=new TreeMap<String, Object>();
			params.put("appid", wxpayConfig.getAppId());
			params.put("partnerid", wxpayConfig.getMchId());
			params.put("prepayid", map.get("prepay_id"));    //这里用
			params.put("noncestr", WxpayUtils.createSign(params, wxpayConfig.getApiKey()));
			params.put("timestamp", getTimestamp());
			params.put("package", "Sign=WXPay");
			params.put("sign", WxpayUtils.createSign(params, wxpayConfig.getApiKey()));

		}
		else{
			logger.error("AppKey:" + wxpayConfig.getApiKey());
			logger.error("requestXML:" + requestXML);
			logger.error("统一下单失败" + response);
			throw new PayException("微信统一下单失败");
		}
		
		return params;
	}
	
	/**
	 * 退款处理
	 * @param orderNo 
	 * @param tradeNo
	 * @param refundNo
	 * @param fee
	 * @param userId
	 * @return
	 */
	public static boolean refund(String orderNo, String tradeNo, String refundNo, BigDecimal fee, Long userId, WxpayConfig wxpayConfig){
		return refund(wxpayConfig.getAppId(), wxpayConfig.getMchId(), wxpayConfig.getCertPath(), wxpayConfig.getApiKey(), orderNo, tradeNo, refundNo, fee, userId);
	}
	
	/**
	 * 退款处理
	 * @param orderNo
	 * @param tradeNo
	 * @param refundNo
	 * @param fee
	 * @param totalFee
	 * @param userId
	 * @return
	 */
	public static boolean refund(String orderNo, String tradeNo, String refundNo, BigDecimal fee, BigDecimal totalFee, Long userId, WxpayConfig wxpayConfig){
		return refund(wxpayConfig.getAppId(), wxpayConfig.getMchId(), wxpayConfig.getCertPath(), wxpayConfig.getApiKey(), orderNo, tradeNo, refundNo, fee, totalFee, userId);
	}
	
	/**
	 * 退款
	 * @param appId
	 * @param mchId
	 * @param certPath
	 * @param apiKey
	 * @param orderNo
	 * @param tradeNo
	 * @param refundNo
	 * @param fee
	 * @param userId
	 * @return
	 */
	public static boolean refund(String appId, String mchId, String certPath, String apiKey,  String orderNo, String tradeNo, String refundNo, BigDecimal fee, Long userId){
		return refund(appId, mchId, certPath, apiKey, orderNo, tradeNo, refundNo, fee, null, userId);
	}
	
	/**
	 * 退款
	 * @param appId
	 * @param mchId
	 * @param certPath
	 * @param apiKey
	 * @param orderNo
	 * @param tradeNo
	 * @param refundNo
	 * @param fee
	 * @param totalFee
	 * @param userId
	 * @return
	 */
	public static boolean refund(String appId, String mchId, String certPath, String apiKey,  String orderNo, String tradeNo, String refundNo, BigDecimal fee, BigDecimal totalFee, Long userId){
		SortedMap<String,Object> parameters = new TreeMap<String, Object>();
		parameters.put("appid", appId);
		parameters.put("mch_id",mchId);
		parameters.put("nonce_str", WxpayUtils.nonceStr());
		if(orderNo != null){
			parameters.put("out_trade_no", orderNo);
		}
		long value = fee.multiply(BigDecimal.valueOf(100L)).longValue();
		long total = value;
		if(totalFee != null){
			total = totalFee.multiply(BigDecimal.valueOf(100L)).longValue();
		}
		parameters.put("out_refund_no", refundNo);
		parameters.put("total_fee", total);
		parameters.put("refund_fee", value);
		parameters.put("op_user_id", userId);
		parameters.put("transaction_id", tradeNo);
		
		String sign = WxpayUtils.createSign(parameters , apiKey);
		parameters.put("sign", sign);
		String requestXML = WxpayUtils.toXml(parameters);
		
		try {
			String response = ClientCustomSSL.doPost(mchId, certPath, WX_REFUND, requestXML);
			System.out.println("=======================微信退款======================");
			System.out.println(response);
			Map<String, String> map = WxpayUtils.doXMLParse(response);
			if("SUCCESS".equals(map.get("return_code")) && "SUCCESS".equals(map.get("result_code"))){
				return true;
			}
		} catch ( KeyManagementException | UnrecoverableKeyException 
				| NoSuchAlgorithmException | CertificateException | KeyStoreException | IOException e) {
			logger.error("微信退款失败" + e.getMessage());
			throw new PayException("微信退款失败");
			
		}
		return false;
	}
	
	/**
	 * 企业付款
	 * @param appId 商户账号appid
	 * @param mchId 商户号
	 * @param openId 用户openid
	 * @param tradeNo 交易单号
	 * @param userName 收款用户姓名
	 * @param amount 金额
	 * @param desc 企业付款描述信息
	 * @param ip Ip地址
	 */
	public static Map<String, String> transfers(String appId, String mchId, String apiKey, String openId, String tradeNo, String userName, BigDecimal amount, String desc, String ip){
		//****************************** 统一下单 start****************************//*
		String checkName = userName == null ? "NO_CHECK" : "FORCE_CHECK";
		SortedMap<String,Object> parameters = new TreeMap<String, Object>();
		parameters.put("mch_appid", appId);
		parameters.put("mchid", mchId);
		parameters.put("nonce_str", WxpayUtils.nonceStr());
		parameters.put("partner_trade_no", tradeNo);
		parameters.put("openid", openId);
		parameters.put("check_name", checkName);
		if(userName != null){
			parameters.put("re_user_name", userName);			
		}
		
		parameters.put("amount", amount.multiply(BigDecimal.valueOf(100)).intValue());
		parameters.put("desc", desc);
		parameters.put("spbill_create_ip", ip);
		String sign = WxpayUtils.createSign(parameters, apiKey);
		parameters.put("sign", sign);
		String requestXML = WxpayUtils.toXml(parameters);
		//*******************************统一下单 end
		
		
		//调用统一下单接口接收返回数据
		String response = HttpUtils.httpsRequest(WX_UNIFIED_ORDER, HttpUtils.METHOD_POST, requestXML);
		//微信返回的信息
		Map<String, String> map = WxpayUtils.doXMLParse(response);
		if("SUCCESS".equals(map.get("result_code"))){
			return map;
		}
		else{
			logger.error("AppKey:" + apiKey);
			logger.error("requestXML:" + requestXML);
			logger.error("企业付款失败" + response);
			throw new PayException("企业付款失败");
		}	
	}
	
	/**
	 * 企业付款
	 * @param config 配置
	 * @param openId 用户openid
	 * @param tradeNo 交易单号
	 * @param userName 收款用户姓名
	 * @param amount 金额
	 * @param desc 企业付款描述信息
	 */
	public static void transfers(WxpayConfig config, String openId, String tradeNo, String userName, BigDecimal amount, String desc){
		transfers(config.getAppId(), config.getMchId(), config.getApiKey(), openId, tradeNo, userName, amount, desc, config.getIp());
	}

	/**
	 * 发放红包
	 * @param config
	 * @param sendName
	 * @param openId
	 * @param orderNo
	 * @param amount
	 * @param wishing
	 * @param activityName
	 * @param remark
	 * @param sceneId  发放红包使用场景，红包金额大于200或者小于1元时必传
	 *
	 * PRODUCT_1:商品促销
	 *
	 * PRODUCT_2:抽奖
	 *
	 * PRODUCT_3:虚拟物品兑奖
	 *
	 * PRODUCT_4:企业内部福利
	 *
	 * PRODUCT_5:渠道分润
	 *
	 * PRODUCT_6:保险回馈
	 *
	 * PRODUCT_7:彩票派奖
	 *
	 * PRODUCT_8:税务刮奖
	 * @param riskInfo
	 * @return
	 */
	public static RedPackageResult sendRedpack(WxpayConfig config,  String sendName, String openId, String orderNo, BigDecimal amount, String wishing, String activityName, String remark, String sceneId, String riskInfo){
		long totalAmount = amount.multiply(BigDecimal.valueOf(100L)).longValue();

		SortedMap<String,Object> parameters = new TreeMap<String, Object>();
		parameters.put("nonce_str", nonceStr());
		parameters.put("mch_billno", orderNo);
		parameters.put("mch_id", config.getMchId());
		parameters.put("wxappid", config.getAppId());
		parameters.put("send_name", sendName);
		parameters.put("re_openid", openId);
		parameters.put("total_amount", totalAmount);
		parameters.put("total_num", 1);
		parameters.put("wishing", wishing);
		parameters.put("client_ip", config.getIp());
		parameters.put("act_name", activityName);
		parameters.put("remark", remark);
		if(sceneId != null){
			parameters.put("scene_id", sceneId);
		}
		if(riskInfo != null){
			parameters.put("risk_info", riskInfo);
		}

		String sign = WxpayUtils.createSign(parameters , config.getApiKey());
		parameters.put("sign", sign);
		String requestXML = WxpayUtils.toXml(parameters);
		
		
		logger.info("请求参数：" + requestXML);
		String message = "发放红包失败";

		try {
			String response = ClientCustomSSL.doPost(config.getMchId(), config.getCertPath(), WX_SEND_RED_PACKAGE, requestXML);
			logger.info("=======================微信红包======================");
			logger.info(response);
			Map<String, String> map = WxpayUtils.doXMLParse(response);
			if("SUCCESS".equals(map.get("return_code")) && "SUCCESS".equals(map.get("result_code"))){
				RedPackageResult result = new RedPackageResult();
				result.setMchBillno(map.get("mch_billno"));
				result.setMchId(map.get("mch_id"));
				result.setWxappid(map.get("wxappid"));
				result.setReOpenid(map.get("re_openid"));
				result.setTotalAmount(new BigDecimal(map.get("total_amount")).divide(new BigDecimal(100), 2, RoundingMode.HALF_DOWN));
				result.setSendListid(map.get("send_listid"));
				result.setMessage(map.get("return_msg"));

				return result;
			}
			message = map.get("return_msg");
		} catch ( KeyManagementException | UnrecoverableKeyException
				| NoSuchAlgorithmException | CertificateException | KeyStoreException | IOException e) {
			logger.error("微信红包失败" + e.getMessage());
			throw new PayException("微信红包失败");

		}
		throw new PayException(message);



	}

	/**
	 * 查询微信红包
	 * @param config
	 * @param orderNo
	 * @return
	 */
	public static RedPackageQueryResult queryRedpack(WxpayConfig config, String orderNo){
		SortedMap<String,Object> parameters = new TreeMap<String, Object>();
		parameters.put("nonce_str", nonceStr());
		parameters.put("mch_billno", orderNo);
		parameters.put("mch_id", config.getMchId());
		parameters.put("appid", config.getAppId());
		parameters.put("bill_type", "MCHT");

		String sign = WxpayUtils.createSign(parameters , config.getApiKey());
		parameters.put("sign", sign);
		String requestXML = WxpayUtils.toXml(parameters);


		logger.info("请求参数：" + requestXML);
		String message = "查询红包失败";

		try {
			String response = ClientCustomSSL.doPost(config.getMchId(), config.getCertPath(), WX_QUERY_RED_PACKAGE, requestXML);
			logger.info("=======================查询微信红包======================");
			logger.info(response);
			JSONObject respJson = XmlUtils.parse(response);
			if("SUCCESS".equals(respJson.getString("return_code")) && "SUCCESS".equals(respJson.getString("result_code"))){

				int totalNum = 0;
				BigDecimal totalAmount = null;
				if(respJson.containsKey("total_num")){
					totalNum = respJson.getInteger("total_num");
				}
				if(respJson.containsKey("total_amount")){
					totalAmount = respJson.getBigDecimal("total_amount").divide(new BigDecimal("100"), 2, RoundingMode.HALF_DOWN);
				}

				Date sendTime = respJson.getDate("send_time");
				Date refundTime = null;
				BigDecimal refundAmount = null;
				if(respJson.containsKey("refund_time")){
					refundTime = respJson.getDate("refund_time");
				}

				if(respJson.containsKey("refund_amount")){
					refundAmount = respJson.getBigDecimal("refund_amount").divide(new BigDecimal("100"), 2, RoundingMode.HALF_DOWN);
				}

				RedPackageQueryResult result = new RedPackageQueryResult();
				result.setMchBillno(respJson.getString("mch_billno"));
				result.setMchId(respJson.getString("mch_id"));
				result.setDetailId(respJson.getString("detail_id"));
				result.setStatus(respJson.getString("status"));
				result.setSendType(respJson.getString("send_type"));
				result.setHbType(respJson.getString("hb_type"));
				result.setTotalNum(totalNum);
				result.setTotalAmount(totalAmount);
				result.setReason(respJson.getString("reason"));
				result.setSendTime(sendTime);
				result.setRefundTime(refundTime);
				result.setRefundAmount(refundAmount);
				result.setWishing(respJson.getString("wishing"));
				result.setRemark(respJson.getString("remark"));
				result.setActName(respJson.getString("act_name"));

				if(respJson.containsKey("hblist")){
					List<Hbinfo> hblist = new ArrayList<Hbinfo>();
					for(int i=0; i< respJson.getJSONArray("hblist").size(); i++){
						JSONObject json = respJson.getJSONArray("hblist").getJSONObject(i);
						Hbinfo hbinfo = new Hbinfo();
						hbinfo.setOpenid(json.getString("openid"));
						hbinfo.setAmount(json.getBigDecimal("amount").divide(new BigDecimal("100"), 2, RoundingMode.HALF_DOWN));
						hbinfo.setRcvTime(json.getDate("rcv_time"));
						hblist.add(hbinfo);
					}
					result.setHblist(hblist);
				}

				return result;
			}
			message = respJson.getString("return_msg");
		} catch ( KeyManagementException | UnrecoverableKeyException
				| NoSuchAlgorithmException | CertificateException | KeyStoreException | IOException e) {
			logger.error("查询微信红包失败" + e.getMessage());
			throw new PayException("查询微信红包失败");

		}
		throw new PayException(message);



	}

	/**
	 * 发送小程序红包
	 * @param config
	 * @param sendName
	 * @param openId
	 * @param orderNo
	 * @param amount
	 * @param wishing
	 * @param activityName
	 * @param remark
	 * @param sceneId
	 * @return
	 */
	public static RedPackageResult sendMiniprogramhb(WxpayConfig config, String sendName, String openId, String orderNo, BigDecimal amount, String wishing, String activityName, String remark, String sceneId){
		long totalAmount = amount.multiply(BigDecimal.valueOf(100L)).longValue();

		SortedMap<String,Object> parameters = new TreeMap<String, Object>();
		parameters.put("nonce_str", nonceStr());
		parameters.put("mch_billno", orderNo);
		parameters.put("mch_id", config.getMchId());
		parameters.put("wxappid", config.getAppId());
		parameters.put("send_name", sendName);
		parameters.put("re_openid", openId);
		parameters.put("total_amount", totalAmount);
		parameters.put("total_num", 1);
		parameters.put("wishing", wishing);
		parameters.put("client_ip", config.getIp());
		parameters.put("act_name", activityName);
		parameters.put("remark", remark);
		parameters.put("notify_way", "MINI_PROGRAM_JSAPI");
		if(sceneId != null){
			parameters.put("scene_id", sceneId);
		}


		String sign = WxpayUtils.createSign(parameters , config.getApiKey());
		parameters.put("sign", sign);
		String requestXML = WxpayUtils.toXml(parameters);


		logger.info("请求参数：" + requestXML);
		String message = "发放红包失败";

		try {
			String response = ClientCustomSSL.doPost(config.getMchId(), config.getCertPath(), WX_WEAPP_RED_PACKAGE, requestXML);
			logger.info("=======================微信红包======================");
			logger.info(response);
			Map<String, String> map = WxpayUtils.doXMLParse(response);
			if("SUCCESS".equals(map.get("return_code")) && "SUCCESS".equals(map.get("result_code"))){
				String packageStr = map.containsKey("package") ? URLEncoder.encode(map.get("package"), "utf-8") : null;
				RedPackageResult result = new RedPackageResult();
				result.setMchBillno(map.get("mch_billno"));
				result.setMchId(map.get("mch_id"));
				result.setWxappid(map.get("wxappid"));
				result.setReOpenid(map.get("re_openid"));
				result.setTotalAmount(new BigDecimal(map.get("total_amount")).divide(new BigDecimal(100), 2, RoundingMode.HALF_DOWN));
				result.setSendListid(map.get("send_listid"));
				result.setPackageStr(packageStr);
				result.setMessage(map.get("return_msg"));

				if(packageStr != null && !"".equals(packageStr)){
					String timeStamp = getTimestamp();
					SortedMap<String, Object>  params=new TreeMap<String, Object>();
					params.put("appId", config.getAppId());
					params.put("timeStamp", timeStamp);
					String nonceStr = WxpayUtils.createSign(params, config.getApiKey());
					params.put("nonceStr", nonceStr);
					params.put("package", packageStr);

					result.setTimeStamp(timeStamp);
					result.setNonceStr(nonceStr);
					result.setSignType("MD5");
					result.setPaySign(WxpayUtils.createSign(params, config.getApiKey()));
				}


				return result;
			}
			message = map.get("return_msg");
		} catch ( KeyManagementException | UnrecoverableKeyException
				| NoSuchAlgorithmException | CertificateException | KeyStoreException | IOException e) {
			logger.error("微信红包失败" + e.getMessage());
			throw new PayException("微信红包失败");

		}
		throw new PayException(message);

	}
	
	/**
	 * 时间戳
	 * @return
	 */
	private static String getTimestamp(){
		return String.valueOf(System.currentTimeMillis() / 1000);
	}


}