package com.crawler.redis.utils;

import java.io.*;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.poi.ss.formula.functions.T;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.crawler.redis.config.JacksonObjectMapper;
import com.crawler.redis.exception.SimpleRuntimeException;
import com.crawler.redis.vo.RedisPage;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;
import redis.clients.jedis.JedisPubSub;
import redis.clients.jedis.exceptions.JedisException;

/**
 * redis 管理器
 * @author rubekid
 *
 * 2016年12月17日 下午10:19:25
 */
@SuppressWarnings("hiding")
public class RedisManager {
	
	/**
	 * 日志
	 */
	private static Logger logger = LoggerFactory.getLogger(RedisManager.class);
	
	/**
	 * 等待超时毫秒数
	 */
	private static int WAIT_TIMEOUT_MSECS = 60000;
	
	/**
	 * 锁自动过期毫秒数
	 */
	private static int LOCK_EXPIRE_MSECS = 60000;
	
	/**
	 * 
	 * jedis池
	 */
	private static JedisPool pool;
	
	/**
	 * 数据库
	 */
	private static int DB_INDEX = 0;


	 /**
	  * 静态代码初始化池配置
	  */
	static {
		init(null);
	}

	public static void init(Properties props){
		try {
			if(pool != null){
				pool.destroy();
			}
			if(props == null){
				String fileName = "redis.properties";
				props = new Properties();
				String userDir = System.getProperty("user.dir");
				String configPath = userDir + File.separator + "config" + File.separator + fileName;
				File configFile = new File(configPath);
				if(configFile.exists()){
					InputStream in = new BufferedInputStream(new FileInputStream(configFile));
					props.load(in);
				}
				else{
					props.load(RedisManager.class.getClassLoader().getResourceAsStream(fileName));
				}
			}

			if(props == null){
				logger.error("redis init fail");
				return ;
			}

			JedisPoolConfig config = new JedisPoolConfig();
			config.setMaxTotal(Integer.valueOf(props.getProperty("redis.pool.maxTotal")));
			config.setMaxIdle(Integer.valueOf(props.getProperty("redis.pool.maxIdle")));
			config.setMaxWaitMillis(Long.valueOf(props.getProperty("redis.pool.maxWaitMillis")));
			config.setTestOnBorrow(Boolean.valueOf(props.getProperty("redis.pool.testOnBorrow")));
			config.setTestOnReturn(Boolean.valueOf(props.getProperty("redis.pool.testOnReturn")));
			config.setTestWhileIdle(Boolean.valueOf(props.getProperty("redis.pool.testWhileIdle")));

			String host = props.getProperty("redis.host");
			Integer port = Integer.valueOf(props.getProperty("redis.port"));
			Integer timeout = Integer.valueOf(props.getProperty("redis.timeout"));
			String password = props.getProperty("redis.password");

			DB_INDEX = Integer.valueOf(props.getProperty("redis.dbindex", "0"));
			StringBuilder sb = new StringBuilder();
			sb.append("---------------- REDIS CONFIG ------------");
			sb.append("\nHOST:").append(host);
			sb.append("\nPORT:").append(port);
			sb.append("\nPASSWORD:").append(password);
			sb.append("\nINDEX:").append(DB_INDEX);

			logger.info(sb.toString());

			if(password !=null){
				pool = new JedisPool(config, host, port, timeout, password);
			}
			else{
				pool = new JedisPool(config, host, port, timeout);
			}

		} catch (IOException ex) {
			logger.error(ex.getMessage(), ex);
		}
	}

	
	/**
	 * 获取线程池信息
	 * @return
	 */
	public static Map<String, Object> getPoolInfo(){
		Map<String, Object> map = new HashMap<String, Object>();
		map.put("maxBorrowWaitTimeMillis", pool.getMaxBorrowWaitTimeMillis());
		map.put("meanBorrowWaitTimeMillis", pool.getMeanBorrowWaitTimeMillis());
		map.put("active", pool.getNumActive());
		map.put("idle", pool.getNumIdle());
		map.put("waiters", pool.getNumWaiters());
		return map;
	}
	
	/**
	 * 获取db index
	 * @return
	 */
	public static int getDbIndex(){
		return DB_INDEX;
	}
	
	/**
	 * 同步锁
	 * @param lockKey
	 * @param waitTimeout
	 * @return
	 */
	public static boolean lock(String lockKey, Integer waitTimeout){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			if(waitTimeout == null){
				waitTimeout = WAIT_TIMEOUT_MSECS;
			}
	        while (waitTimeout >= 0) {
	            long time = System.currentTimeMillis() + LOCK_EXPIRE_MSECS + 1;
	            String expireAt = String.valueOf(time);
	            if (jedis.setnx(lockKey, expireAt) == 1){
	                return true;
	            }

	            String currentValue = jedis.get(lockKey); 
	            if (currentValue != null && Long.parseLong(currentValue) < System.currentTimeMillis()) {

	                String oldValue = jedis.getSet(lockKey, expireAt);
	                //获取上一个锁到期时间，并设置现在的锁到期时间，
	                //只有一个线程才能获取上一个线上的设置时间，因为jedis.getSet是同步的
	                if(oldValue != null && oldValue.equals(currentValue)) {
	                    //如过这个时候，多个线程恰好都到了这里，但是只有一个线程的设置值和当前值相同，他才有权利获取锁
	                    return true;
	                }
	            }
	            waitTimeout -= 100;
	            try {
	            	if(waitTimeout >= 0){	            		
	            		Thread.sleep(100);
	            	}
				} catch (InterruptedException ex) {
					logger.error(ex.getMessage(), ex);
				}
	        }
	        return false;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}

		}
		
	}

	/**
	 * 同步锁
	 * @param lockKey
	 * @return
	 */
	public static boolean lock(String lockKey){
		return lock(lockKey, null);
	}
	
	/**
	 * 解除锁定
	 * @param lockKey
	 */
	public static void unlock(String lockKey){
		Jedis jedis = null;
		try{
			jedis = getJedis();
	        jedis.del(lockKey);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 尝试获取锁
	 * @param lockKey
	 * @param seconds
	 * @return
	 */
	public static boolean tryLock(String lockKey, int seconds){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			boolean bool = jedis.setnx(lockKey, ".lock") == 1;
			// 获取成功设置过期时间
			if(bool && seconds > 0) {
				jedis.expire(lockKey, seconds);
			}
			return bool;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 设置过期时间
	 * @param key
	 * @param seconds
	 * @return
	 */
	public static boolean expire(String key, int seconds) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.expire(key, seconds) == 1;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}

	/**
	 * 设置过期时间点
	 * @param key
	 * @param timestamp
	 * @return
	 */
	public static boolean expireAt(String key, long timestamp) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.expireAt(key, timestamp) == 1;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 获取剩余过期时间 -2 不存在 -1 没有设置过期时间 
	 * @param key
	 * @return
	 */
	public static long ttl(String key) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.ttl(key);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}

	/**
	 * 移除过期设置
	 * @param key
	 * @return
	 */
	public static boolean persist(String key) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.persist(key)  == 1;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 批量删除
	 * @param prefix
	 */
	public static void batchDeleteByKey(String prefix){
		Jedis jedis = null;
		try{
			jedis = getJedis();
	        Set<String> set = jedis.keys(prefix + "*");  
	        Iterator<String> it = set.iterator();
	        List<String> keys = new ArrayList<String>();
	        while(it.hasNext()){
	        	keys.add(it.next());
	        	if(keys.size() > 0 && keys.size() % 100 == 0){
	        		jedis.del(keys.toArray(new String[keys.size()]));
	        		keys.clear();
	        	}
	        	
	        }   
	        if(keys.size() > 0){
	        	jedis.del(keys.toArray(new String[keys.size()]));  
	        }
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}

	/**
     * 获取Jedis实例
     * 
     * @return
     */
    public synchronized static Jedis getJedis() {
    	Jedis resource = null;
        try {
            if (pool != null) {
                resource = pool.getResource();
                resource.select(DB_INDEX);
            }
        } catch (RuntimeException ex) {
        	 if(resource != null ) {
        		 logger.error("=========== Jedis 获取 resource 失败 ============");
                 resource.close();
             }
            logger.error(ex.getMessage(), ex);
        }
        if(resource== null){        	
        	throw new NullPointerException("jedis is null, please check the redis server.");
        }
        return resource;
    }
	
	/**
	 * 关闭
	 * @param jedis
	 */
	public static void close(Jedis jedis) {
		jedis.close();
	}
	
	/**
	 * 通过key获取
	 * @param key
	 * @return
	 */
	public static String get(String key, String defaultValue){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			String value = jedis.get(key);
			return value == null ? defaultValue : value;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 通过key获取
	 * @param key
	 * @return
	 */
	public static String get(String key){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			String value = jedis.get(key);
			return value;
		}
		catch(JedisException ex){
			logger.error(ex.getMessage(), ex);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
		return null;
	}
	
	/**
	 * 保存
	 * @param key
	 * @param value
	 */
	public static void set(String key, String value){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.set(key, value);
		}
		catch(JedisException ex){
			logger.error(ex.getMessage(), ex);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 保存哈希类型
	 * @param key
	 * @param obj
	 */
	public static void hset(String key, Object obj){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			Map<String, String> hash = toMap(obj);
			jedis.hmset(key, hash);
		}
		catch(JedisException ex){
			logger.error(ex.getMessage(), ex);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 设置某一字段
	 * @param key
	 * @param field
	 * @param value
	 */
	public static Long hset(String key, String field, Object value) {
		return hset(key, field, String.valueOf(value));
	}

	/**
	 * 设置某一字段
	 * @param key
	 * @param field
	 * @param value
	 */
	public static Long hset(String key, String field, String value) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.hset(key, field, value);
		}
		catch(JedisException ex){
			logger.error(ex.getMessage(), ex);
		}
		finally{
			if(null != jedis){
				jedis.close();
			}
		}
		return -1L;
	}
	
	/**
	 * 保存哈希类型 并设置过期时间
	 * @param key
	 * @param obj
	 */
	public static void hsetExpireIn(String key, Object obj, int seconds){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			Map<String, String> hash = toMap(obj);
			jedis.hmset(key, hash);
			jedis.expire(key, seconds);
		}
		catch(JedisException ex){
			logger.error(ex.getMessage(), ex);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}

	/**
	 * 虎丘哈希表中某字段
	 * @param key
	 * @param field
	 * @return
	 */
	public static String hget(String key, String field){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.hget(key, field);
		}
		catch(JedisException ex){
			logger.error(ex.getMessage(), ex);
		}
		finally{
			if(null != jedis){
				jedis.close();
			}
		}
		return null;
	}
	
	/**
	 * 通过哈希类型获取对象
	 * @param key
	 * @param clazz
	 */
	public static <T> T  hgetAll(String key, Class<T> clazz){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			Map<String, String> hash = jedis.hgetAll(key);
			if(hash != null && hash.size() > 0){
				return parse(toJSONString(hash), clazz);
			}
		}
		catch(JedisException ex){
			logger.error(ex.getMessage(), ex);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
		return null;
	}

    /**
     * 设置键的字符串值(指定超时)
     * @param key
     * @param data
     * @param seconds
     */
    public static void set(String key, Object data, int seconds){
    	String value = null;
    	if(data instanceof String){
    		value = String.valueOf(data);
    	}
    	else{
    		value = toJSONString(data);
    	}
    	Jedis jedis = null;
		try{
			jedis  = getJedis();
			jedis.setex(key, seconds, value);
		}
		catch(JedisException ex){
			logger.error(ex.getMessage(), ex);
		}
		finally{
			close(jedis);
		}
    }
	
	/**
	 * 根据key 获取对象
	 * @param key
	 * @param clazz
	 * @return
	 */
	public static <T> T get(String key, Class<T> clazz){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			String value = jedis.get(key);
			if(isNullOrEmpty(value)){
				return null;
			}
			return parse(value, clazz);
		}
		catch(JedisException ex){
			logger.error(ex.getMessage(), ex);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
		return null;
		 
	}
	
	/**
	 * 根据key 获取对象
	 * @param index
	 * @param key
	 * @param clazz
	 * @return
	 */
	public static <T> T get(int index, String key, Class<T> clazz){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			jedis.select(index);
			String value = jedis.get(key);
			if(isNullOrEmpty(value)){
				return null;
			}
			return parse(value, clazz);
		}
		catch(JedisException ex){
			logger.error(ex.getMessage(), ex);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
		return null;
	}
	
	/**
	 * 根据key 获取对象
	 * @param index
	 * @param key
	 * @return
	 */
	public static String get(int index, String key){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			jedis.select(index);
			return jedis.get(key);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 根据key 获取列表
	 * @param key
	 * @param clazz
	 * @return
	 */
	public static <T> List<T> findList(String key, Class<T> clazz, String ... exceptKeys){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			List<String> data = new ArrayList<String>();
			if(key.contains("*")){
				Set<String> keys = jedis.keys(key);
				List<String> exceptKeyList = Arrays.asList(exceptKeys);
				for(String k : keys){
					if(exceptKeyList.contains(k)){
						continue;
					}
					data.addAll(jedis.lrange(k, 0, -1));
				}
			}
			else{
				data = jedis.lrange(key, 0, -1);
			}

			List<T> list = new ArrayList<T>();
			for(String string : data){
				list.add(parse(string, clazz));
			}
			return list;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
		
	}
	
	/**
	 *  根据前缀获取列表
	 * @param <T>
	 * @param prefix
	 * @param clazz
	 * @return
	 */
	public static <T> List<T> findListLike(String prefix,  Class<T> clazz){
		List<T> list = new ArrayList<T>();
		Jedis jedis = null;
		try{
			jedis  = RedisManager.getJedis();
			Set<String> keys = jedis.keys(prefix + "*");
			for(String key :keys) {
				String value = jedis.get(key);
				if(value == null || "".equals(value)){
					continue ;
				}
				list.add(parse(value, clazz));
			}
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
		return list;
		
	}
	
	
	/**
	 * 获取列表
	 * @param key
	 * @return
	 */
	public static List<String> findList(String key){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			List<String> data = new ArrayList<String>();
			data = jedis.lrange(key, 0, -1);
			return data;
			
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
		
	}
	
	/**
	 * 分页
	 * @param page
	 * @param key
	 * @return
	 */
	public static RedisPage<T> findPage(RedisPage<T> page, String key){
		Jedis jedis = null;
		try{
			int start = page.getFirst();
			int end = start + page.getPageSize();
			jedis  = getJedis();
			List<String> data = jedis.lrange(key, start, end);
			List<T> list = new ArrayList<T>();
			for(String string : data){
				list.add(parse(string, T.class));
			}
			page.setItems(list);
			return page;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 设置值
	 * @param key
	 * @param data
	 */
	public static void put(String key, Object data){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			jedis.set(key, toJSONString(data));
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 设置列表
	 * @param key
	 * @param list
	 */
	public static <E> void setList(String key, List<E> list){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.del(key);
			String[] strings = toStrings(list);
			if(strings.length > 0){				
				jedis.rpush(key, strings);
			}
			else{
				throw new SimpleRuntimeException("数据错误");
			}
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 新增
	 * @param key
	 * @param item
	 */
	public static void insert(String key, Object item){
		rpush(key, item);
	}
	
	/**
	 * 从右侧新增
	 * @param key
	 * @param item
	 */
	public static void append(String key, Object item) {
		rpush(key, item);
	}
	
	/**
	 * 批量加入列表
	 * @param <E>
	 * @param key
	 * @param items
	 */
	public static <E> void batchAppend(String key, List<E> items) {
		if(items.size() > 0) {
			Jedis jedis = null;
			try{
				jedis = getJedis();
				String[] strings = toStrings(items);
				if(strings.length > 0){				
					jedis.rpush(key, strings);
				}
				else{
					throw new SimpleRuntimeException("数据错误");
				}
			}
			finally{
				if(null != jedis){				
					jedis.close();
				}
			}
		}
	}
	
	/**
	 * rpush
	 * @param key
	 * @param item
	 */
	public static void rpush(String key, Object item){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.rpush(key, toJSONString(item));
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * lpush
	 * @param key
	 * @param item
	 */
	public static void lpush(String key, Object item) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.lpush(key, toJSONString(item));
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 获取键值列表
	 * @param pattern
	 * @return
	 */
	public static Set<String> keys(String pattern){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			Set<String> keys =  jedis.keys(pattern);
			return keys;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 设置值
	 * @param key
	 * @param obj
	 */
	public static void set(String key, Object obj){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.set(key, toJSONString(obj));
		}
		catch(JedisException ex){
			logger.error(ex.getMessage(), ex);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 设置值
	 * @param index
	 * @param key
	 * @param obj
	 */
	public static void set(int index, String key, Object obj){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.select(index);
			jedis.set(key, toJSONString(obj));
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 自增
	 * @param index
	 * @param key
	 * @param value
	 */
	public static long incrBy(int index, String key, long value) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.select(index);
			return jedis.incrBy(key, value);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 自增
	 * @param key
	 * @param value
	 */
	public static long incrBy(String key, long value) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.incrBy(key, value);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 *   自减
	 * @param index
	 * @param key
	 * @param value
	 */
	public static long decrBy(int index, String key, long value) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.select(index);
			return jedis.decrBy(key, value);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 *  自减
	 * @param key
	 * @param value
	 */
	public static long decrBy(String key, long value) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.decrBy(key, value);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 自增加 浮点型
	 * @param index
	 * @param key
	 * @param value
	 */
	public static void incrByFloat(int index, String key, float value) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.select(index);
			jedis.incrByFloat(key, value);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 自增加 浮点型
	 * @param key
	 * @param value
	 */
	public static void incrByFloat(String key, float value) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.incrByFloat(key, value);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 哈希表字段自增
	 * @param index
	 * @param key
	 * @param field
	 * @param value
	 */
	public static long hincrBy(int index, String key, String field, long value) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.select(index);
			return jedis.hincrBy(key, field, value);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 哈希表字段自增
	 * @param key
	 * @param field
	 * @param value
	 */
	public static long hincrBy(String key, String field, long value) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.hincrBy(key, field, value);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	
	/**
	 * lset
	 * @param key
	 * @param index
	 * @param obj
	 */
	public static void lset(String key, int index, Object obj){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.lset(key, index, toJSONString(obj));
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 删除
	 * @param key
	 */
	public static void remove(String key){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.del(key);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	
	/**
	 * 发布
	 * @param channel
	 * @param obj
	 */
	public static void publish(String channel, Object obj){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			String data = toJSONString(obj);
			jedis.publish(channel, data);
			logger.info("\n================== 发布消息 ========================\n"
					+ "{} {}"
					+ "\n================== 发布消息结束 ========================", channel, data);

		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 订阅
	 * @param jedisPubSub
	 * @param channels
	 */
	public static void subscribe(final JedisPubSub jedisPubSub, final String... channels){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.subscribe(jedisPubSub, channels);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 根据key pop对象
	 * @param key
	 * @param clazz
	 * @return
	 */
	public static <T> T pop(String key, Class<T> clazz){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			String value = jedis.lpop(key);
			if(isNullOrEmpty(value)){
				return null;
			}

			return parse(value, clazz);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 根据key pop 字符串
	 * @param key
	 * @return
	 */
	public static String pop(String key){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			return jedis.lpop(key);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 根据key rpop 字符串
	 * @param key
	 * @return
	 */
	public static String rpop(String key){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			return jedis.rpop(key);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 根据key pop对象
	 * @param key
	 * @param clazz
	 * @return
	 */
	public static <T> T blpop(String key, Class<T> clazz){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			List<String> result = jedis.blpop(0, key);
			if(result.size() == 2 && result.get(0).equals(key)) {
				return parse(result.get(1), clazz);
			}
			return null;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 根据key pop对象
	 * @param key
	 * @param clazz 
	 * @param timeout 超时 （秒）
	 * @return
	 */
	public static <T> T blpop(String key, Class<T> clazz, int timeout){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			List<String> result = jedis.blpop(timeout, key);
			if(result.size() == 2 && result.get(0).equals(key)) {
				return parse(result.get(1), clazz);
			}
			return null;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 根据key pop 字符串
	 * @param key
	 * @return
	 */
	public static String blpop(String key){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			List<String> result = jedis.blpop(0, key);
			if(result.size() == 2 && result.get(0).equals(key)) {
				return result.get(1);
			}
			return null;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 移除列表中与参数 VALUE 相等的元素
	 * @param key
	 * @param value
	 */
	public static void lrem(String key, String value) {
		lrem(key, 0, value);
	}
	
	/**
	 * 移除列表中与参数 VALUE 相等的元素
	 * @param key
	 * @param count
	 * @param value
	 */
	public static void lrem(String key, Integer count, String value) {
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			jedis.lrem(key, count, value);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	
	/**   集合   **/
	
	/**
	 * 向集合添加成员
	 */
	public static void sadd(String key, Object ... items) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			String[] strings = toStrings(items);
			if(strings.length > 0){				
				jedis.sadd(key, strings);
			}
			else{
				throw new SimpleRuntimeException("数据错误");
			}
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 向集合添加成员
	 */
	public static <E> void sadd(String key, List<E> list) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			String[] strings = toStrings(list);
			if(strings.length > 0){				
				jedis.sadd(key, strings);
			}
			else{
				throw new SimpleRuntimeException("数据错误");
			}
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}

	/**
	 * 返回集合中的一个随机元素
	 * @param key
	 */
	public static String srandmember(String key) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.srandmember(key);
		}
		finally{
			if(null != jedis){
				jedis.close();
			}
		}
	}

	/**
	 * 返回集合中的几个随机元素
	 * @param key
	 * @param count
	 * @return
	 */
	public static List<String> srandmember(String key, int count) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.srandmember(key, count);
		}
		finally{
			if(null != jedis){
				jedis.close();
			}
		}
	}
	
	/**
	 *   命令判断成员元素是否是集合的成员
	 * @param key
	 * @param item
	 * @return
	 */
	public static boolean sismember(String key, Object item) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.sismember(key, toJSONString(item));
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 *   返回集合中的所有成员
	 * @param <T>
	 * @param key
	 * @param clazz
	 * @return
	 */
	public static <T> List<T> smembers(String key, Class<T> clazz){
		Jedis jedis = null;
		List<T> list = new ArrayList<T>();
		try{
			jedis = getJedis();
			Set<String> data = jedis.smembers(key);
			for(String string : data){
				list.add(parse(string, clazz));
			}
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
		return list;
	}
	
	/**
	 * 返回集合中的所有成员
	 * @param key
	 * @return
	 */
	public static List<String> smembers(String key){
		Jedis jedis = null;
		List<String> list = new ArrayList<String>();
		try{
			jedis = getJedis();
			Set<String> data = jedis.smembers(key);
			for(String string : data){
				list.add(string);
			}
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
		return list;
	}
	
	/**
	 *  返回集合中元素的数量
	 * @param key
	 * @return
	 */
	public static Long scard(String key) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.scard(key);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 *  移除集合中一个或多个成员
	 */
	public static void srem(String key, Object ... items) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			String[] strings = toStrings(items);
			if(strings.length > 0){				
				jedis.srem(key, strings);
			}
			else{
				throw new SimpleRuntimeException("数据错误");
			}
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	

	
	/**
	 * 根据key pop对象
	 * @param key
	 * @param clazz
	 * @return
	 */
	public static <T> T spop(String key, Class<T> clazz){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			String value = jedis.spop(key);
			if(isNullOrEmpty(value)){
				return null;
			}

			return parse(value, clazz);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 根据key pop 字符串
	 * @param key
	 * @return
	 */
	public static String spop(String key){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			return jedis.spop(key);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	
	
	
	/** 有序集合 **/
	
	/**
	 * 新增记录
	 * @param key
	 * @param score
	 * @param item
	 */
	public static void zadd(String key, Double score, Object item) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.zadd(key, score, toJSONString(item));
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 *  通过分数返回有序集合指定区间内的成员
	 * @param <T>
	 * @param key
	 * @param minScore
	 * @param maxScore
	 * @param clazz
	 * @return
	 */
	public static <T> List<T> zrangeByScore(String key, Double minScore, Double maxScore, Class<T> clazz){
		List<T> list = new ArrayList<T>();
		Jedis jedis = null;
		try{
			jedis = getJedis();
			Set<String> data = jedis.zrangeByScore(key, minScore, maxScore);
			
			
			for(String string : data){
				list.add(parse(string, clazz));
			}
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
		return list;
	}
	
	/**
	 *  返通过索引区间返回有序集合成指定区间内的成员
	 * @param <T>
	 * @param key
	 * @param start
	 * @param end
	 * @param clazz
	 * @return
	 */
	public static <T> List<T> zrange(String key, Long start, Long end, Class<T> clazz){
		List<T> list = new ArrayList<T>();
		Jedis jedis = null;
		try{
			jedis = getJedis();
			Set<String> data = jedis.zrange(key, start, end);
			for(String string : data){
				list.add(parse(string, clazz));
			}
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
		return list;
	}
	
	
	/**
	 *  返回有序集中指定分数区间内的成员，分数从高到低排序
	 * @param <T>
	 * @param key
	 * @param maxScore
	 * @param minScore
	 * @param clazz
	 * @return
	 */
	public static <T> List<T> zrevrangeByScore(String key, Double maxScore, Double minScore, Class<T> clazz){
		List<T> list = new ArrayList<T>();
		Jedis jedis = null;
		try{
			jedis = getJedis();
			Set<String> data = jedis.zrevrangeByScore(key, maxScore, minScore);
			
			
			for(String string : data){
				list.add(parse(string, clazz));
			}
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
		return list;
	}
	
	/**
	 *  返回有序集中指定区间内的成员，通过索引，分数从高到底
	 * @param <T>
	 * @param key
	 * @param start
	 * @param end
	 * @param clazz
	 * @return
	 */
	public static <T> List<T> zrevrange(String key, Long start, Long end, Class<T> clazz){
		List<T> list = new ArrayList<T>();
		Jedis jedis = null;
		try{
			jedis = getJedis();
			Set<String> data = jedis.zrevrange(key, start, end);
			for(String string : data){
				list.add(parse(string, clazz));
			}
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
		return list;
	}
	
	/**
	 * 移除有序集合中给定的分数区间的所有成员
	 * @param key
	 * @param minScore
	 * @param maxScore
	 */
	public static void zremrangeByScore(String key, Double minScore, Double maxScore) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.zremrangeByScore(key, minScore, maxScore);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 *   命令用于移除有序集合中给定的字典区间的所有成员。
	 * @param key
	 * @param min
	 * @param max
	 */
	public static void zremrangeByLex(String key, String min, String max) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.zremrangeByLex(key, min, max);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	  *  移除有序集合中给定的排名区间的所有成员
	 * @param key
	 * @param start
	 * @param end
	 */
	public static void zremrangeByRank(String key, Long start, Long end) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.zremrangeByRank(key, start, end);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 *  移除有序集合中一个或多个成员
	 */
	public static void zrem(String key, Object ... items) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			String[] strings = toStrings(items);
			if(strings.length > 0){				
				jedis.zrem(key, strings);
			}
			else{
				throw new SimpleRuntimeException("数据错误");
			}
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 *  返回有序集合中元素的数量
	 * @param key
	 * @return
	 */
	public static Long zcard(String key) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.zcard(key);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 返回有序集中指定成员的排名。其中有序集成员按分数值递增(从小到大)顺序排列
	 * @param key
	 * @param member
	 * @return
	 */
	public static Long zrank(String key, Object member) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.zrank(key, toJSONString(member));
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 返回有序集中，成员的分数值
	 * @param key
	 * @param member
	 * @return
	 */
	public static Double zscore(String key, Object member) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.zscore(key, toJSONString(member));
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}

	
	
	/**
	 * 判断key 是否存在
	 * @param key
	 * @return
	 */
	public static boolean exists(String key){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			return jedis.exists(key);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}

	
	/**
	 * 转字符串
	 * @param object
	 * @return
	 */
	public static String toJSONString(Object object) {
		if(object instanceof String){
			return (String) object;
		}
		ObjectMapper objectMapper = new JacksonObjectMapper();
		try {
			return objectMapper.writeValueAsString(object);
		} catch (JsonProcessingException ex) {
			logger.error(ex.getMessage(), ex);
		}
		return "";
	}
	
	/**
	 * 转出Map对象
	 * @param object
	 * @return
	 */
	public static Map<String, String> toMap(Object object){
		Map<String, String> map = new HashMap<String, String>();
		Map<?, ?> objMap = new HashMap<Object, Object>();
		if(object instanceof Map){
			objMap = (Map<?, ?>)object;
			
		}
		else{
			if(!(object instanceof String)){
				objMap = parse(toJSONString(object), Map.class);
			}
			else{
				objMap = parse((String)object, Map.class);
			}
		}
		for(Map.Entry<?, ?> entry : objMap.entrySet()){
			Object obj = entry.getValue();
			String value = String.valueOf(entry.getValue());
			if(obj instanceof List) {
				value = toJSONString(obj); 
			}
			map.put(String.valueOf(entry.getKey()), value);
		}
		
		return map;
	}
	
	/**
	 * 转对象
	 * @param value
	 * @param clazz
	 * @return
	 * @throws IOException
	 */
	@SuppressWarnings({ "unchecked"})
	public static <T> T parse(String value, Class<T> clazz){
		if(clazz == null || clazz.equals(String.class)){
			return (T) value;
		}
		ObjectMapper objectMapper = new JacksonObjectMapper();
		try {
			return objectMapper.readValue(value, clazz);
		} catch (IOException ex) {
			logger.error(ex.getMessage(), ex);
		}
		return null;
	}
	
	/**
	 * 列表转字符串数组
	 * @param list
	 * @return
	 */
	public static final <E> String[] toStrings(List<E> list){
		String[] items = new String[list.size()];
		for(int i=0; i<list.size(); i++){
			items[i] = toJSONString(list.get(i));
		}
		return items;
	}
	
	/**
	 * 对象数组转字符串数组
	 * @param <E>
	 * @param list
	 * @return
	 */
	public static final <E> String[] toStrings(E[] list){
		String[] items = new String[list.length];
		for(int i=0; i<list.length; i++){
			items[i] = toJSONString(list[i]);
		}
		return items;
	}
	

	/**
	 * 判断字符串是否为空
	 * 
	 * @param string
	 * @return
	 */
	public static boolean isNullOrEmpty(String string) {
		return string == null || "".equals(string);
	}

	

	

}
