package com.crawler.ffmpeg.utils;

import com.crawler.ffmpeg.bean.Audio;
import com.crawler.ffmpeg.bean.Video;
import com.crawler.ffmpeg.exception.FFmpegException;
import com.crawler.res.common.ResProperties;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


/**
 * ffmpeg 工具
 * 
 * @author rubekid
 *
 */
public class FFMpegUtils {
	
	private static final String CONTEXT_PATH = ResProperties.getProperty("ffmpeg.context.path", "/data");

	private static final String FONT_FILE = ResProperties.getProperty("ffmpeg.font.file");

	private static String ffmpegUri ;
	
	private static String mencoderUri;

	static{
		if(ffmpegUri==null){
			Properties prop = System.getProperties();
			String os = prop.getProperty("os.name");
			if(os.toLowerCase().startsWith("win")){
				ffmpegUri = CONTEXT_PATH + "/ffmpeg/windows/ffmpeg.exe";
				mencoderUri = CONTEXT_PATH + "/ffmpeg/windows/mencoder.exe";
			}
			else{
				ffmpegUri = CONTEXT_PATH + "/ffmpeg/linux/ffmpeg";
				mencoderUri = CONTEXT_PATH + "/ffmpeg/linux/mencoder";
			}
		}
	}

	public static String getFfmpegUri(){
		return ffmpegUri;
	}

	public static String getMencoderUri(){
		return mencoderUri;
	}
	
	/**
	 * 获取音频信息
	 * @param filePath 源文件路径
	 * @throws InterruptedException 
	 * @throws IOException 
	 * @throws Exception 
	 */
	public static Audio parseAudio(String filePath) throws IOException, InterruptedException {
		List<String> cmd = new ArrayList<String>();
		cmd.add(getFfmpegUri());
		cmd.add("-i");
		cmd.add(filePath);
		Audio audio = new Audio();
		String result = exec(cmd);
		if(result == ""){
			throw new FFmpegException("视频文件解析失败");
		}
		Matcher matcherInput = Pattern.compile("Input #0, (.*?), from (.*?)").matcher(result);
		Matcher baseMatcher  = Pattern.compile("Duration: (.*?),(.*?)bitrate: (\\d*) kb\\/s").matcher(result);
	    Matcher audioMatcher = Pattern.compile("Audio: (\\w*)(.*?), (\\d*) Hz").matcher(result);
	    if(matcherInput.find()){
	    	audio.setType(matcherInput.group(1));
	    }
	    if(baseMatcher.find()){
	    	audio.setDuration( runtimeToSecond(baseMatcher.group(1)));
	    	audio.setBitRate(Long.parseLong(baseMatcher.group(3)));
	    }
	    if(audioMatcher.find()){
	    	audio.setFormat(audioMatcher.group(1));
	    }
		
		return audio;	
	}
	
	/**
	 * 获取视频信息
	 * @param filePath 源文件路径
	 * @throws InterruptedException 
	 * @throws IOException 
	 * @throws Exception 
	 */
	public static Video parseVideo(String filePath) throws IOException, InterruptedException{
		List<String> cmd = new ArrayList<String>();
		cmd.add(getFfmpegUri());
		cmd.add("-i");
		cmd.add(filePath);
		Video video = new Video();

		video.setSrcFile(filePath);
		String result = exec(cmd);
		if(result == ""){
			throw new FFmpegException("视频文件解析失败");
		}
		
		Matcher matcherInput = Pattern.compile("Input #0, (.*?), from (.*?)").matcher(result);
		Matcher baseMatcher  = Pattern.compile("Duration: (.*?), start: (.*?), bitrate: (\\d*) kb\\/s").matcher(result);
		Matcher videoMatcher = Pattern.compile("Video: (\\w*)(.*?), (.*?), (.*?)[\\s\\S]*").matcher(result);  
	    Matcher audioMatcher = Pattern.compile("Audio: (\\w*)(.*?), (\\d*) Hz").matcher(result);
	    Matcher rotateMatcher = Pattern.compile("rotate(\\s*):(\\s*)(\\d+)").matcher(result);
	    if(matcherInput.find()){
	    	video.setType(matcherInput.group(1));
	    }
	    if(baseMatcher.find()){
	    	video.setDuration( runtimeToSecond(baseMatcher.group(1)));
	    	video.setBitRate(Integer.parseInt(baseMatcher.group(3)));
	    }
	    if(rotateMatcher.find()){
	    	int rotate = Integer.parseInt(rotateMatcher.group(3));
	    	video.setRotate((rotate+360) % 360);
	    }
	    if(videoMatcher.find()){
	    	String videoInfo = videoMatcher.group(0);
	    	Matcher m = Pattern.compile("([1-9]\\d*x[1-9]\\d*)").matcher(videoInfo);
	    	if(m.find()){
	    		video.setResolution(m.group(0));
	    		video.setDestResolution();
	    	}
	    	m = Pattern.compile("Unknown format").matcher(videoInfo);
	    	if(m.find()){
	    		video.setSupport(false);
	    	}
	    	else{
	    		video.setSupport(true);
	    	}
	    	video.setVideoFormat(videoMatcher.group(1));
	    }
	    if(audioMatcher.find()){
	    	video.setAudioFormat(audioMatcher.group(1));
	    }
	    
		return video;	
	}
	
	public static Process processAvi(String originFileUri , String fileSavePath) throws IOException, InterruptedException {
		List<String> commend = new ArrayList<String>();
		commend.add(getMencoderUri());
		commend.add(originFileUri);
		commend.add("-ovc");
		commend.add("lavc");
		commend.add("-oac");
		commend.add("mp3lame");
		commend.add("-lavcopts");
		commend.add("acodec=mp3:abitrate=64");
		commend.add("-xvidencopts");
		commend.add("bitrate=600");
		commend.add("-vf");
		commend.add("scale=320:-3");
		commend.add("-of");
		commend.add("avi");
		commend.add("-o");
		commend.add(fileSavePath);

		ProcessBuilder processBuilder = new ProcessBuilder();
		processBuilder.command(commend);
		Process p = processBuilder.start();
		return p;
	}
	
	/**
	 * 生成视频截图
	 * @param filePath 源文件路径
	 * @param imageSavePath 截图文件保存全路径
	 * @param screenSize 截图大小 如640x480
	 * @param transpose 旋转 
	 * @throws Exception 
	 */
	public static void makeScreenCut( String filePath, String imageSavePath , String screenSize ,String transpose ) throws IOException, InterruptedException{
		List<String> cmd = new ArrayList<String>();
		cmd.add(getFfmpegUri());
		cmd.add("-i");
		cmd.add(filePath);
		cmd.add("-y");
		cmd.add("-f");
		cmd.add("image2");
		cmd.add("-ss");
		cmd.add("1");
		cmd.add("-t");
		cmd.add("0.001");
		if(!StringUtils.isNullOrEmpty(transpose)){
			cmd.add("-vf");
			cmd.add(transpose);
		}
		cmd.add("-s");
		cmd.add(screenSize);
		cmd.add(imageSavePath);
		
		exec(cmd);
	}
	
	/**
	 * 音频转换
	 * @param filePath 源文件路径
	 * @param fileSavePath 文件保存全路径
	 * @param audioByte 音频比特率
	 * @param audioCollection 音频采样率
	 * @throws InterruptedException 
	 * @throws IOException 
	 * @throws Exception 
	 */
	public static void audioTransfer(String filePath, String fileSavePath,  int audioByte, int audioCollection) throws IOException, InterruptedException{
		List<String> cmd = new ArrayList<String>();
		cmd.add(getFfmpegUri());
		cmd.add("-i");
		cmd.add(filePath);
		cmd.add("-y");
		cmd.add("-ab");
		cmd.add( Integer.toString(audioByte) );
		cmd.add("-ar");
		cmd.add( Integer.toString(audioCollection) );
		cmd.add("-ac");
		cmd.add("1");
		cmd.add( fileSavePath );
		exec(cmd);
	}
	
	/**
	 * 视频转换(转为libxvid)
	 * @param filePath 源文件路径
	 * @throws Exception 
	 */
	public static String toLibxvid(String filePath) throws Exception{
		String ext = StringUtils.getExtension(filePath);
		String fileSavePath = filePath.replace("." + ext, ".tmp."+ext );
		List<String> cmd = new ArrayList<String>();
		cmd.add(getFfmpegUri());
		cmd.add("-i");
		cmd.add(filePath);
		cmd.add("-y");
		cmd.add("-c:v");
		cmd.add("libxvid");			
		cmd.add( fileSavePath );
		exec(cmd);
		return fileSavePath;
	}
	
	/**
	 * mp4压缩
	 * @param filePath
	 * @param savePath
	 * @throws InterruptedException 
	 * @throws IOException 
	 */
	public static void resize(String filePath, String savePath, String bitRate, String resolution) throws IOException, InterruptedException{
		List<String> cmd = new ArrayList<String>();
		cmd.add(getFfmpegUri());
		cmd.add("-i");
		cmd.add(filePath);
		cmd.add("-b:v");
		cmd.add( bitRate );
		cmd.add("-s");
		cmd.add( resolution);		
		cmd.add( savePath );
		exec(cmd);
	}
	
	/**
	 * 视频转换
	 * @param filePath 源文件路径
	 * @param savePath 文件保存全路径
	 * @param resolution 视频分辨率 如640x480
	 * @param audioByte 音频比特率
	 * @param audioCollection 音频采样率
	 * @param quality 视频质量(1-51)越低质量越好( 默认23)
	 * @param preset (压制速度placebo<veryslow<slower<slow<medium<fast<faster<veryfast<superfast<ultrafast) 默认medium,压缩率与速度成反比
	 * @param fps 每秒帧数（15或29.97）
	 * @param transpose 旋转
	 * @throws InterruptedException 
	 * @throws IOException 
	 * @throws Exception 
	 * 
	 * 
	 */
	public static void videoTransfer(String filePath, String savePath, String resolution, int audioByte, int audioCollection, double quality, String preset,  double fps, String transpose) throws IOException, InterruptedException {

		List<String> cmd = new ArrayList<String>();
		cmd.add(getFfmpegUri());
		cmd.add("-i");
		cmd.add(filePath);
		cmd.add("-y");
		cmd.add("-ab");
		cmd.add( Integer.toString(audioByte) );
		cmd.add("-ar");
		cmd.add( Integer.toString(audioCollection) );
		cmd.add("-r");
		cmd.add( Double.toString(fps) );
		cmd.add("-crf");
		cmd.add( Double.toString(quality) );
		cmd.add("-preset");
		cmd.add(preset);
		cmd.add("-profile:v");
		cmd.add("baseline");
		if(!StringUtils.isNullOrEmpty(transpose)){
			cmd.add("-vf");
			cmd.add(transpose);
			cmd.add("-metadata:s:v:0");
			cmd.add("rotate=0"); 
		}
		cmd.add("-s");
		cmd.add(resolution);			
		cmd.add( savePath );
		exec(cmd);
	}
	
	/**
	 * 
	 * @param filePath
	 * @param savePath
	 * @param resolution
	 * @param transpose
	 * @throws InterruptedException 
	 * @throws IOException 
	 * @throws Exception 
	 */
	public static void videoTransfer(String filePath, String savePath, String resolution, String transpose) throws IOException, InterruptedException{
		videoTransfer(filePath, savePath, resolution, 43000, 44100, 23, "medium", 15, transpose);
	}

	/**
	 * 设置文本
	 * @param filePath
	 * @param savePath
	 * @param text
	 * @param x
	 * @param y
	 * @param fontSize
	 * @throws IOException
	 * @throws InterruptedException
	 */
	public static void markText(String filePath, String savePath, String text, int x, int y, int fontSize) throws IOException, InterruptedException {
		List<String> cmd = new ArrayList<String>();
		cmd.add(getFfmpegUri());
		cmd.add("-i");
		cmd.add(filePath);
		cmd.add("-vf");
		cmd.add("drawtext=fontcolor=white:fontsize="+fontSize+":fontfile='"+ FONT_FILE +"':text='"+ text +"':x="+ x +":y=" + y);
		cmd.add("-y");
		cmd.add( savePath );
		exec(cmd);


	}
	
	/**
	 * 把 hh:mm:ss 转为秒
	 * @param str
	 * @return
	 */
	private static int runtimeToSecond(String str){
		int second = 0;
		String[] s = str.split(":");
		if(s.length == 3){
			second = Integer.parseInt(s[0])*60*60 + Integer.parseInt(s[1])*60 + (int)Math.abs(Double.parseDouble(s[2]));
		}
		return second;
	}
	
	/**
	 * 执行指令
	 * @param cmd 执行指令
	 * @throws IOException 
	 * @throws InterruptedException 
	 * @throws Exception 
	 */
	public static String exec( List<String> cmd) throws IOException, InterruptedException{
		String outPut = "";
		ProcessBuilder builder = new ProcessBuilder();
        builder.command(cmd);
        builder.redirectErrorStream(true);
        Process proc = builder.start();
        BufferedReader stdout = new BufferedReader(new InputStreamReader(proc.getInputStream()));
        String line;
        while ((line = stdout.readLine()) != null) {
        	outPut +="\n" + line;
        	System.out.println(line);
        	if(line.indexOf("Error") == 0){
        		//logger.error(line);
        	}
		}
        proc.waitFor();   
        stdout.close();
        String lastLine = outPut.substring(outPut.lastIndexOf("\n")+1);
        if(lastLine.indexOf("Error") == 0){
        	throw new RuntimeException(lastLine);
        }
		return outPut;
	}
}
