package com.crawler.res.common;

import java.awt.image.BufferedImage;
import java.io.*;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.UUID;
import java.util.regex.Matcher;

import javax.imageio.ImageIO;

import com.crawler.res.utils.ImageUtils;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.multipart.commons.CommonsMultipartFile;

import com.crawler.ffmpeg.bean.Audio;
import com.crawler.ffmpeg.bean.Video;
import com.crawler.ffmpeg.utils.FFMpegUtils;
import com.crawler.res.image.Image;
import com.crawler.res.qiniu.QiniuClient;
import com.crawler.res.utils.FileUtils;
import com.crawler.res.utils.QiNiuUtils;
import com.qiniu.common.QiniuException;
import com.qiniu.http.Client;

import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

/**
 * 上传基类
 * @author Rubekid
 *
 * 2017年4月15日 下午9:57:43
 */
public abstract class UploaderBaseController {
	
	
	/**
	 * 获取上传token
	 * @return
	 */
	@RequestMapping(value = "/token", method = RequestMethod.GET)
	public Object token(){
		String token = QiNiuUtils.getUpToken();
		JSONObject json = new JSONObject();
		json.put("value", token);
		return json;
	}

	/**
	 * 文件上传
	 * @param multipartFile
	 * @return
	 * @throws IOException
	 */
	@ResponseBody
	@RequestMapping(method = RequestMethod.POST, headers = "content-type=multipart/form-data")
	public JSONObject upload( @RequestParam("file") CommonsMultipartFile multipartFile) throws IOException{
		return doFileUpload(multipartFile);
		
	}
	
	/**
	 * 多文件上传
	 * @param multipartFiles1
	 * @param multipartFiles2
	 * @return
	 * @throws IOException
	 */
	@ResponseBody
	@RequestMapping(value = "/files", method = RequestMethod.POST, headers = "content-type=multipart/form-data")
	public JSONArray multiUpload( @RequestParam("files") List<CommonsMultipartFile> multipartFiles1, @RequestParam("files[]") List<CommonsMultipartFile> multipartFiles2) throws IOException{
		List<CommonsMultipartFile> multipartFiles = multipartFiles1 == null ? multipartFiles2 : multipartFiles1;
		JSONArray items = new JSONArray();
		for(CommonsMultipartFile multipartFile : multipartFiles){
			items.add(doFileUpload(multipartFile));
		}
		return items;
	}
	
	/**
	 * 图片上传
	 * @param multipartFile
	 * @return
	 * @throws Exception 
	 */
	@ResponseBody
	@RequestMapping(value = "/image", method = RequestMethod.POST, headers = "content-type=multipart/form-data")
	public JSONObject imageUpload( @RequestParam("image") CommonsMultipartFile multipartFile,
			@RequestParam(value = "showInfo", required = false, defaultValue="false") Boolean showInfo,
			@RequestParam(value = "width", required = false) Integer width,
			@RequestParam(value = "height", required = false) Integer height) throws Exception{
		return doImageUpload(multipartFile, showInfo, width, height);
	}

	/**
	 * 图片批量上传
	 * @param multipartFiles1
	 * @param multipartFiles2
	 * @param showInfo
	 * @param width
	 * @param height
	 * @return
	 * @throws Exception
	 */
	@ResponseBody
	@RequestMapping(value = "/images", method = RequestMethod.POST, headers = "content-type=multipart/form-data")
	public JSONArray batchImageUpload( @RequestParam(value = "images", required = false) List<CommonsMultipartFile> multipartFiles1, 
			@RequestParam(value = "images[]", required = false) List<CommonsMultipartFile> multipartFiles2,
			@RequestParam(value = "showInfo", required = false, defaultValue="false") Boolean showInfo,
			@RequestParam(value = "width", required = false) Integer width,
			@RequestParam(value = "height", required = false) Integer height) throws Exception{
		List<CommonsMultipartFile> multipartFiles = multipartFiles1 == null ? multipartFiles2 : multipartFiles1;
		JSONArray items = new JSONArray();
		for(CommonsMultipartFile multipartFile : multipartFiles){
			items.add(doImageUpload(multipartFile, showInfo, width, height));
		}
		return items;
	}

	/**
	 * 音频上传
	 * @param multipartFile
	 * @return
	 * @throws IOException
	 */
	@ResponseBody
	@RequestMapping(value = "/audio", method = RequestMethod.POST, headers = "content-type=multipart/form-data")
	public JSONObject audioUpload( @RequestParam("audio") CommonsMultipartFile multipartFile) throws IOException{
		return doFileUpload(multipartFile, ".mp3");
		
	}
	
	/**
	 * 视频上传
	 * @param multipartFile
	 * @return
	 * @throws IOException
	 */
	@ResponseBody
	@RequestMapping(value = "/video", method = RequestMethod.POST, headers = "content-type=multipart/form-data")
	public JSONObject videoUpload( @RequestParam("video") CommonsMultipartFile multipartFile,
			@RequestParam(value = "privately", defaultValue="false") Boolean privately,
			@RequestParam(value = "forceResize", defaultValue="false") Boolean forceResize,
			@RequestParam(value = "resolution", defaultValue="640x360") String resolution) throws IOException{
		String oFileName = multipartFile.getOriginalFilename();
		String ext = ".mp4";
		if(oFileName.contains(".")){
			ext = oFileName.substring(oFileName.lastIndexOf(".")).toLowerCase();
		}
		String baseFileName = FileUtils.createFileName();
		String fileName =  baseFileName + ext;
		String newFileName = baseFileName + ".mp4";
		JSONObject result = new JSONObject();
		
		// 不是Mp4类型或者强制压缩
		if(!newFileName.equals(fileName) || forceResize){
			fileName = "tmp/" + fileName;
			String url =  QiNiuUtils.uploadVideo(multipartFile.getBytes(), fileName);
			String persistentId = QiNiuUtils.toMp4(fileName, newFileName,resolution);
			result.put("persistentId", persistentId);
			result.put("url", url );
			result.put("preview", privately ? QiNiuUtils.getPrivateDownloadUrl(url) : url);
			return  result;
		}
		else{
			String url = QiNiuUtils.uploadVideo(multipartFile.getBytes(), newFileName);
			JSONObject info = QiNiuUtils.avinfo(url);
			result.put("url",  url);
			result.put("preview", privately ? QiNiuUtils.getPrivateDownloadUrl(url) : url);
			result.put("cover", QiNiuUtils.getVideoCover(url, info.getInt("width"), info.getInt("height")));
			result.put("size", info.get("size"));
			result.put("width", info.get("width"));
			result.put("height", info.get("height"));
			result.put("duration", info.get("duration"));
			return  result;
		}
	}
	
	/**
	 * 视频上传并转成m3u8
	 * @param multipartFile
	 * @return
	 * @throws IOException 
	 * @throws InterruptedException 
	 * @throws Exception 
	 */
	@ResponseBody
	@RequestMapping(value = "/m3u8", method = RequestMethod.POST, headers = "content-type=multipart/form-data")
	public JSONObject m3u8( @RequestParam("video") CommonsMultipartFile multipartFile, 
			@RequestParam(value = "privately", defaultValue="false") Boolean privately,
			@RequestParam(value = "forceResize", defaultValue="false") Boolean forceResize) throws IOException, InterruptedException{
		String oFileName = multipartFile.getOriginalFilename();
		String ext = ".tmp";
		if(oFileName.contains(".")){
			ext = oFileName.substring(oFileName.lastIndexOf(".")).toLowerCase();
		}
		String baseFileName = FileUtils.createFileName();
		String fileName =  "video/mp4/" + baseFileName + ".mp4";
		JSONObject result = new JSONObject();
		String url = "";
		
		// 不是Mp4类型或者强制压缩
		if(!".mp4".equalsIgnoreCase(ext)){
			String srcFilePath = ResConstant.TEMP_PATH + baseFileName + ext;
			String destFilePath = ResConstant.TEMP_PATH + baseFileName + ".mp4";
			File srcFile = new File(srcFilePath);
			if(!srcFile.exists()){
				srcFile.createNewFile();
			}
			multipartFile.transferTo(srcFile);
			Video video = FFMpegUtils.parseVideo(srcFilePath);
			FFMpegUtils.videoTransfer(srcFilePath, destFilePath, video.getDestResolution(), video.getTranspose());
			
			File destFile = new File(destFilePath);
			url = QiNiuUtils.uploadVideo(destFilePath, fileName);
			srcFile.delete();
			destFile.delete();
		}
		else if(forceResize){
			String srcFilePath = ResConstant.TEMP_PATH + baseFileName + ext;
			String destFilePath = ResConstant.TEMP_PATH + baseFileName + "_resize.mp4";
			File srcFile = new File(srcFilePath);
			if(!srcFile.exists()){
				srcFile.createNewFile();
			}
			multipartFile.transferTo(srcFile);
			Video video = FFMpegUtils.parseVideo(srcFilePath);
			if(video.getBitRate() > 900){
				FFMpegUtils.resize(srcFilePath, destFilePath, "800k", video.getDestResolution());
				File destFile = new File(destFilePath);
				url = QiNiuUtils.uploadVideo(destFilePath, fileName);
				srcFile.delete();
				destFile.delete();
			}
			else{
				url = QiNiuUtils.uploadVideo(multipartFile.getBytes(), fileName);
			}
		}
		else{
			url = QiNiuUtils.uploadVideo(multipartFile.getBytes(), fileName);
		}
		
		String persistentId = QiNiuUtils.m3u8(fileName);
		result.put("persistentId", persistentId);
		result.put("url", url );
		result.put("preview", privately ? QiNiuUtils.getPrivateDownloadUrl(url) : url);
		result.put("m3u8", QiNiuUtils.getM3u8FullPath(fileName));
		return  result;
	}
	
	
	/**
	 * 获取视频封面
	 * @param url
	 * @param width
	 * @param height
	 * @return
	 */
	@ResponseBody
	@RequestMapping(value = "/cover", method = RequestMethod.GET)
	public JSONObject getCover(@RequestParam("url") String url, 
			@RequestParam("width") Integer width, @RequestParam("height") Integer height){
		JSONObject response = new JSONObject();
		response.put("cover", QiNiuUtils.uploadVideoCover(url, width, height));
		return response;
	}
	
	/**
	 * 查询视频状态
	 * @return
	 * @throws QiniuException 
	 */
	@ResponseBody
	@RequestMapping(value = "/video/query/status/{persistentId:.+}", method = RequestMethod.GET)
	public JSONObject queryVideoStatus(@PathVariable("persistentId") String persistentId) throws QiniuException{
		Client client = new Client();
		return JSONObject.fromObject(client.get("http://api.qiniu.com/status/get/prefop?id=" + persistentId).bodyString());
	}
	
	/**
	 * 获取私有下载地址
	 * @return
	 * @throws QiniuException 
	 */
	@ResponseBody
	@RequestMapping(value = "/download", method = RequestMethod.GET)
	public JSONObject download(@RequestParam("url") String baseUrl, @RequestParam("expires") Long expires) throws QiniuException{
		JSONObject json = new JSONObject();
		json.put("url", QiNiuUtils.getPrivateDownloadUrl(baseUrl, expires));
		return json;
	}
	
	/**
	 * 文件上传
	 * 
	 * @param multipartFile
	 * @return
	 * @throws IOException
	 * @throws InterruptedException
	 * @throws Exception
	 */
	@ResponseBody
	@RequestMapping(value = "/local", method = RequestMethod.POST, headers = "content-type=multipart/form-data")
	public JSONObject uploadLocal(@RequestParam("file") CommonsMultipartFile multipartFile,
			@RequestParam(value = "privately", defaultValue = "false") Boolean privately,
			@RequestParam(value = "fileType", defaultValue = "file") FileType fileType)
			throws IOException, InterruptedException {
		return doLocalUpload(multipartFile, fileType);
	}
	
	/**
	 * 证件上传
	 * 
	 * @param multipartFile
	 * @return
	 * @throws IOException
	 * @throws InterruptedException
	 * @throws Exception
	 */
	@ResponseBody
	@RequestMapping(value = "/certification", method = RequestMethod.POST, headers = "content-type=multipart/form-data")
	public JSONObject uploadCertification(@RequestParam("image") CommonsMultipartFile multipartFile)
			throws IOException, InterruptedException {
		return doLocalUpload(multipartFile, FileType.image, false, true, ResConstant.WATERMARK_CERTIFICATION);
	}
	
	/**
	 * 文件上传
	 * 
	 * @param multipartFile
	 * @return
	 * @throws IOException
	 * @throws InterruptedException
	 * @throws Exception
	 */
	@ResponseBody
	@RequestMapping(value = "/temp", method = RequestMethod.POST, headers = "content-type=multipart/form-data")
	public JSONObject uploadTemp(@RequestParam("file") CommonsMultipartFile multipartFile,
			@RequestParam(value = "privately", defaultValue = "false") Boolean privately,
			@RequestParam(value = "fileType", defaultValue = "file") FileType fileType)
			throws IOException, InterruptedException {
		return doTempUpload(multipartFile, fileType);
	}
	
	/**
	 * 多文件上传
	 * 
	 * @param multipartFiles1
	 * @param multipartFiles2
	 * @return
	 * @throws IOException
	 */
	@ResponseBody
	@RequestMapping(value = "/locals", method = RequestMethod.POST, headers = "content-type=multipart/form-data")
	public JSONArray multiUploadLocal(@RequestParam(value = "files", required = false) List<CommonsMultipartFile> multipartFiles1,
			@RequestParam(value = "files[]", required = false) List<CommonsMultipartFile> multipartFiles2,
			@RequestParam(value = "fileType", defaultValue = "file") FileType fileType) throws IOException {
		List<CommonsMultipartFile> multipartFiles = multipartFiles1 == null ? multipartFiles2 : multipartFiles1;
		JSONArray items = new JSONArray();
		for (CommonsMultipartFile multipartFile : multipartFiles) {
			items.add(doLocalUpload(multipartFile, fileType));
		}
		return items;
	}
	
	/**
	 * 多文件上传临时目录
	 * 
	 * @param multipartFiles1
	 * @param multipartFiles2
	 * @return
	 * @throws IOException
	 */
	@ResponseBody
	@RequestMapping(value =  "/temps", method = RequestMethod.POST, headers = "content-type=multipart/form-data")
	public JSONArray multiUploadTemp(@RequestParam(value = "files", required = false) List<CommonsMultipartFile> multipartFiles1,
			@RequestParam(value = "files[]", required = false) List<CommonsMultipartFile> multipartFiles2,
			@RequestParam(value = "fileType", defaultValue = "file") FileType fileType) throws IOException {
		List<CommonsMultipartFile> multipartFiles = multipartFiles1 == null ? multipartFiles2 : multipartFiles1;
		JSONArray items = new JSONArray();
		for (CommonsMultipartFile multipartFile : multipartFiles) {
			items.add(doTempUpload(multipartFile, fileType));
		}
		return items;
	}
	

	
	/**
	 * 图片上传操作
	 * @param multipartFile
	 * @return
	 * @throws Exception 
	 */
	protected JSONObject doImageUpload(CommonsMultipartFile multipartFile, Boolean showInfo, Integer width, Integer height) throws Exception{
		if(width == null){
			width = ResConstant.MAX_WIDTH;
		}
		if(height == null){
			height = 0;
		}
		Image image = new Image(multipartFile);
		byte[] bytes;
		if(width > 0){
			ByteArrayOutputStream outputStream = image.resize(width, 0, ResConstant.LIMIT_SIZE);
			image.removeTemp();
			bytes = outputStream.toByteArray();
		}
		else{
			bytes = image.getBytes();
		}

		String dateDir = getDateDir();
		String dir = dateDir + File.separator + FileType.image;
		String fileName = dir + File.separator + image.getFullName();
		JSONObject result = new JSONObject();
		if(ResConstant.IS_QINIU_MODE) {
			QiniuClient client = getQiniuClient();
			String url = "";
			if(client!= null){
				url = client.upload(bytes, fileName);
			}
			else{
				url = QiNiuUtils.upload(bytes, fileName);
			}
			
			result.put("url", url );
			if(showInfo!= null && showInfo){
				JSONObject info = QiNiuUtils.imageInfo(url);
				result.put("size", info.get("size"));
				result.put("width", info.get("width"));
				result.put("height", info.get("height"));
			}
		}
		else {
			String filePath = ResConstant.FILE_SERVER_PATH + File.separator + fileName;
			String url = ResConstant.FILE_SERVER_DOMAIN + "/" + fileName.replaceAll(Matcher.quoteReplacement(File.separator), "/");
			File distFile = new File(filePath);
			FileUtils.bytes2File(bytes, distFile);
			
			result.put("url", url );
			if(showInfo!= null && showInfo){
				BufferedImage bufferedImage = ImageIO.read(distFile);
				result.put("size", distFile.length());
				result.put("width", bufferedImage.getWidth());
				result.put("height", bufferedImage.getHeight());
			}
		}
		
		
		return  result;
	}
	
	/**
	 * 上传到本地
	 * @param multipartFile
	 * @return
	 */
	protected JSONObject doLocalUpload(CommonsMultipartFile multipartFile){
		return doLocalUpload(multipartFile, null, false, false);
	}
	
	/**
	 * 上传到本地
	 * @param multipartFile
	 * @param fileType
	 * @return
	 */
	protected JSONObject doLocalUpload(CommonsMultipartFile multipartFile, FileType fileType){
		boolean extra = false;
		if(FileType.audio.equals(fileType) || FileType.video.equals(fileType)){
			extra = true;
		}
		return doLocalUpload(multipartFile, fileType, extra, false);
	}
	
	/**
	 * 上传到临时目录
	 * @param multipartFile
	 * @param fileType
	 * @return
	 */
	protected JSONObject doTempUpload(CommonsMultipartFile multipartFile, FileType fileType){
		boolean extra = false;
		if(FileType.audio.equals(fileType) || FileType.video.equals(fileType)){
			extra = true;
		}
		return doLocalUpload(multipartFile, fileType, extra, true);
	}
	
	/**
	 * 本地上传
	 * @param multipartFile
	 * @param fileType
	 * @param extra
	 * @param temp
	 * @return
	 */
	protected JSONObject doLocalUpload(CommonsMultipartFile multipartFile, FileType fileType, boolean extra, boolean temp){
		return doLocalUpload(multipartFile, fileType, extra, temp, null);
	}
	
	/**
	 * 上传本地
	 * 
	 * @param multipartFile
	 * @param fileType 文件类型
	 * @param extra 是否返回扩展信息
	 * @param temp 是否临时存储
	 * @param watermark 图片水印
	 * @return
	 * @throws IOException
	 */
	protected JSONObject doLocalUpload(CommonsMultipartFile multipartFile, FileType fileType, boolean extra, boolean temp, String watermark) {
		String oFileName = multipartFile.getOriginalFilename();
		String ext = getExt(oFileName, fileType);
		String simpleFileName = FileUtils.createFileName();
		String fileName = simpleFileName + ext;
		String dateDir = getDateDir();
		String subfix = dateDir + File.separator + fileType;
		if(temp){
			subfix = ResConstant.FILE_SERVER_TEMP_DIR + File.separator + subfix;
		}
		JSONObject result = new JSONObject();
		String url = "";
		String original = ""; // 原始保存图片
		String thumb = "";	// 缩略图
		if (!multipartFile.isEmpty()) {
			String storePath = ResConstant.FILE_SERVER_PATH + File.separator + subfix;// 存放我们上传的文件路径
			File filepath = new File(storePath, fileName);
			if (!filepath.getParentFile().exists()) {
				filepath.getParentFile().mkdirs();// 如果目录不存在，创建目录
			}
			try {
				File distFile = new File(storePath + File.separator + fileName);
				multipartFile.transferTo(distFile);// 把文件写入目标文件地址

				if(fileType.equals(FileType.image)){
					String thumbName = simpleFileName + "_thumb" + ext;
					File thumbFile = new File(storePath + File.separator + thumbName);
					ImageUtils.crop(new FileInputStream(distFile), new FileOutputStream(thumbFile), ResConstant.THUMB_MAX_WIDTH, ResConstant.THUMB_MAX_HEIGHT, 1F);
					thumb = ResConstant.FILE_SERVER_DOMAIN + "/" + subfix.replaceAll(Matcher.quoteReplacement(File.separator), "/") + "/"  + thumbName;

					// 打水印
					if(watermark != null && !"".equals(watermark)){
						if(ResConstant.WATERMARK_CERTIFICATION.equals(watermark)){
							String originalFileName = simpleFileName + "_original" + ext;
							File oFile = new File(storePath + File.separator + originalFileName);
							FileUtils.copy(distFile, oFile);
							FileUtils.watermark(distFile, new File(watermark));
							original = ResConstant.FILE_SERVER_DOMAIN + "/" + subfix.replaceAll(Matcher.quoteReplacement(File.separator), "/") + "/"  + originalFileName;

						}
					}
				}


				
				url = ResConstant.FILE_SERVER_DOMAIN + "/" + subfix.replaceAll(Matcher.quoteReplacement(File.separator), "/") + "/"  + fileName;

				if(extra){
					if (FileType.audio.equals(fileType)) {
						Audio audio = FFMpegUtils.parseAudio(storePath + File.separator + fileName);
						result.putAll(JSONObject.fromObject(audio));
					} else if (FileType.video.equals(fileType)) {
						String fullPath = storePath + File.separator + fileName;
						Video video = FFMpegUtils.parseVideo(fullPath);
						FFMpegUtils.makeScreenCut(fullPath, fullPath + ".jpg", video.getResolution(), null);
						result.putAll(JSONObject.fromObject(video));
						result.put("cover", url +".jpg");
						result.remove("srcFile");
						result.remove("destFile");
					}
				}
				
				
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		if("".equals(original)){
			original = url;
		}
		if(!"".equals(thumb)){
			result.put("thumb", thumb);
		}
		result.put("url", url);
		result.put("preview", url);
		result.put("original", original);
		return result;
	}
	
	
	/**
	 * 普通文件上传
	 * @param multipartFile
	 * @return
	 * @throws IOException
	 */
	protected JSONObject doFileUpload(CommonsMultipartFile multipartFile, String ext) throws IOException{
		if(ext == null){
			ext = ".tmp";
		}
		String oFileName = multipartFile.getOriginalFilename();
		if(oFileName.contains(".")){
			ext = oFileName.substring(oFileName.lastIndexOf(".")).toLowerCase();
		}
		String fileName =  UUID.randomUUID() + ext;
		byte[] bytes = multipartFile.getBytes();
		QiniuClient client = getQiniuClient();
		String url = "";
		if(client!= null){
			url = client.upload(bytes, fileName);
		}
		else{
			url = QiNiuUtils.upload(bytes, fileName);
		}
		JSONObject result = new JSONObject();
		result.put("url", url );
		return  result;
	}
	
	/**
	 * 普通文件上传
	 * @param multipartFile
	 * @return
	 * @throws IOException
	 */
	protected JSONObject doFileUpload(CommonsMultipartFile multipartFile) throws IOException{
		return doFileUpload(multipartFile, ".tmp");
	}
	
	
	/**
	 * 获取文件后缀
	 * @param fileName
	 * @param fileType
	 * @return
	 */
	protected String getExt(String fileName, FileType fileType){
		if(fileName.contains(".")){
			return fileName.substring(fileName.lastIndexOf(".")).toLowerCase();
		}
		
		if(FileType.image.equals(fileType)){
			return ".jpg";
		}
		else if(FileType.audio.equals(fileType)){
			return ".mp3";
		}
		else if(FileType.video.equals(fileType)){
			return ".mp4";
		}
		return ".tmp";
	}
	
	/**
	 * 获取当前日期目录
	 * @return
	 */
	protected String getDateDir(){
		SimpleDateFormat df = new SimpleDateFormat("yyyyMMdd");
		String dateDir = df.format(new Date());
		return dateDir;
	}

	/**
	 * 获取七牛客户端
	 * @return
	 */
	protected QiniuClient getQiniuClient(){
		return null;
		
	}
	
}
