package com.crawler.res.utils;

import com.crawler.ffmpeg.bean.Video;
import com.crawler.ffmpeg.utils.FFMpegUtils;
import com.crawler.res.bean.FileBaseInfo;
import com.crawler.res.common.FileType;
import com.crawler.res.common.ResConstant;
import com.crawler.res.exception.ResException;
import eu.medsea.mimeutil.MimeUtil;
import net.sf.json.JSONObject;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.binary.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.multipart.commons.CommonsMultipartFile;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.*;
import java.net.URL;
import java.net.URLConnection;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 文件工具类
 * @author rubekid
 *
 * 2018年2月22日 上午10:40:29
 */
public class FileUtils {
	
	private static Logger logger = LoggerFactory.getLogger(FileUtils.class);
	
	static{
		MimeUtil.registerMimeDetector("eu.medsea.mimeutil.detector.MagicMimeMimeDetector");
	}

	/**
	 * 获取名称
	 * @return
	 */
	public static String createFileName(){
		String md5 = md5(UUID.randomUUID().toString(), System.currentTimeMillis());
		return md5.substring(8, 24);
	}
	
	/**
	 * 16Md5
	 * @param message
	 * @param salt
	 * @return
	 */
	public static String shortMd5(String message, String salt){
		String md5 = md5(message, salt);
		return md5.substring(8, 24);
	}
	
	/**
	 * 生成MD5
	 * @param message
	 * @return
	 */
	public static String md5(String message, Object salt){
		try{
			MessageDigest messageDigest = MessageDigest.getInstance("MD5");
			if (salt != null && !"".equals(salt)) {
				message =  message + "{" + salt.toString() + "}";
			}
			byte[] digest = messageDigest.digest(message.getBytes("utf-8"));
			return Hex.encodeHexString(digest);
		}
		catch(NoSuchAlgorithmException | UnsupportedEncodingException ex){}
		
		return String.valueOf(System.currentTimeMillis());
	}
	
	
	/**
	 * 同步到七牛云
	 * @param url
	 * @return
	 */
	public static String syncToQiniuAndDelete(String url){
		return syncToQiniu(url, null, false, true);
	}

	/**
	 * 同步到七牛云
	 * @param url
	 * @return
	 */
	public static String syncToQiniu(String url){
		return syncToQiniu(url, null, false, false);
	}
	
	/**
	 * 同步到七牛云
	 * @param url
	 * @param compressed
	 * @param remove
	 * @return
	 */
	public static String syncToQiniu(String url, boolean compressed, boolean remove){
		return syncToQiniu(url, null, compressed, remove);
	}
	
	/**
	 * 临时文件保存成正式
	 * @param tempUrl
	 * @return
	 */
	public static String save(String tempUrl){
		String url = tempUrl;
		if(isTemp(tempUrl)){
			String srcFilePath = getLocalPath(tempUrl);
			String destFilePath = tempUrl.replace(ResConstant.FILE_SERVER_DOMAIN + "/" + ResConstant.FILE_SERVER_TEMP_DIR, ResConstant.FILE_SERVER_PATH).replaceAll("/", Matcher.quoteReplacement(File.separator));
			FileUtils.copy(srcFilePath, destFilePath);
			url = tempUrl.replace(ResConstant.FILE_SERVER_DOMAIN + "/" + ResConstant.FILE_SERVER_TEMP_DIR, ResConstant.FILE_SERVER_DOMAIN);
		}
		return url;
	}
	
	/**
	 * 替换临时文件为正式文件
	 * @param content
	 * @return
	 */
	public static String replace(String content) {
		Pattern pattern = Pattern.compile(ResConstant.FILE_SERVER_DOMAIN + "/" + ResConstant.FILE_SERVER_TEMP_DIR + "/[^\",;]+");
		Matcher matcher = pattern.matcher(content);
		while(matcher.find()) {
			String tempUrl = matcher.group();
			String url = save(tempUrl);
			content = content.replace(tempUrl, url);
		}
		
		return content;
	}

	
	/**
	 * 同步到七牛云
	 * @param url
	 * @param fileType
	 * @param compressed
	 * @param remove 删除本地文件
	 * @return
	 * @throws InterruptedException 
	 * @throws IOException 
	 */
	public static String syncToQiniu(String url, FileType fileType, boolean compressed, boolean remove){
		try{
			String fileName =  url.substring(url.lastIndexOf("/")+1);
			if(!url.startsWith(ResConstant.FILE_SERVER_DOMAIN)){
				return url;
			}
			String ext = "";
			if(fileName.contains(".")){
				ext =  fileName.substring(fileName.lastIndexOf("."));
			}
			String srcFilePath = getLocalPath(url);
			File srcFile = new File(srcFilePath);
			if(!srcFile.exists()){ // 不存在返回源地址
				return url;
			}
			if(fileType == null){
				fileType = getFileType(srcFile);
			}
			String qiniuFileName = url.replace(ResConstant.FILE_SERVER_DOMAIN + "/", "");
			if (FileType.video.equals(fileType)) {
				
				String baseFileName = fileName.substring(0, fileName.lastIndexOf("."));
				// 不是Mp4类型或者强制压缩
				if(!".mp4".equals(ext)){
					String destFilePath = ResConstant.TEMP_PATH + baseFileName + ".mp4";
					Video video = FFMpegUtils.parseVideo(srcFilePath);
					FFMpegUtils.videoTransfer(srcFilePath, destFilePath, video.getDestResolution(), video.getTranspose());
					
					File destFile = new File(destFilePath);
					url = QiNiuUtils.uploadVideo(destFilePath, qiniuFileName);
					destFile.delete();
				}
				else if(compressed){
					String destFilePath = ResConstant.TEMP_PATH + baseFileName + "_min.mp4";
					Video video = FFMpegUtils.parseVideo(srcFilePath);
					if(video.getBitRate() > 900){
						FFMpegUtils.resize(srcFilePath, destFilePath, "800k", video.getDestResolution());
						File destFile = new File(destFilePath);
						url = QiNiuUtils.uploadVideo(destFilePath, qiniuFileName);
						destFile.delete();
					}
					else{
						url = QiNiuUtils.uploadVideo(srcFilePath, qiniuFileName);
					}
				}
				else{
					url = QiNiuUtils.uploadVideo(srcFilePath, qiniuFileName);
				}
			} else {
				url = QiNiuUtils.upload(srcFilePath, qiniuFileName);
			}
			
			// 是否移除本地源文件
			if(remove){				
				srcFile.delete();
			}
		}
		catch(IOException | InterruptedException ex){
			logger.error(ex.getMessage(), ex);
		}
		return url;
	}
	
	/**
	 * 获取图片信息
	 * @param url
	 * @return
	 */
	public static JSONObject imageInfo(String url) {
		if(isLoacl(url)){
			JSONObject info = new JSONObject();
			try{
				String srcFilePath = getLocalPath(url);
				File file = new File(srcFilePath);
				if(file.exists()){
					BufferedImage image = ImageIO.read(new FileInputStream(file));
					info.put("width", image.getWidth());
					info.put("height", image.getHeight());
					info.put("size", file.length());
				}
				
			}
			catch(IOException ex){
				logger.error(ex.getMessage(), ex);
			}
			return info;
			
		}
		return QiNiuUtils.imageInfo(url);
	}
	
	public static JSONObject avinfo(String url) {
		if(isLoacl(url)){
			JSONObject info = new JSONObject();
			try{
				String srcFilePath = getLocalPath(url);
				File file = new File(srcFilePath);
				if(file.exists()){
					Video video = FFMpegUtils.parseVideo(srcFilePath);
					info.putAll(JSONObject.fromObject(video));
					info.remove("srcFile");
					info.remove("destFile");
					info.put("size", file.length());
				}
			}
			catch(IOException | InterruptedException ex){
				logger.error(ex.getMessage(), ex);
			}
			
			return info;
		}
		return QiNiuUtils.avinfo(url);
	}
	
	/**
	 * 获取视频封面
	 * @param url
	 * @return
	 */
	public static String getVideoCover(String url, int width, int height){
		if(isLoacl(url)){
			
			try{
				String srcFilePath = getLocalPath(url);
				File file = new File(srcFilePath);
				if(file.exists()){
					Video video = FFMpegUtils.parseVideo(srcFilePath);
					String extName = createFileName().substring(0, 3) + ".jpg";
					FFMpegUtils.makeScreenCut(srcFilePath, srcFilePath.substring(0, srcFilePath.length() - 7) + extName, video.getResolution(), null);
					return url.substring(0, url.length() - 7) + extName;
				}
			}
			catch(IOException | InterruptedException ex){
				logger.error(ex.getMessage(), ex);
			}
			
			return "";
		}
		return QiNiuUtils.getVideoCover(url, width, height);
	}
	
	/**
	 * 是否本地
	 * @param url
	 * @return
	 */
	public static boolean isLoacl(String url){
		return url.startsWith(ResConstant.FILE_SERVER_DOMAIN);
	}
	
	/**
	 * 是否临时文件
	 * @param url
	 * @return
	 */
	public static boolean isTemp(String url){
		return url.startsWith(ResConstant.FILE_SERVER_DOMAIN + "/" + ResConstant.FILE_SERVER_TEMP_DIR + "/");
	}
	
	/**
	 * 获取本地路径
	 * @param url
	 * @return
	 */
	public static String getLocalPath(String url){
		return url.replace(ResConstant.FILE_SERVER_DOMAIN, ResConstant.FILE_SERVER_PATH).replaceAll("/", Matcher.quoteReplacement(File.separator));
	}
	
	/**
	 * 获取图片本地路径
	 * @param url
	 * @param forceDownload
	 * @return
	 */
	public static String getLocalPath(String url, boolean forceDownload){
		if(!isLoacl(url)){
			try {
				url = syncToLocal(url);
			} catch (IOException ex) {
				logger.error(ex.getMessage(), ex);
				throw new ResException("网络文件下载失败");
			}
		}
		return getLocalPath(url);
	}
	
	/**
	 * 获取本地文件对象
	 * @param url
	 * @return
	 */
	public static File getLocalFile(String url){
		String path = getLocalPath(url);
		File file = new File(path);
		if(file.exists()){
			return file;
		}
		return null;
	}
	
	/**
	 * 获取MIME TYPE
	 * @param file
	 * @return
	 * @Exception
	 */
	public static String getMimeType(File file){
		Collection<?> mimeTypes = MimeUtil.getMimeTypes(file);
		return mimeTypes.toString();
	}
	
	/**
	 * 获取MIME TYPE
	 * @param file
	 * @return
	 * @Exception
	 */
	public static String getMimeType(CommonsMultipartFile file){
		Collection<?> mimeTypes = MimeUtil.getMimeTypes(file.getBytes());
		return mimeTypes.toString();
	}
	
	/**
	 * 获取文件类型
	 * @param file
	 * @return
	 */
	public static FileType getFileType(File file){
		String mimeType = getMimeType(file);
		if(mimeType != null){
			if(mimeType.startsWith("video")){
				return FileType.video;
			}
			else if(mimeType.startsWith("audio")){
				return FileType.audio;
			}
			else if(mimeType.startsWith("image")){
				return FileType.image;
			}
			// TODO
		}
		
		return FileType.file;
		
	}
	
	/**
	 * 复制文件
	 * @param srcFile
	 * @param destFile
	 * @throws IOException
	 */
	public static void copy(File srcFile, File destFile) {
		try{
			org.apache.commons.io.FileUtils.copyFile(srcFile, destFile);					
		}
		catch(IOException ex){
			logger.error(ex.getMessage(), ex);
			throw new ResException("文件复制失败");
		}
	}
	
	/**
	 * 移动文件
	 * @param srcFilePath
	 * @param destFilePath
	 */
	public static void copy(String srcFilePath, String destFilePath){
		File srcFile = new File(srcFilePath);
		File destFile = new File(destFilePath);
		if(!srcFile.exists()){
			throw new ResException("源文件不存在");
		}
		try{
			org.apache.commons.io.FileUtils.copyFile(srcFile, destFile);					
		}
		catch(IOException ex){
			logger.error(ex.getMessage(), ex);
			throw new ResException("文件复制失败");
		}
	}
	
	/**
	 * 移动文件
	 * @param srcFilePath
	 * @param destFilePath
	 */
	public static void move(String srcFilePath, String destFilePath){
		File srcFile = new File(srcFilePath);
		File destFile = new File(destFilePath);
		if(!srcFile.exists()){
			throw new ResException("源文件不存在");
		}
		try{
			org.apache.commons.io.FileUtils.moveFile(srcFile, destFile);					
		}
		catch(IOException ex){
			logger.error(ex.getMessage(), ex);
			throw new ResException("文件移动失败");
		}
	}
	
	/**
	 * 网络资源转byte[]
	 * @param spec 网络地址
	 * @return
	 * @throws IOException
	 */
	public static byte[] urlToByte(String spec){
		try{
			URL url = new URL(spec);  
	        URLConnection con = url.openConnection();
	        con.setConnectTimeout(10*1000);  
	        InputStream is = con.getInputStream();
	        ByteArrayOutputStream output = new ByteArrayOutputStream();
	        byte[] buffer = new byte[1024];
	        int n = 0;
	        while (-1 != (n = is.read(buffer))) {
	            output.write(buffer, 0, n);
	        }
	        is.close();  
	        return output.toByteArray();
		}
        catch (IOException ex) {
			logger.error(ex.getMessage(), ex);
		}
		return null;
	}
	
	/**
	 * 同步到本地
	 * @param spec
	 * @param dir
	 * @return
	 * @throws IOException
	 */
	public static String syncToLocal(String spec, String dir, String ext) throws IOException{
		String fileName =  spec.substring(spec.lastIndexOf("/")+1);
		if(spec.startsWith(ResConstant.FILE_SERVER_DOMAIN)){
			return spec;
		}
		if(ext == null) {
			ext = "";
		}
		
		if("".equals(ext) && fileName.contains(".")){
			ext =  fileName.substring(fileName.lastIndexOf("."));
		}
		fileName = FileUtils.createFileName() + ext;
		String dateDir = getDateDir();
		String subfix = dateDir + File.separator + dir;
		String storePath = ResConstant.FILE_SERVER_PATH + File.separator + subfix;// 存放我们上传的文件路径
		File filepath = new File(storePath, fileName);
		if (!filepath.getParentFile().exists()) {
			filepath.getParentFile().mkdirs();// 如果目录不存在，创建目录
		}
		
		URL url = new URL(spec);
        URLConnection con = url.openConnection();
        FileOutputStream out = new FileOutputStream(filepath);
        InputStream ins = con.getInputStream();
        byte[] b = new byte[1024];
        int i=0;
        while((i=ins.read(b))!=-1){
            out.write(b, 0, i);
        }
        ins.close();
        out.close();
        
        return ResConstant.FILE_SERVER_DOMAIN + "/" + subfix.replaceAll(Matcher.quoteReplacement(File.separator), "/") + "/"  + fileName;
	
	}
	
	/**
	 * 同步到本地
	 * @param spec
	 * @return
	 * @throws IOException
	 */
	public static String syncToLocal(String spec) throws IOException{
		return syncToLocal(spec, "download");
	}
	
	/**
	 * 同步到本地
	 * @param spec
	 * @param dir
	 * @return
	 * @throws IOException
	 */
	public static String syncToLocal(String spec, String dir) throws IOException{
		return syncToLocal(spec, "download", null);
	}

	/**
	 * 保存到本地
	 * @param spec
	 * @param filePath
	 * @return
	 */
	public static String saveToLoacl(String spec, String filePath, String fileName) {
		try{
			if(spec.startsWith(ResConstant.FILE_SERVER_DOMAIN)){
				return spec;
			}

			if(fileName == null || "".equals(fileName)){
				fileName =  spec.substring(spec.lastIndexOf("/")+1);
			}

			String storePath = ResConstant.FILE_SERVER_PATH + File.separator + filePath;  // 存放我们上传的文件路径
			File filepath = new File(storePath, fileName);
			if (!filepath.getParentFile().exists()) {
				filepath.getParentFile().mkdirs();// 如果目录不存在，创建目录
			}

			URL url = new URL(spec);
			URLConnection con = url.openConnection();
			FileOutputStream out = new FileOutputStream(filepath);
			InputStream ins = con.getInputStream();
			byte[] b = new byte[1024];
			int i=0;
			while((i=ins.read(b))!=-1){
				out.write(b, 0, i);
			}
			ins.close();
			out.close();
			return ResConstant.FILE_SERVER_DOMAIN + "/" + filePath.replaceAll(Matcher.quoteReplacement(File.separator), "/") + "/"  + fileName;

		}
		catch(IOException ex){
			logger.error(ex.getMessage(), ex);
		}
		return spec;
	}

	/**
	 * 保存到本地
	 * @param spec
	 * @param filePath
	 * @return
	 */
	@Deprecated
	public static String transfer(String spec, String filePath) {
		return saveToLoacl(spec, filePath, null);
	}
	
	/**
	 * 获取当前日期目录
	 * @return
	 */
	public static String getDateDir(){
		SimpleDateFormat df = new SimpleDateFormat("yyyyMMdd");
		String dateDir = df.format(new Date());
		return dateDir;
	}
	
	/**
	 * 图片转base64
	 * @param path
	 * @return
	 */
	public static String toBase64(String path){
		InputStream inputStream = null;
		byte[] data = null;
		try {
			inputStream = new FileInputStream(path);
		    data = new byte[inputStream.available()];
		    inputStream.read(data);
		    inputStream.close();
		} catch (IOException ex) {
			logger.error(ex.getMessage(), ex);
		}
		
		return Base64.encodeBase64String(data);
	}
	
	
	/**
	 * 水印
	 * @param imageFile
	 * @param watermak
	 * @throws IOException
	 */
    public static void watermark(File imageFile, File watermak) throws IOException{
       
        if(watermak.exists()){
        	
        	BufferedImage bufferedImage = ImageIO.read(imageFile);
            Graphics2D gs = bufferedImage.createGraphics();
            int width = bufferedImage.getWidth();
            int height = bufferedImage.getHeight();
            BufferedImage watermakImage = ImageIO.read(watermak);
            float rate = ResConstant.WATERMARK_RATE; //显示为图片的 scale;
            
            //水印显示大小
            int lw = (int) (width * rate);
            int lh = (int) (watermakImage.getHeight() * lw / watermakImage.getWidth() );
            int wx = (width - lw) /2;
            int wy = (height -lh) / 2;
            gs.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_ATOP, 1));
            gs.drawImage(watermakImage, wx, wy, lw, lh, null); // 图片水印文件结束
            gs.dispose();
            
            FileOutputStream os = new FileOutputStream(imageFile);
            ImageIO.write(bufferedImage, "JPEG", os);
        }
        
    }
    
    /**
     * byte[] 转文件
     * @param bytes
     * @param distFile
     * @throws IOException 
     */
	public static void bytes2File(byte[] bytes, File distFile) throws IOException {
		BufferedOutputStream bos = null;
        FileOutputStream fos = null;
        try {
        	if(!distFile.getParentFile().exists()) {
        		distFile.getParentFile().mkdirs();// 如果目录不存在，创建目录
        	}
        	fos = new FileOutputStream(distFile);
			bos = new BufferedOutputStream(fos);
            bos.write(bytes);
        	
        } finally {
            if (bos != null) {
                try {
                    bos.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            if (fos != null) {
                try {
                    fos.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
	}
	
	/**
	 * 获取文件信息
	 * @param fileName
	 * @return
	 */
	public static FileBaseInfo getFileInfo(String fileName) {
		String  separator =  java.io.File.separator;
		final String url = ResConstant.FILE_SERVER_DOMAIN + "/" + fileName;
		if("\\".equals(separator)) {
			fileName = fileName.replaceAll("/", "\\\\");
		}
		String filePath = ResConstant.FILE_SERVER_PATH + separator + fileName; 
		String fileDirPath = filePath.substring(0, filePath.lastIndexOf(separator));
		java.io.File fileDir = new java.io.File(fileDirPath);
		if(!fileDir.exists()) {
			fileDir.mkdirs();
		}
		FileBaseInfo info = new FileBaseInfo();
		info.setPath(filePath);
		info.setUrl(url);
		return info;
	}
	
	/**
	 * 获取目录绝对地址
	 * @param dirName
	 * @return
	 */
	public static String getAbsolutePath(String dirName) {
		return ResConstant.FILE_SERVER_PATH + File.separator + dirName;
	}
	
	/**
	 * 删除目录
	 * @param dir
	 */
	public static void deleteDirectory(File dir) {
		if(dir == null) {
			return ;
		}
		if(!dir.exists()) {
			return ;
		}
		if(dir.isDirectory()) {
			File[] files = dir.listFiles();
			for(File file : files) {
				deleteDirectory(file);
			}
			dir.delete();
		}
		else if(dir.isFile()) {
			dir.delete();
		}
	}
}
