package com.crawler.res.image;

import java.awt.AlphaComposite;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;

import javax.imageio.ImageIO;

import org.summercool.image.AnimatedGifEncoder;
import org.summercool.image.GifDecoder;
import org.summercool.image.Scalr;
import org.summercool.image.Scalr.Method;
import org.summercool.image.Scalr.Mode;

import com.crawler.res.utils.FileUtils;
import com.sun.image.codec.jpeg.ImageFormatException;
import com.sun.image.codec.jpeg.JPEGCodec;
import com.sun.image.codec.jpeg.JPEGEncodeParam;
import com.sun.image.codec.jpeg.JPEGImageEncoder;

import net.sf.jmimemagic.MagicException;
import net.sf.jmimemagic.MagicMatchNotFoundException;
import net.sf.jmimemagic.MagicParseException; 

/**
 * 图片压缩处理
 * @author Rubekid
 *
 */
@SuppressWarnings("restriction")
public class ImageScale {
	
	public static final String TYPE_JPEG = "jpg";
	
	public static final String TYPE_PNG = "png";
	
	public static final String TYPE_GIF = "gif";
	
	public static final String TYPE_BMP = "bmp";
	
	/**
	 * 图片缩放
	 * @param srcFile
	 * @param destFile
	 * @param maxWidth
	 * @param maxHeight
	 * @param type
	 * @param quality
	 * @throws IOException 
	 */
	public static void resize(File srcFile, File destFile, int maxWidth, int maxHeight, String type, float quality) throws IOException{
		if(TYPE_GIF.equals(type)){
			resizeGif(srcFile, destFile, maxWidth, maxHeight);
		}
		else if(TYPE_PNG.equals(type)){
			resizePng(srcFile, destFile, maxWidth, maxHeight);
		}
		else{
			resizeJpeg(srcFile, destFile, maxWidth, maxHeight, quality);
		}
	}
	
	/**
	 * 图片缩放
	 * @param srcFile
	 * @param destFile
	 * @param maxWidth
	 * @param maxHeight
	 * @param type
	 * @throws IOException 
	 */
	public static void resize(File srcFile, File destFile, int maxWidth, int maxHeight, String type) throws IOException{
		resize(srcFile, destFile, maxWidth, maxHeight, type, 0.8f);
	}
	
	/**
	 * 图片缩放
	 * @param srcFile
	 * @param destFile
	 * @param maxWidth
	 * @param maxHeight
	 * @throws Exception 
	 */
	public static void resize(File srcFile, File destFile, int maxWidth, int maxHeight, float quality) throws Exception{
		resize(srcFile, destFile, maxWidth, maxHeight, getType(srcFile), quality);
	}
	
	/**
	 * 图片缩放
	 * @param srcFile
	 * @param destFile
	 * @param maxWidth
	 * @param maxHeight
	 * @throws Exception 
	 */
	public static void resize(File srcFile, File destFile, int maxWidth, int maxHeight) throws Exception{
		resize(srcFile, destFile, maxWidth, maxHeight, getType(srcFile));
	}
	
	/**
	 * JPEG图片缩放
	 * @param srcFile
	 * @param destFile
	 * @param maxWidth
	 * @param maxHeight
	 * @throws IOException 
	 */
	public static void resizeJpeg(File srcFile, File destFile, int maxWidth, int maxHeight, float quality) throws IOException{
		BufferedImage image = ImageIO.read(srcFile);
	
		//double scale = 0.8f;
		//AffineTransform atf = AffineTransform.getTranslateInstance(scale, scale);
		
		//AffineTransformOp affineTransformOp = new AffineTransformOp(xform, interpolationType)
		
		double rate = getResizeRate(srcFile, maxWidth, maxHeight);
		int width = (int)(image.getWidth() * rate);
		int height =(int) (image.getHeight() * rate);
        image = Scalr.resize(image, Method.AUTOMATIC, Mode.AUTOMATIC, width, height);  
        BufferedImage bufferedImage = new BufferedImage(image.getWidth(), image.getHeight(), BufferedImage.TYPE_INT_RGB);  
        Graphics2D g = bufferedImage.createGraphics();  
        g.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_ATOP, 1));  
        g.fillRect(0, 0, image.getWidth(), image.getHeight());  
        g.drawImage(image, 0, 0, null);  
        image = bufferedImage;  
        
        FileOutputStream out = new FileOutputStream(destFile);
        JPEGImageEncoder encoder = JPEGCodec.createJPEGEncoder(out);  
        JPEGEncodeParam param = encoder.getDefaultJPEGEncodeParam(image);  
        param.setQuality(quality, false);  
        encoder.setJPEGEncodeParam(param);  
        encoder.encode(image);
	}
	
	
	
	/**
	 * PNG图片缩放
	 * @param srcFile
	 * @param destFile
	 * @param maxWidth
	 * @param maxHeight
	 * @throws IOException 
	 */
	public static void resizePng(File srcFile, File destFile, int maxWidth, int maxHeight) throws IOException{
		BufferedImage bufferedImage = ImageIO.read(srcFile);
		double rate = getResizeRate(srcFile, maxWidth, maxHeight);
		if(rate == 1){
			FileUtils.copy(srcFile, destFile);
		}
		else{
			int width = (int)(bufferedImage.getWidth() * rate);
			int height =(int) (bufferedImage.getHeight() * rate);
			BufferedImage result  = Scalr.resize(bufferedImage,Method.AUTOMATIC, Mode.AUTOMATIC, width, height);
			ImageIO.write(result, "png", destFile);
		}
	}
	
	/**
	 * GIF图片缩放
	 * @param srcFile
	 * @param destFile
	 * @param maxWidth
	 * @param maxHeight
	 * @throws IOException
	 */
	public static void resizeGif(File srcFile, File destFile, int maxWidth, int maxHeight) throws IOException{
		GifDecoder gd = new GifDecoder();
		int status = gd.read(new FileInputStream(srcFile));
		if (status != GifDecoder.STATUS_OK) {
			return;
		}
		double rate = getResizeRate(srcFile, maxWidth, maxHeight);
		if(rate == 1){
			FileUtils.copy(srcFile, destFile);
		}
		else{
			AnimatedGifEncoder ge = new AnimatedGifEncoder();
			ge.start(new FileOutputStream(destFile));
			ge.setRepeat(0);
			for (int i = 0; i < gd.getFrameCount(); i++) {
				BufferedImage frame = gd.getFrame(i);
				int width = frame.getWidth();
				int height = frame.getHeight();
				width = (int) (width * rate);
				height = (int) (height * rate);
	
				BufferedImage rescaled = Scalr.resize(frame, Mode.FIT_EXACT, width, height);
				int delay = gd.getDelay(i);
				ge.setDelay(delay);
				ge.addFrame(rescaled);
			}
			ge.finish();
		}
	}
	
	/**
	 * JPEG图片缩放
	 * @param srcFile
	 * @param maxWidth
	 * @param maxHeight
	 * @throws IOException 
	 * @throws ImageFormatException 
	 */
	public static ByteArrayOutputStream resizeJpeg(File srcFile,  int maxWidth, int maxHeight, float quality) throws IOException{
		BufferedImage image = ImageIO.read(srcFile);
	
		//double scale = 0.8f;
		//AffineTransform atf = AffineTransform.getTranslateInstance(scale, scale);
		
		//AffineTransformOp affineTransformOp = new AffineTransformOp(xform, interpolationType)
		
		double rate = getResizeRate(srcFile, maxWidth, maxHeight);
		int width = (int)(image.getWidth() * rate);
		int height =(int) (image.getHeight() * rate);
        image = Scalr.resize(image, Method.AUTOMATIC, Mode.AUTOMATIC, width, height);  
        BufferedImage bufferedImage = new BufferedImage(image.getWidth(), image.getHeight(), BufferedImage.TYPE_INT_RGB);  
        Graphics2D g = bufferedImage.createGraphics();  
        g.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_ATOP, 1));  
        g.fillRect(0, 0, image.getWidth(), image.getHeight());  
        g.drawImage(image, 0, 0, null);  
        image = bufferedImage;  
        
        ByteArrayOutputStream  out = new ByteArrayOutputStream ();
        JPEGImageEncoder encoder = JPEGCodec.createJPEGEncoder(out);  
        JPEGEncodeParam param = encoder.getDefaultJPEGEncodeParam(image);  
        param.setQuality(quality, false);  
        encoder.setJPEGEncodeParam(param);  
        encoder.encode(image);
        return out;
	}
	
	/**
	 * PNG图片缩放
	 * @param maxWidth
	 * @param maxHeight
	 * @throws IOException 
	 */
	public static ByteArrayOutputStream resizePng(File srcFile, int maxWidth, int maxHeight) throws IOException{
		BufferedImage bufferedImage = ImageIO.read(srcFile);
		double rate = getResizeRate(srcFile, maxWidth, maxHeight);
		if(rate == 1){
			FileInputStream inputStream = new FileInputStream(srcFile);
			return toOutputStream(inputStream);
		}
		else{
			int width = (int)(bufferedImage.getWidth() * rate);
			int height =(int) (bufferedImage.getHeight() * rate);
			BufferedImage result  = Scalr.resize(bufferedImage, Method.AUTOMATIC, Mode.AUTOMATIC, width, height);
			ByteArrayOutputStream  out = new ByteArrayOutputStream ();
			ImageIO.write(result, "png", out);
			return  out;
		}
	}
	
	/**
	 * GIF图片缩放
	 * @param srcFile
	 * @param maxWidth
	 * @param maxHeight
	 * @throws IOException
	 */
	public static ByteArrayOutputStream resizeGif(File srcFile, int maxWidth, int maxHeight) throws IOException{
		GifDecoder gd = new GifDecoder();
		int status = gd.read(new FileInputStream(srcFile));
		if (status != GifDecoder.STATUS_OK) {
			return toOutputStream(new FileInputStream(srcFile));
		}
		double rate = getResizeRate(srcFile, maxWidth, maxHeight);
		if(rate == 1){
			return toOutputStream(new FileInputStream(srcFile));
		}
		else{
			ByteArrayOutputStream  out = new ByteArrayOutputStream ();
			AnimatedGifEncoder ge = new AnimatedGifEncoder();
			ge.start(out);
			ge.setRepeat(0);
			for (int i = 0; i < gd.getFrameCount(); i++) {
				BufferedImage frame = gd.getFrame(i);
				int width = frame.getWidth();
				int height = frame.getHeight();
				width = (int) (width * rate);
				height = (int) (height * rate);
	
				BufferedImage rescaled = Scalr.resize(frame, Mode.FIT_EXACT, width, height);
				int delay = gd.getDelay(i);
				ge.setDelay(delay);
				ge.addFrame(rescaled);
			}
			ge.finish();
			
			return out;
		}
	}
	
	/**
	 * 获取缩放比例
	 * @param srcFile
	 * @param maxWidth
	 * @param maxHeight
	 * @return
	 * @throws IOException 
	 */
	public static double getResizeRate(File srcFile, int maxWidth, int maxHeight) throws IOException{
		double rate = 1;
		if(maxWidth==0 && maxHeight == 0){
			return rate;
		}
		BufferedImage bImage = ImageIO.read(srcFile);
		int width = bImage.getWidth();
		int height = bImage.getHeight();
		if(maxWidth == 0 && height > maxHeight){
			rate = (double)maxHeight / height;
		}
		else if(maxHeight == 0 && width > maxWidth){
			rate = (double)maxWidth / width;
		}
		else if((width > maxWidth || height > maxHeight) && maxWidth > 0 && maxHeight > 0){
			rate = (double)maxWidth / width > (double)maxHeight / height ?  (double)maxHeight / height : (double)maxWidth / width ;
		}
		if(rate > 1){
			rate = 1;
		}
		return rate;
	}
	
	
	
	/**
	 * 获取图片类型（后缀）
	 * @param file
	 * @return
	 * @throws MagicException 
	 * @throws MagicMatchNotFoundException 
	 * @throws MagicParseException 
	 */
	public static String getType(File file) throws Exception{
		String mimeType = com.crawler.res.utils.FileUtils.getMimeType(file);
		return getType(mimeType);
	}
	
	
	/**
	 * 图片缩放
	 * @param srcFile
	 * @param maxWidth
	 * @param maxHeight
	 * @param type
	 * @param quality
	 * @throws IOException 
	 */
	public static ByteArrayOutputStream resize(File srcFile, int maxWidth, int maxHeight, String type, float quality) throws IOException{
		if(TYPE_GIF.equals(type)){
			return resizeGif(srcFile, maxWidth, maxHeight);
		}
		else if(TYPE_PNG.equals(type)){
			return resizePng(srcFile, maxWidth, maxHeight);
		}
		else{
			return resizeJpeg(srcFile, maxWidth, maxHeight, quality);
		}
	}
	
	/**
	 * 图片缩放
	 * @param srcFile
	 * @param maxWidth
	 * @param maxHeight
	 * @param type
	 * @throws IOException 
	 */
	public static ByteArrayOutputStream resize(File srcFile, int maxWidth, int maxHeight, String type) throws IOException{
		return resize(srcFile, maxWidth, maxHeight, type, 0.8F);
	}

	
	/**
	 * inputStream 转 outputStream
	 * @param inputStream
	 * @return
	 * @throws IOException
	 */
	public static ByteArrayOutputStream toOutputStream(InputStream inputStream) throws IOException{
		ByteArrayOutputStream swapStream = new ByteArrayOutputStream();
        int ch;
        while ((ch = inputStream.read()) != -1) {   
            swapStream.write(ch);   
        }
        return swapStream;
	}
	
	/**
	 * 获取图片类型（后缀）
	 * @param mimeType
	 * @return
	 */
	public static String getType(String mimeType){
		if("image/gif".equals(mimeType)){
			return TYPE_GIF;
		}
		else if("image/png".equals(mimeType) || "image/x-png".equals(mimeType)){
			return TYPE_PNG;
		}
		else if("image/bmp".equals(mimeType) || "image/x-ms-bmp".equals(mimeType)){
			return TYPE_BMP;
		}
		return TYPE_JPEG;
	}
}
