package com.crawler.res.utils;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.crawler.res.common.QiniuProperties;
import com.qiniu.common.QiniuException;
import com.qiniu.common.Zone;
import com.qiniu.http.Client;
import com.qiniu.http.Response;
import com.qiniu.processing.OperationManager;
import com.qiniu.storage.BucketManager;
import com.qiniu.storage.Configuration;
import com.qiniu.storage.UploadManager;
import com.qiniu.storage.model.FileInfo;
import com.qiniu.util.Auth;
import com.qiniu.util.StringMap;
import com.qiniu.util.UrlSafeBase64;

import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

/**
 * 七牛工具
 * @author rubekid
 *
 * 2016年12月31日 下午11:20:35
 */
public class QiNiuUtils {
	
	private static final Logger logger = LoggerFactory.getLogger(QiNiuUtils.class);

	public static final String DOMAIN = QiniuProperties.getProperty("qiniu.domain");
	public static final String ACCESS_KEY = QiniuProperties.getProperty("qiniu.access_key");
	public static final String SECRET_KEY = QiniuProperties.getProperty("qiniu.secret_key");
	public static final String BUCKET_NAME = QiniuProperties.getProperty("qiniu.bucket");
	public static final String ZONE =  QiniuProperties.getProperty("qiniu.zone");
	public static final String VIDEO_DOMAIN = QiniuProperties.getProperty("qiniu.video.domain", DOMAIN);
	public static final String VIDEO_BUCKET_NAME = QiniuProperties.getProperty("qiniu.video.bucket", BUCKET_NAME);
	public static final String PIPELINE =  QiniuProperties.getProperty("qiniu.pipeline", "admin_merge_radio");
	public static final String NOTIFY_URL =  QiniuProperties.getProperty("qiniu.notify");
	public static final String SALT =  QiniuProperties.getProperty("qiniu.salt", "crawler");

	private static Auth auth = null;
	
	private static Configuration cfg = null;
	
	/**
	 * 初始化配置
	 * 华东	Zone.zone0()
	 * 华北	Zone.zone1()
	 * 华南	Zone.zone2()
	 * 北美	Zone.zoneNa0()
	 */
	static{
		if(ACCESS_KEY != null && SECRET_KEY != null){
			auth = Auth.create(ACCESS_KEY, SECRET_KEY);
			Zone zone = Zone.autoZone();
			if("zone0".equals(ZONE)){
				zone = Zone.zone0();
			}
			else if("zone1".equals(ZONE)){
				zone = Zone.zone1();
			}
			else if("zone2".equals(ZONE)){
				zone = Zone.zone2();
			}
			else if("zoneNa0".equals(ZONE)){
				zone = Zone.zoneNa0();
			}

			cfg = new Configuration(zone);
		}

	}

	/**
	 * 简单上传，使用默认策略，只需要设置上传的空间名就可以了
	 * @return
	 */
	public static String getUpToken() {
		return getUpToken(BUCKET_NAME);
	}
	
	/**
	 * 获取上传Token
	 * @param override
	 * @return
	 */
	public static String getUpToken(String key, boolean override) {
		return getUpToken(BUCKET_NAME, key, override);
	}

	/**
	 * 根据空间名获取
	 * @param bucketName
	 * @return
	 */
	public static String getUpToken(String bucketName){
		return getUpToken(bucketName, null, false);
	}
	
	/**
	 * 获取上传Token
	 * @param bucketName
	 * @param override
	 * @return
	 */
	public static String getUpToken(String bucketName, String key, boolean override){
		StringMap policy =  null;
		if(override){
			return auth.uploadToken(bucketName, key);
		}
		return auth.uploadToken(bucketName, null, 3600 * 100, policy, true);
	}
	
	
	/**
	 * 获取上传token
	 * @param accessKey
	 * @param secretKey
	 * @param bucketName
	 * @return
	 */
	public static String getUpToken(String accessKey, String secretKey, String bucketName){
		Auth auth = Auth.create(accessKey, secretKey);
		return getUpToken(auth, bucketName);
	}
	
	/**
	 * 获取上传token
	 * @param auth
	 * @param bucketName
	 * @return
	 */
	public static String getUpToken(Auth auth, String bucketName){
		return auth.uploadToken(bucketName, null, 3600 * 100, null, true);
	}
	
	/**
	 * 获取私有下载地址
	 * @param baseUrl
	 * @param expires
	 * @return
	 */
	public static String getPrivateDownloadUrl(String baseUrl, Long expires){
		if(baseUrl == null || "".equals(baseUrl)){
			return baseUrl;
		}
		
		if(expires == null){
			return getPrivateDownloadUrl(baseUrl);
		}
		
		return auth.privateDownloadUrl(baseUrl, expires);
	}
	
	/**
	 * 获取私有下载地址
	 * @param baseUrl
	 * @return
	 */
	public static String getPrivateDownloadUrl(String baseUrl){
		if(baseUrl == null || "".equals(baseUrl)){
			return baseUrl;
		}
		
		return auth.privateDownloadUrl(baseUrl);
	}

	/**
	 * @param bucketName 空间名称
	 * @param data ;
	 * @param fileName 上传到七牛后保存的文件名
	 * @throws IOException
	 */
	public static String upload(String bucketName, byte[] data, String fileName){
		try {
			fileName = fileName.replace("\\", "/").replaceAll("^/", "");

			// 创建上传对象
			UploadManager uploadManager = new UploadManager(cfg);

			// 调用put方法上传
			Response res = uploadManager.put(data, fileName, getUpToken(bucketName));
			// 打印返回的信息
			logger.info("上传响应信息：" + res.bodyString());
			return getFullurl(fileName);
		} catch (QiniuException e) {
			logger.error(e.getMessage());
			throw new RuntimeException("上传失败");
		}
	}

	/**
	 * 上传
	 * @param bucket 空间名称
	 * @param filePath "/.../...";
	 * @param fileName 上传到七牛后保存的文件名
	 * @param override 是否覆盖
	 * @return
	 */
	public static String upload(String bucket, String filePath, String fileName, boolean override){
		try {

			fileName = fileName.replace("\\", "/").replaceAll("^/", "");
			// 创建上传对象
			UploadManager uploadManager = new UploadManager(cfg);

			// 调用put方法上传
			Response res = uploadManager.put(filePath, fileName, getUpToken(bucket, fileName, override));
			// 打印返回的信息
			logger.info("上传响应信息：" + res.bodyString());
			return getFullurl(fileName);
		} catch (QiniuException e) {
			logger.error(e.getMessage());
			throw new RuntimeException("上传失败");
		}
	}

	/**
	 * 上传
	 * @param bucket 空间名称
	 * @param bytes
	 * @param fileName 上传到七牛后保存的文件名
	 * @param override 是否覆盖
	 * @return
	 */
	public static String upload(String bucket, byte[] bytes, String fileName, boolean override){
		try {

			fileName = fileName.replace("\\", "/").replaceAll("^/", "");
			// 创建上传对象
			UploadManager uploadManager = new UploadManager(cfg);

			// 调用put方法上传
			Response res = uploadManager.put(bytes, fileName, getUpToken(bucket, fileName, override));
			// 打印返回的信息
			logger.info("上传响应信息：" + res.bodyString());
			return getFullurl(fileName);
		} catch (QiniuException e) {
			logger.error(e.getMessage());
			throw new RuntimeException("上传失败");
		}
	}
	
	/**
	 * 上传
	 * @param bucketName
	 * @param inputStream
	 * @param fileName
	 * @param mimeType
	 * @return
	 */
	public static String upload(String bucketName, InputStream inputStream, String fileName, String mimeType){
		try {
			fileName = fileName.replace("\\", "/").replaceAll("^/", "");
			// 创建上传对象
			UploadManager uploadManager = new UploadManager(cfg);

			// 调用put方法上传
			Response res = uploadManager.put(inputStream, fileName, getUpToken(bucketName), null, mimeType);
			// 打印返回的信息
			logger.info("上传响应信息：" + res.bodyString());
			return getFullurl(fileName);
		} catch (QiniuException e) {
			logger.error(e.getMessage());
			throw new RuntimeException("上传失败");
		}
	}
	
	/**
	 * 
	 * @param bucketName "/.../...";
	 * @param fileName 上传到七牛后保存的文件名
	 * @throws IOException
	 */
	public static String uploadInputStream(String bucketName, InputStream inputStream, String fileName){
		try {
			fileName = fileName.replace("\\", "/").replaceAll("^/", "");
			ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();  
			byte[] buff = new byte[1024];  
			int rc = 0;  
			while ((rc = inputStream.read(buff, 0, 1024)) > 0) {  
				byteArrayOutputStream.write(buff, 0, rc);  
			}  
			byte[] data = byteArrayOutputStream.toByteArray(); 
			return upload(bucketName, data, fileName);
		} catch (IOException e) {
			logger.error(e.getMessage());
			throw new RuntimeException("上传失败");
		}
	}

	/**
	 * 视频上传
	 * @param bucketName 空间名称
	 * @param data
	 * @param fileName 上传到七牛后保存的文件名
	 * @throws IOException
	 */
	public static String uploadVideo(String bucketName, byte[] data, String fileName){
		try {
			fileName = fileName.replace("\\", "/").replaceAll("^/", "");
			// 创建上传对象
			UploadManager uploadManager = new UploadManager(cfg);

			// 调用put方法上传
			Response res = uploadManager.put(data, fileName, getUpToken(bucketName));
			// 打印返回的信息
			logger.info("上传响应信息：" + res.bodyString());
			return QiNiuUtils.VIDEO_DOMAIN  + "/" + fileName;
		} catch (QiniuException e) {
			logger.error(e.getMessage());
			throw new RuntimeException("上传失败");
		}
	}

	/**
	 *  视频上传
	 * @param bucket 空间名称
	 * @param filePath "/.../...";
	 * @param fileName 上传到七牛后保存的文件名
	 * @throws IOException
	 */
	public static String uploadVideo(String bucket, String filePath, String fileName){
		try {
			fileName = fileName.replace("\\", "/").replaceAll("^/", "");
			// 创建上传对象
			UploadManager uploadManager = new UploadManager(cfg);

			// 调用put方法上传
			Response res = uploadManager.put(filePath, fileName, getUpToken(bucket));
			// 打印返回的信息
			logger.info("上传响应信息：" + res.bodyString());
			return QiNiuUtils.VIDEO_DOMAIN  + "/" + fileName;
		} catch (QiniuException e) {
			logger.error(e.getMessage());
			throw new RuntimeException("上传失败");
		}
	}
	
	/**
	 * 
	 * @param data
	 * @param fileName 上传到七牛后保存的文件名
	 * @throws IOException
	 */
	public static String upload(byte[] data, String fileName){
		return upload(BUCKET_NAME, data, fileName);
	}
	
	/**
	 * 
	 * @param inputStream 
	 * @param fileName 上传到七牛后保存的文件名
	 * @throws IOException
	 */
	public static String uploadInputStream(InputStream inputStream, String fileName){
		return uploadInputStream(BUCKET_NAME, inputStream, fileName);
	}
	
	/**
	 * 
	 * @param inputStream 
	 * @param fileName 上传到七牛后保存的文件名
	 * @param mimeType
	 * @throws IOException
	 */
	public static String upload(InputStream inputStream, String fileName, String mimeType){
		return upload(BUCKET_NAME, inputStream, fileName, mimeType);
	}
	
	/**
	 * 
	 * @param inputStream 
	 * @param fileName 上传到七牛后保存的文件名
	 * @throws IOException
	 */
	public static String upload(InputStream inputStream, String fileName){
		return upload(BUCKET_NAME, inputStream, fileName, null);
	}
	
	
	
	
	/**
	 * 
	 * @param bucket 空间名称
	 * @param filePath "/.../...";
	 * @param fileName 上传到七牛后保存的文件名
	 * @throws IOException
	 */
	public static String upload(String bucket, String filePath, String fileName){
		return upload(bucket, filePath, fileName, false);
	}
	

	
	/**
	 * 
	 * @param filePath "/.../...";
	 * @param fileName 上传到七牛后保存的文件名
	 * @throws IOException
	 */
	public static String upload(String filePath, String fileName){
		return upload(BUCKET_NAME, filePath, fileName);
	}
	
	/**
	 * 
	 * @param filePath "/.../...";
	 * @param fileName 上传到七牛后保存的文件名
	 * @param override 是否覆盖上传
	 * @throws IOException
	 */
	public static String upload(String filePath, String fileName, boolean override){
		return upload(BUCKET_NAME, filePath, fileName, override);
	}
	

	
	/**
	 *  视频上传
	 * @param data
	 * @param fileName 上传到七牛后保存的文件名
	 * @throws IOException
	 */
	public static String uploadVideo(byte[] data, String fileName){
		return uploadVideo(VIDEO_BUCKET_NAME, data, fileName);
	}


	

	
	
	/**
	 * 视频上传
	 * @param filePath "/.../...";
	 * @param fileName 上传到七牛后保存的文件名
	 * @throws IOException
	 */
	public static String uploadVideo(String filePath, String fileName){
		return uploadVideo(VIDEO_BUCKET_NAME, filePath, fileName);
	}
	
	/**
	 * 根据地址获取图片信息
	 * @param url
	 * @return
	 */
	public static JSONObject imageInfo(String url){
		
		JSONObject result = new JSONObject();
		try {
			Client client = new Client();
			url = url.split("\\?")[0];
			Response response = client.get(url + "?imageInfo");
			result = JSONObject.fromObject(response.bodyString());
		} catch (QiniuException e) {
			e.printStackTrace();
		}
		return result;
	}
	
	/**
	 * 视频信息
	 * @param url
	 * @return
	 */
	public static JSONObject avinfo(String url){
		JSONObject result = new JSONObject();
		try{
			Client client = new Client();
			url = url.split("\\?")[0];
			Response response = client.get(url + "?avinfo");
			JSONObject json =  JSONObject.fromObject(response.bodyString());
			JSONArray streams = json.getJSONArray("streams");
			
			JSONObject format = json.getJSONObject("format");
			int width = 0;
			int height = 0;
			int rotate = 0;
			for(int i=0; i<streams.size(); i++){
				JSONObject baseInfo = streams.getJSONObject(i);
				if(baseInfo.containsKey("width")){
					width = baseInfo.getInt("width");
					height = baseInfo.getInt("height");
					if(baseInfo.containsKey("tags")){						
						JSONObject tags = baseInfo.getJSONObject("tags");
						if(tags.containsKey("rotate")){
							rotate = Math.abs(tags.getInt("rotate"));
							if(rotate == 90 || rotate == 270){
								int tmp = width;
								width = height;
								height = tmp;
							}
						}
					}
					break;
				}
			}
			if(width == 0){
				width = 960;
				height = 544;
			}
			result.put("width", width);
			result.put("height", height);

			
			result.put("size", format.get("size"));
			result.put("duration", format.get("duration"));
		}
		catch(QiniuException e){
			e.printStackTrace();
		}
		return result;
	}
	
	/**
	 * 获取缩率图
	 * @param url
	 * @param width
	 * @param height
	 * @param quality
	 * @return
	 */
	public static String getThumb(String url, int width, int height, int quality){
		return String.format("%s?imageView2/2/w/%d/h/%d/q/%d", url, width, height, quality);
		
	}
	
	/**
	 *  获取缩率图
	 * @param url
	 * @param width
	 * @param height
	 * @return
	 */
	public static String getThumb(String url, int width, int height){
		return String.format("%s?imageView2/2/w/%d/h/%d", url, width, height);
	}
	
	/**
	 *  获取缩率图
	 * @param url
	 * @param width
	 * @return
	 */
	public static String getThumb(String url, int width){
		return String.format("%s?imageView2/2/w/%d", url, width);
	}
	
	/**
	 * 图片裁剪
	 * @param url
	 * @param width
	 * @param height
	 * @return
	 */
	public static String clip(String url, int width, int height){
		url = url.split("\\?")[0];
		return String.format("%s?imageView2/1/w/%d/h/%d", url, width, height);
	}
	
	/**
	 * 上传视频封面
	 * @param url
	 * @param width
	 * @param height
	 * @param offset
	 * @param 
	 * @return
	 */
	public static String uploadVideoCover(String url, int width, int height, int offset){
		String cover = String.format("%s?vframe/jpg/offset/%d/w/%d/h/%d", url, offset, width, height);
		String fileName = UUID.randomUUID() +  ".jpg";
		return  transfer(cover, fileName);
	}
	
	/**
	 * 上传视频封面
	 * @param url
	 * @param width
	 * @param height
	 * @return
	 */
	public static String uploadVideoCover(String url, int width, int height){
		return uploadVideoCover(url, width, height, 1);
	}
	
	/**
	 * 获取视频封面
	 * @param url
	 * @param width
	 * @param height
	 * @param offset
	 * @return
	 */
	public static String getVideoCover(String url, int width, int height, int offset){
		return String.format("%s?vframe/jpg/offset/%d/w/%d/h/%d", url, offset, width, height);
	}
	
	/**
	 * 获取视频封面
	 * @param url
	 * @param width
	 * @param height
	 * @return
	 */
	public static String getVideoCover(String url, int width, int height){
		return getVideoCover(url, width, height, 1);
	}
	
	/**
	 * 网络图片迁移
	 * @param url
	 * @param fileName
	 * @throws IOException 
	 */
	public static String transfer(String url, String fileName){
		if(fileName == null){
			fileName = UUID.randomUUID() + url.substring(url.lastIndexOf("."));
		}
		byte[] types = toByte(url);
		if(types == null){
			return null;
		}
		QiNiuUtils.upload(types, fileName);
		return getFullurl(fileName);
	}
	
	/**
	 * 图片迁移
	 * @param url
	 * @return
	 * @throws IOException 
	 */
	public static String transfer(String url){
		return transfer(url, null);
	}
	
	/**
	 * 网络资源转byte[]
	 * @param urlString 网络地址
	 * @return
	 * @throws IOException
	 */
	private static byte[] toByte(String urlString){
		try{
			URL url = new URL(urlString);  
	        URLConnection con = url.openConnection();
	        con.setConnectTimeout(5*1000);  
	        InputStream is = con.getInputStream();
	        ByteArrayOutputStream output = new ByteArrayOutputStream();
	        byte[] buffer = new byte[4096];
	        int n = 0;
	        while (-1 != (n = is.read(buffer))) {
	            output.write(buffer, 0, n);
	        }
	        is.close();  
	        return output.toByteArray();
		}
        catch (IOException ex) {
			logger.error(ex.getMessage(), ex);
		}
		return null;
	}

	
	/**
	 * 执行转码和另存 操作
	 * @return
	 * @throws QiniuException 
	 */
	public static String toMp4(String fileName, String newFileName) throws QiniuException{
        return toMp4(fileName, newFileName, "640x360");
	}
	
	/**
	 * 执行转码和另存 操作
	 * @return
	 * @throws QiniuException 
	 */
	public static String toMp4(String fileName, String newFileName, String resolution) throws QiniuException{
		if(resolution == null || "".equals(resolution)){
			resolution = "640x360";
		}
        String pipeline = PIPELINE;  //处理队列
        String saveAs = UrlSafeBase64.encodeToString(VIDEO_BUCKET_NAME + ":" + newFileName);        //saveas接口 参数
        String fops = "avthumb/mp4/s/" + resolution +  "/vcodec/libx264|saveas/" + saveAs;                //处理命令 avthumb 和 saveas 通过管道符 |  进行连接
	    OperationManager operationMgr = new OperationManager(auth, cfg);
        String persistentId = operationMgr.pfop(VIDEO_BUCKET_NAME, fileName, fops, pipeline, NOTIFY_URL);
        return persistentId;
	}
	
	/**
	 * 转换成 m3u8
	 * @param fileName
	 * @return
	 * @throws QiniuException
	 */
	public static String m3u8(String fileName) throws QiniuException{
		String m3u8Path = getM3u8Path(fileName);
		String pipeline = PIPELINE;  //处理队列
        String saveAs = UrlSafeBase64.encodeToString(VIDEO_BUCKET_NAME + ":" + m3u8Path);  //saveas接口 参数
        String fops = "avthumb/m3u8/noDomain/1|saveas/" + saveAs;                //处理命令 avthumb 和 saveas 通过管道符 |  进行连接
	    OperationManager operationMgr = new OperationManager(auth, cfg);
        String persistentId = operationMgr.pfop(VIDEO_BUCKET_NAME, fileName, fops, pipeline, NOTIFY_URL);
        return persistentId;
	}
	
	/**
	 * 删除
	 * @param url
	 * @throws QiniuException
	 */
	public static void remove(String url) throws QiniuException{
		// 创建上传对象
		BucketManager bucketManager = new BucketManager(auth, cfg);
		url = url.split("\\?")[0];
		String key = url.replace(QiNiuUtils.DOMAIN + "/", "");
		bucketManager.delete(BUCKET_NAME, key);
	}
	
	/**
	 * 获取m3u8 路径
	 * @param fileName
	 * @return
	 */
	public static String getM3u8Path(String fileName){
		if(fileName.contains("/")){
			fileName = fileName.substring(fileName.lastIndexOf("/"));
		}
		return "video/m3u8/" + FileUtils.shortMd5(fileName, SALT) + ".m3u8";
	}
	
	/**
	 * 获取m3u8完整地址
	 * @param fileName
	 * @return
	 */
	public static String getM3u8FullPath(String fileName){
		return VIDEO_DOMAIN + "/" + getM3u8Path(fileName);
	}
	
	/**
	 * 获取m3u8私有地址
	 * @param fileName
	 * @return
	 */
	public static String getM3u8PrivatePath(String fileName){
		String path = getM3u8FullPath(fileName);
		return getPrivateDownloadUrl(path + "?pm3u8/0");
	}
	
	/**
	 * 文件列表
	 * @param bucket 空间名称
	 * @param prefix 文件名前缀
	 * @param limit 每次迭代的长度限制，最大1000，推荐值 1000
	 * @param delimiter 指定目录分隔符，列出所有公共前缀（模拟列出目录效果）。缺省值为空字符串
	 * @return
	 */
	public static List<FileInfo> listFiles(String bucket, String prefix, int limit, String delimiter){
		BucketManager bucketManager = new BucketManager(auth, cfg);
		BucketManager.FileListIterator fileListIterator = bucketManager.createFileListIterator(bucket, prefix, limit, delimiter);
		List<FileInfo> list = new ArrayList<FileInfo>();
		while (fileListIterator.hasNext()) {
		    FileInfo[] items = fileListIterator.next();
		    for (FileInfo item : items) {
		    	list.add(item);
		    }
		}
		return list;
	}
	
	/**
	 * 文件列表
	 * @param bucket
	 * @return
	 */
	public static List<FileInfo> listFiles(String bucket){
		return listFiles(bucket, "", 1000, "");
	}
	
	/**
	 * 文件列表
	 * @return
	 */
	public static List<FileInfo> listFiles(){
		return listFiles(BUCKET_NAME);
	}

	/**
	 * 获取完整路径
	 * @param fileName
	 * @return
	 */
	public static String getFullurl(String fileName){
		return QiNiuUtils.DOMAIN.replaceAll("/$", "") + "/" + fileName;
	}
}
