package com.crawler.weixin.utils;

import java.io.IOException;
import java.util.Date;

import javax.servlet.http.HttpServletResponse;

import org.apache.commons.codec.digest.DigestUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.crawler.redis.utils.RedisManager;
import com.crawler.social.config.SocialProperties;
import com.crawler.social.config.WeixinConstant;
import com.crawler.social.utils.BeanUtils;
import com.crawler.utils.http.SimpleHttpClient;
import com.crawler.weixin.common.StringUtils;
import com.crawler.weixin.enums.QrcodeActionName;
import com.crawler.weixin.model.Menu;
import com.crawler.weixin.vo.AccessToken;
import com.crawler.weixin.vo.FansUserInfo;
import com.crawler.weixin.vo.QrcodeResponse;
import com.crawler.weixin.vo.Ticket;
import com.crawler.weixin.vo.WeixinAccessToken;
import com.crawler.weixin.vo.WeixinUser;
import com.crawler.weixin.vo.WxJsConfig;

import net.sf.json.JSONObject;

/**
 * 微信工具类
 * @author rubekid
 *
 * 2017年10月14日 下午7:33:50
 */
public class WeixinUtils {
	
	private static final Logger logger = LoggerFactory.getLogger(WeixinUtils.class);
	
	/**
	 * 缓存前缀
	 */
	private static final String WEIXIN_CACHE_PREFIX = SocialProperties.getProperty("app.id", "social") +  ":weixin";
	
	/**
	 * 获取公众号token
	 */
	private static final String ACCESS_TOKEN_URL = "https://api.weixin.qq.com/cgi-bin/token?grant_type=client_credential&appid=%s&secret=%s";
	
	/**
	 * 获取公众号票据
	 */
	private static final String TICKET_URL = "https://api.weixin.qq.com/cgi-bin/ticket/getticket?access_token=%s&type=jsapi";
	
	/**
	 * 获取tokenURL
	 */
	private static final String USER_ACCESS_TOKEN_URL = "https://api.weixin.qq.com/sns/oauth2/access_token?appid=%s&secret=%s&code=%s&grant_type=authorization_code";
	
	/**
	 * 获取用户信息URL
	 */
	private static final String USERINFO_URL = "https://api.weixin.qq.com/sns/userinfo?access_token=%s&openid=%s&lang=zh_CN";
	
	/**
	 * 授权登录
	 */
	private static final String AUTHORIZE_URL  = "https://open.weixin.qq.com/connect/oauth2/authorize?appid=%s&redirect_uri=%s&response_type=code&scope=snsapi_userinfo&state=1#wechat_redirect";
	
	/**
	 * 获取粉丝信息
	 */
	private static final String FANS_USER_INFO_URL = "https://api.weixin.qq.com/cgi-bin/user/info?access_token=%s&openid=%s&lang=zh_CN";
	
	/**
	 * 二维码
	 */
	private static final String CREATE_QRCODE = "https://api.weixin.qq.com/cgi-bin/qrcode/create?access_token=%s";
	
	/**
	 * 菜单
	 */
	private static final String CREATE_MENU = "https://api.weixin.qq.com/cgi-bin/menu/create?access_token=%s";

	
	/**
	 * 通过code换取网页授权access_token
	 * @param appId
	 * @param appSecret
	 * @param code
	 */
	public static WeixinAccessToken getUserAccessToken(String appId, String appSecret, String code){
		String url = String.format(USER_ACCESS_TOKEN_URL, appId, appSecret, code);
		return SimpleHttpClient.getForObject(url, WeixinAccessToken.class);		
	}
	
	/**
	 * 通过code换取网页授权access_token
	 * @param code
	 */
	public static WeixinAccessToken getUserAccessToken(String code){
		String url = String.format(USER_ACCESS_TOKEN_URL, WeixinConstant.APP_ID, WeixinConstant.APP_SECRET, code);
		return SimpleHttpClient.getForObject(url, WeixinAccessToken.class);		
	}
	
	/**
	 * 获取用户信息
	 * @param accessToken
	 * @param openid
	 */
	public static WeixinUser getUserInfo(String accessToken, String openid){
		String url = String.format(USERINFO_URL, accessToken, openid);
		return SimpleHttpClient.getForObject(url, WeixinUser.class);
	}
	
	
	/**
	 * 获取Token
	 * @param appId
	 * @param appSecret
	 * @param code
	 * @return
	 */
	public static AccessToken getAccessToken(String appId, String appSecret){
		String key = accessTokenKey(appId);
		AccessToken accessToken = RedisManager.get(key, AccessToken.class);
		logger.info("AccessToken：{}", BeanUtils.toJSONString(accessToken));
		if(accessToken == null){
			String url = String.format(ACCESS_TOKEN_URL, appId, appSecret);
			accessToken =  SimpleHttpClient.getForObject(url, AccessToken.class);
			accessToken.setExpireAt(new Date(System.currentTimeMillis() + accessToken.getExpiresIn() * 1000));
			RedisManager.set(key, accessToken, accessToken.getExpiresIn());
			
			logger.info("新获取到AccessToken：{}", BeanUtils.toJSONString(accessToken));
		}
		return accessToken;
	}
	
	/**
	 * 获取票据
	 * @param accessToken
	 * @return
	 */
	public static Ticket getTicket(String accessToken){
		String key = ticketKey(accessToken);
		Ticket ticket = RedisManager.get(key, Ticket.class);
		if(ticket == null){
			String url = String.format(TICKET_URL, accessToken);
			ticket =  SimpleHttpClient.getForObject(url, Ticket.class);
			ticket.setExpireAt(new Date(System.currentTimeMillis() + ticket.getExpiresIn() * 1000));
			RedisManager.set(key, ticket, ticket.getExpiresIn());
		}
		return ticket;
	}
	
	public static WxJsConfig signature(String url){
		return signature(WeixinConstant.APP_ID, WeixinConstant.APP_SECRET, url);
	}
	
	/**
	 * 获取微信签名
	 * @param appId
	 * @param secret
	 * @param url
	 * @return
	 */
	public static WxJsConfig signature(String appId, String secret, String url){
		WxJsConfig wxJsConfig = new WxJsConfig();
		wxJsConfig.setAppId(appId);
		wxJsConfig.setNonceStr(StringUtils.getRandomString(16));
		wxJsConfig.setTimestamp(System.currentTimeMillis() / 1000);
		AccessToken accessToken = getAccessToken(appId, secret);
		Ticket ticket = getTicket(accessToken.getAccessToken());
		String str = String.format("jsapi_ticket=%s&noncestr=%s&timestamp=%s&url=%s", ticket.getTicket(), wxJsConfig.getNonceStr(), wxJsConfig.getTimestamp(),url);
		String signature = DigestUtils.shaHex(str.getBytes());
		wxJsConfig.setSignature(signature);
		return wxJsConfig;
	}
	
	/**
	 * 授权登录
	 * @param response
	 * @param redirectUrl
	 * @throws IOException
	 */
	public static void authorize(HttpServletResponse response, String redirectUrl) throws IOException{
		authorize(response, WeixinConstant.APP_ID, redirectUrl);
	}
	
	/**
	 * 授权登录
	 * @param response
	 * @param appId
	 * @param redirectUrl
	 * @throws IOException
	 */
	public static void authorize(HttpServletResponse response, String appId, String redirectUrl) throws IOException{
		String location = String.format(AUTHORIZE_URL, appId, redirectUrl);
		response.sendRedirect(location);
	}
	
	/**
	 * 获取粉丝用户信息
	 * @param accessToken
	 * @param openid
	 */
	public static FansUserInfo getFansUserInfo(String accessToken, String openid){
		String url = String.format(FANS_USER_INFO_URL, accessToken, openid);
		return SimpleHttpClient.getForObject(url, FansUserInfo.class);
	}
	
	/**
	 * 获取粉丝信息
	 * @param openid
	 * @return
	 */
	public static FansUserInfo getFansUserInfo(String openid){
		return getFansUserInfo(WeixinConstant.APP_ID, WeixinConstant.APP_SECRET, openid);
	}
	
	/**
	 * 获取粉丝信息
	 * @param openid
	 * @return
	 */
	public static FansUserInfo getFansUserInfo(String appId, String appSecret, String openid){
		AccessToken accessToken = getAccessToken(appId, appSecret);
		return getFansUserInfo(accessToken.getAccessToken(), openid);
	}
	
	/**
	 * 生成二维码
	 * @param actionName
	 * @param sceneId
	 * @param expireSeconds
	 * @return
	 */
	public static QrcodeResponse createQrcode(QrcodeActionName actionName, String sceneId, Integer expireSeconds){
		return createQrcode(WeixinConstant.APP_ID, WeixinConstant.APP_SECRET, actionName, sceneId, expireSeconds);
	}
	
	
	/**
	 * 生成二维码
	 * @param appId
	 * @param appSecret
	 * @param actionName
	 * @param sceneId
	 * @param expireSeconds
	 * @return
	 */
	public static QrcodeResponse createQrcode(String appId, String appSecret, QrcodeActionName actionName, String sceneId, Integer expireSeconds){
		AccessToken accessToken = getAccessToken(appId, appSecret);
		return createQrcode(accessToken, actionName, sceneId, expireSeconds);
	}
	
	/**
	 * 生成二维码
	 * @param accessToken AccessToken
	 * @param actionName
	 * @param sceneId
	 * @param expireSeconds
	 * @return
	 */
	public static QrcodeResponse createQrcode(AccessToken accessToken, QrcodeActionName actionName, String sceneId, Integer expireSeconds){
		JSONObject scene = new JSONObject();
		if(StringUtils.isNumeric(sceneId)){				
			scene.put("scene_id", Long.parseLong(sceneId));
		}
		else{
			scene.put("scene_str", sceneId);
		}
		JSONObject body = new JSONObject();
		JSONObject actionInfo = new JSONObject();
		actionInfo.put("scene", scene);
		body.put("action_name", actionName);
		if((actionName.equals(QrcodeActionName.QR_SCENE)
				|| actionName.equals(QrcodeActionName.QR_STR_SCENE))
			&& expireSeconds != null){
			body.put("expire_seconds", expireSeconds);
		}
		body.put("action_info", actionInfo);

		String url = String.format(CREATE_QRCODE, accessToken.getAccessToken());
		return SimpleHttpClient.postForObject(url, body, QrcodeResponse.class);
	}
	
	
	/**
	 * 创建菜单
	 * @param menu
	 */
	public static boolean createMenu(Menu menu) {
		return createMenu(menu, WeixinConstant.WECHAT_APP_ID, WeixinConstant.WECHAT_APP_SECRET);
	}
	
	/**
	 * 创建菜单
	 * @param menu
	 * @param appId
	 * @param appSecret
	 */
	public static boolean createMenu(Menu menu, String appId, String appSecret) {
		AccessToken accessToken = getAccessToken(appId, appSecret);
		String url = String.format(CREATE_MENU, accessToken.getAccessToken());
		String content = SimpleHttpClient.post(url, JSONObject.fromObject(BeanUtils.toJSONString(menu)));
		logger.info("响应内容：{}", content);
		if(content.startsWith("{") && content.endsWith("}")) {
			JSONObject response = JSONObject.fromObject(content);
			if(response.getInt("errcode") == 0) {
				return true;
			}
		}
		return false;
	}
	
	/**
	 * Access Token 缓存key
	 * @param appId
	 * @return
	 */
	private static String accessTokenKey(String appId){
		return String.format("%s:%s:%s", WEIXIN_CACHE_PREFIX, "access_token", appId);
	}
	
	/**
	 * 票据 缓存key
	 * @param accessToken
	 * @return
	 */
	private static String ticketKey(String accessToken){
		return String.format("%s:%s:%s", WEIXIN_CACHE_PREFIX, "ticket", accessToken);
	}
	
	

}