package com.crawler.utils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

import com.crawler.beans.TimeDuration;

/**
 * 时间通用方法静态类
 * @author rubekid
 *
 * 2017年1月12日 下午1:41:00
 */
public class DateUtils {
	
	private static SimpleDateFormat longFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
	private static SimpleDateFormat myFormat = new SimpleDateFormat("yyyy-MM-dd");
	private static SimpleDateFormat shortFormat = new SimpleDateFormat("yyyy-MM");
	private static SimpleDateFormat monthFormat = new SimpleDateFormat("MM");

	/**
	 * 取得指定月份的第一天
	 * 
	 * @param strdate
	 *            String
	 * @return String
	 */
	public static String getMonthBegin(String strdate) {
		Date date = StringToDate(strdate);
		return formatDateByFormat(date, "yyyy-MM") + "-01";
	}

	/**
	 * 取得指定月份的最后一天
	 * 
	 * @param strdate
	 *            String
	 * @return String
	 */
	public static String getMonthEnd(String strdate) {
		Date date = StringToDate(getMonthBegin(strdate));

		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		calendar.add(Calendar.MONTH, 1);
		calendar.add(Calendar.DAY_OF_YEAR, -1);
		return formatDate(calendar.getTime());
	}

	/**
	 * 取得指定月份的总天数
	 * 
	 * @param strdate
	 *            String
	 * @return String
	 */
	public static int getMonthDaynum(String strdate) {
		String enddate = getMonthEnd(strdate);
		return Integer.parseInt(enddate.substring(enddate.length() - 2, enddate.length()));
	}

	/**
	 * 得到strDate日期的周的星期一
	 * 
	 * @param strDate
	 * @return
	 */
	public static String getWeekBegin(String strDate) {
		Date date = StringToDate(strDate);
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		int dayOfWeek = calendar.get(Calendar.DAY_OF_WEEK);// 1为周日.
		if (dayOfWeek == 1) {
			dayOfWeek = 6;
		} else {
			dayOfWeek = dayOfWeek - 2;
		}
		calendar.add(Calendar.DAY_OF_YEAR, 0 - dayOfWeek);
		return formatDate(calendar.getTime());
	}

	/**
	 * 常用的格式化日期
	 * 
	 * @param date
	 *            Date
	 * @return String
	 */
	public static String formatDate(Date date) {
		return formatDateByFormat(date, "yyyy-MM-dd");
	}

	/**
	 * 是否在同一分钟
	 * 
	 * @param d1
	 * @param d2
	 * @return
	 */
	@SuppressWarnings("deprecation")
	public static boolean isSameMinute(Date d1, Date d2) {
		boolean result;
		if (d1.getTime() - d2.getTime() >= 60 * 1000 || d1.getTime() - d2.getTime() < -60 * 1000) {
			result = false;
		} else {
			result = d1.getMinutes() == d2.getMinutes();
		}
		return result;
	}

	/**
	 * 常用的格式化日期
	 * 
	 * @param date Date
	 * @return String
	 */
	public static String formatDateTime(Date date) {
		return formatDateByFormat(date, "yyyy-MM-dd HH:mm:ss");
	}

	/**
	 * 以指定的格式来格式化日期
	 * 
	 * @param date
	 *            Date
	 * @param format
	 *            String
	 * @return String
	 */
	public static String formatDateByFormat(Date date, String format) {
		String result = "";
		if (date != null) {
			try {
				SimpleDateFormat sdf = new SimpleDateFormat(format);
				result = sdf.format(date);
			} catch (Exception ex) {
				ex.printStackTrace();
			}
		}
		return result;
	}





	/**
	 * 取得系统的当前年份,类型为String
	 * 
	 * @return String
	 */
	public static String getYearNow() {
		Date now = new Date();
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy");
		return sdf.format(now);
	}

	/**
	 * 取得系统的当前月份,类型为String
	 * 
	 * @return String
	 */
	public static String getMonthNow() {
		Date now = new Date();
		SimpleDateFormat sdf = new SimpleDateFormat("MM");
		return sdf.format(now);
	}


	/**
	 * 获取当前日期
	 * 
	 * @return
	 */
	public static String getDateNow() {
		Date now = new Date();
		SimpleDateFormat sdf = new SimpleDateFormat("dd");
		return sdf.format(now);
	}

	

	/**
	 * date型转化为String 格式为yyyy/MM/dd
	 * 
	 * @param date
	 * @param strDefault
	 * @return
	 */
	public static String DateToString(java.sql.Date date, String strDefault) {
		String strTemp = strDefault;
		if (date != null) {
			SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
			strTemp = formatter.format(date);
		}
		return strTemp;
	}

	/**
	 * date型转化为String 格式为yyyy/MM/dd
	 * 
	 * @param date
	 * @param strDefault
	 * @return
	 */
	public static String DateToString(Date date) {
		String strTemp = "";
		if (date != null) {
			SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
			strTemp = formatter.format(date);
		}
		return strTemp;
	}

	/**
	 * date型转化为String 格式为hh/mm/ss
	 * 
	 * @param date
	 * @param strDefault
	 * @return
	 */
	public static String DateToString2(Date date) {
		String strTemp = "";
		if (date != null) {
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm");
			strTemp = formatter.format(date);
		}
		return strTemp;
	}

	/**
	 * date型转化为String 格式为hh/mm/ss
	 * 
	 * @param date
	 * @param strDefault
	 * @return
	 */
	public static String DateToString3(Date date) {
		String strTemp = "";
		if (date != null) {
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			strTemp = formatter.format(date);
		}
		return strTemp;
	}

	/**
	 * date型转化为String 格式为yyyy-MM
	 * 
	 * @param date
	 * @param strDefault
	 * @return
	 */
	public static String DateToString3(java.sql.Date date, String strDefault) {
		String strTemp = strDefault;
		if (date != null) {
			SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM");
			strTemp = formatter.format(date);
		}
		return strTemp;
	}

	public static String DateToString2(java.sql.Date date, String strDefault) {
		String strTemp = strDefault;
		if (date != null) {
			SimpleDateFormat formatter = new SimpleDateFormat("yyyyMMdd");
			strTemp = formatter.format(date);
		}
		return strTemp;
	}

	public static String DateToString(java.sql.Date date) {
		return DateToString(date, null);
	}

	

	/**
	 * String转化为java.sql.date类型，
	 * 
	 * @param strDate
	 * @return
	 */
	public static java.sql.Date StringToDate(String strDate) {
		if (strDate != null && !strDate.equals("")) {
			try {
				SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
				Date d = formatter.parse(strDate);
				java.sql.Date numTime = new java.sql.Date(d.getTime());
				return numTime;
			} catch (Exception e) {
				return null;
			}
		} else {
			return null;
		}
	}

	/**
	 * String转化为java.sql.date类型，
	 * 
	 * @param strDate
	 * @return
	 */
	public static java.sql.Date StringToDateLong(String strDate) {
		if (strDate != null && !strDate.equals("")) {
			try {
				SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
				Date d = formatter.parse(strDate);
				java.sql.Date numTime = new java.sql.Date(d.getTime());
				return numTime;
			} catch (Exception e) {
				return null;
			}
		} else {
			return null;
		}
	}

	/**
	 * 得到上个月的日期
	 * 
	 * @param date
	 * @return
	 */
	public static Date getPreDate(Date date) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		calendar.add(Calendar.MONTH, -1);
		return new Date(calendar.getTime().getTime());
	}
	
	/**
	 * 得到昨天的日期
	 * 
	 * @param date
	 * @return
	 */
	public static Date getYesterDayDate() {
		Calendar calendar = Calendar.getInstance();
		calendar.add(Calendar.DATE, -1);
		return new Date(calendar.getTime().getTime());
	}

	/**
	 * 得到上个月的月份（返回格式MM）
	 * 
	 * @param date
	 * @return
	 */
	public static String getPreMonth() {
		Calendar calendar = Calendar.getInstance();
		calendar.add(Calendar.MONTH, -1);
		return monthFormat.format(calendar.getTime());
	}

	/**
	 * 得到下一个月的字符串
	 * 
	 * @param strDate
	 * @return
	 * @throws ParseException
	 */
	public static String getPreMonthStr(String strDate) {
		Date fromDate = StringToDate(strDate);
		Date toDate = getPreDate(fromDate);
		return DateToString(toDate);
	}

	


	/**
	 * 得到日期在月份的第几周
	 * 
	 * @param date
	 * @return
	 */
	public static int getWeekOfMonth(String strDate) {
		Date date = StringToDate(strDate);
		Calendar calendar = Calendar.getInstance();
		calendar.setFirstDayOfWeek(Calendar.MONDAY);
		calendar.setTime(date);

		return calendar.get(Calendar.WEEK_OF_MONTH);
	}

	/**
	 * 传入日期加1
	 * 
	 * @param date
	 * @return (date + 1)
	 */
	public static String getQueryDate(String date) {
		Date d = StringToDate(date);
		return formatDate(getDateAfter(d, 1));
	}

	/**
	 * 给指定日期加减天数，返回结果日期
	 * 
	 * @param date
	 * @param day
	 * @return
	 */
	public static Date getDateAfter(Date date, int day) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		calendar.add(Calendar.DAY_OF_YEAR, day);
		Date newdate = calendar.getTime();
		return newdate;
	}

	/**
	 * 返回当前时间(格式：yyyy-MM-dd hh:mm:ss)
	 */
	public static String getLongNow() {
		Calendar calendar = Calendar.getInstance();
		return longFormat.format(calendar.getTime());
	}

	/**
	 * 将字符串时间转换为Date时间格式(字符串格式：yyyy-MM-dd hh:mm:ss)
	 * 
	 * @param str
	 * @return
	 */
	public static Date getDateByLongFormat(String str) {
		if (str == null || str.length() < 1)
			return null;
		Date result = null;
		try {
			result = longFormat.parse(str);
		} catch (Exception e) {
		}
		return result;
	}

	/**
	 * 返回时间(格式：yyyy-MM-dd hh:mm:ss)
	 */
	public static String getLongTime(long time) {
		return longFormat.format(time);
	}

	/**
	 * 返回当前时间(格式：yyyy-MM-dd)
	 */
	public static String getNow() {
		Calendar calendar = Calendar.getInstance();
		return myFormat.format(calendar.getTime());
	}

	/**
	 * 返回当前年月（格式：yyyy-MM）
	 * 
	 * @return
	 */
	public static String getYearMonth() {
		Calendar calendar = Calendar.getInstance();
		return shortFormat.format(calendar.getTime());
	}

	/**
	 * 返回当前小时数(24小时)
	 * 
	 * @return
	 */
	public static int getHour() {
		Calendar calendars = Calendar.getInstance();
		int hour = calendars.get(Calendar.HOUR_OF_DAY);
		return hour;
	}

	/**
	 * 获得昨天的日期
	 */
	public static String getYesterday() {
		Date d = new Date();
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(d);
		calendar.add(Calendar.DAY_OF_YEAR, -1);
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
		return sdf.format(calendar.getTime());
	}

	/**
	 * 获得昨天的月份
	 */
	public static String getYesterdayMonth() {
		Date d = new Date();
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(d);
		calendar.add(Calendar.DAY_OF_YEAR, -1);
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM");
		return sdf.format(calendar.getTime());
	}

	/**
	 * 获得一个员工在公司的完整年数
	 * 
	 * @param date
	 * @return
	 */
	public static int getYearCount(Date date, Date date2) {
		int count = 0;
		// 获得员工在公司的完整年数
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy");
		int year = Integer.parseInt(sdf.format(date));
		int cyear = Integer.parseInt(sdf.format(date2));
		count = cyear - year - 1;
		count = count < 0 ? 0 : count;
		return count;
	}

	public static String getChineseDateStr(String strdate) {
		Date date = StringToDate(strdate);
		String datearr[] = strdate.split("-");
		String week[] = { "日", "一", "二", "三", "四", "五", "六" };
		String s = datearr[0] + "年" + datearr[1] + "月" + datearr[2] + "日  星期";
		@SuppressWarnings("deprecation")
		int weekday = date.getDay();
		return s + week[weekday];
	}

	public static String getChineseDateStr2(String strdate) {
		String datearr[] = strdate.split("-");
		if (datearr[1].startsWith("0"))
			datearr[1] = datearr[1].substring(1, datearr[1].length());
		String s = datearr[1] + "月" + datearr[2] + "日 ";
		return s;
	}

	public static String getChineseDateStr3(String strdate) {
		String datearr[] = strdate.split("-");
		String s = datearr[0] + "年" + datearr[1] + "月" + datearr[2] + "日 ";
		return s;
	}
	
	/**
	 * 转换unix long时间为Date对象
	 * @param time
	 * @return
	 */
	public static Date fromUnixLongToDate(long time){
		time = fromServerTimeToTimeStamp(time);
		return new Date(time);
	}

	/**
	 * 得到某年的所有自然周：注意，calendar将周日做为周的第一天
	 * 
	 * @param year
	 * @return String[2]:String[0] 每周一字符串 String[1] 每周日字符串
	 */
	public static String[] getAllWeekStr(int year) {
		String[] result = { "", "" };
		String firstday = String.valueOf(year) + "-01-01";
		String lastday = String.valueOf(year) + "-12-31";
		Calendar c = Calendar.getInstance();
		c.set(Calendar.YEAR, year);// 设置年
		c.set(Calendar.WEEK_OF_YEAR, 1);// 设置周
		Calendar d = (Calendar) c.clone();
		c.set(Calendar.DAY_OF_WEEK, 1);// 设置周的第一天
		if (myFormat.format(new Date(c.getTimeInMillis())).equals(firstday)) {// 如果01-01刚好是周日
			result[0] = result[0] + firstday + "|";
			result[1] = result[1] + firstday + "|";
		}
		c.add(Calendar.DAY_OF_YEAR, 1);
		d.set(Calendar.DAY_OF_WEEK, 7);// 设置周的最后一天
		d.add(Calendar.DAY_OF_YEAR, 1);
		String weekmonday = myFormat.format(new Date(c.getTimeInMillis()));// 得到第一周的周一
		String weeksunday = myFormat.format(new Date(d.getTimeInMillis()));// 得到第一周的周日
		if (weekmonday.compareTo(firstday) > 0)
			result[0] = result[0] + weekmonday + "|";
		else
			result[0] = result[0] + firstday + "|";
		result[1] = result[1] + weeksunday + "|";
		weekmonday = firstday;
		while (weekmonday.startsWith(String.valueOf(year))) {// 循环得到所有周的周一
			c.add(Calendar.DAY_OF_YEAR, 7);
			weekmonday = myFormat.format(new Date(c.getTimeInMillis()));
			if (!weekmonday.startsWith(String.valueOf(year)))
				break;
			result[0] = result[0] + weekmonday + "|";
		}
		while (weeksunday.startsWith(String.valueOf(year))) {// 循环得到所有周的周日
			d.add(Calendar.DAY_OF_YEAR, 7);
			weeksunday = myFormat.format(new Date(d.getTimeInMillis()));
			if (!weeksunday.startsWith(String.valueOf(year))) {
				result[1] = result[1] + lastday + "|";
				break;
			}
			result[1] = result[1] + weeksunday + "|";
			if (weeksunday.equals(lastday)) {
				break;
			}
		}
		result[0] = result[0].substring(0, result[0].length() - 1);
		result[1] = result[1].substring(0, result[1].length() - 1);
		return result;
	}

	/**
	 * 得到当前日期年月日字符串 result[0]:year result[1]:month result[2]:day
	 */
	public static String[] getYearMonthDay() {
		String[] result = new String[3];
		Calendar c = Calendar.getInstance();
		c.setTime(new Date());
		int month = c.get(Calendar.MONTH) + 1;
		int day = c.get(Calendar.DAY_OF_MONTH);
		if (month < 10) {
			result[1] = String.valueOf("0" + month);
		} else {
			result[1] = String.valueOf(month);
		}
		if (day < 10) {
			result[2] = String.valueOf("0" + day);
		} else {
			result[2] = String.valueOf(day);
		}
		result[0] = String.valueOf(c.get(Calendar.YEAR));
		return result;
	}

	/**
	 * 返回10位的int型当前时间
	 * 
	 * @return
	 */
	public static long getCurrTime() {
		return System.currentTimeMillis() / 1000;
	}


	/**
	 * 获得当前时间的指定相差月份的时间
	 * 
	 * @param differenceMonth
	 *            相差的月份，正数则往前推，负数则往后推
	 * @return
	 */
	public static String getPrevOrNextMonthTime(int differenceMonth) {
		String time = longFormat.format(getPrevOrNextMonthDate(differenceMonth));
		return time;
	}

	/**
	 * 获得当前时间的指定相差月份的时间
	 * @param differenceMonth
	 * @return
	 */
	public static Date getPrevOrNextMonthDate(int differenceMonth) {
		Calendar c = Calendar.getInstance();
		c.add(Calendar.MONTH, differenceMonth);
		return c.getTime();
	}

	/**
	 * 处理服务器时间（少了3位）
	 * 
	 * @param serverTime
	 * @return
	 */
	public static long fromServerTimeToTimeStamp(long serverTime) {
		return serverTime * 1000;
	}

	

	/**
	 * 比较时间大小
	 */
	public static int compare(Date date1, Date date2) {
		Calendar cal1 = Calendar.getInstance();
		Calendar cal2 = Calendar.getInstance();
		cal1.setTime(date1);
		cal2.setTime(date2);
		return cal2.getTime().compareTo(cal1.getTime());
	}

	public static int compare(Date date) {
		return compare(new Date(), date);
	}

	/**
	 * 时间差
	 */
	public static int yearDiff(Date date1, Date date2) {
		Calendar cal1 = Calendar.getInstance();
		Calendar cal2 = Calendar.getInstance();
		cal1.setTime(date1);
		cal2.setTime(date2);
		return cal2.get(Calendar.YEAR) - cal1.get(Calendar.YEAR);
	}

	/**
	 * 是否转月
	 * 
	 * @param date
	 * @return
	 */
	public static boolean isMonthDiff() {
		String nowMonth = formatDateByFormat(new Date(), "yyyy-MM");
		String yesterDayMonth = getYesterdayMonth();
		return !nowMonth.equals(yesterDayMonth);
	}
	
	/**
	 * 比较是否同一天
	 * @param time1
	 * @param time2
	 * @return
	 */
	public static boolean isSameDate(long time1,long time2){
		if (String.valueOf(time1).length() == 10) {
			time1 = time1 * 1000;
		}
		if (String.valueOf(time2).length() == 10) {
			time2 = time2 * 1000;
		}
		String time1Str = myFormat.format(new Date(time1));
		String time2Str = myFormat.format(new Date(time2));
		return time1Str.equals(time2Str);
	}
	
	/**
	 * 格式化日期（yyyy-MM-dd）
	 * @param time
	 * @return
	 */
	public static String formatLongToDate(long time){
		if (String.valueOf(time).length() == 10) {
			time = time * 1000;
		}
		return myFormat.format(new Date(time));
	}
	
	/**
	 * 字符串按格式转为Date 类型
	 * @param source
	 * @param pattern
	 * @return
	 * @throws ParseException
	 */
	public static Date toDate(String source, String pattern){
		try{
			SimpleDateFormat format = new SimpleDateFormat(pattern);
			return format.parse(source);
		}
		catch(ParseException ex){}
		return null;
		
	}
	
	/**
	 * Date转字String
	 * @param source
	 * @return
	 * @throws ParseException
	 */
	public static String toString(Date date, String pattern){
		return toString(date, pattern, null);
	}
	
	/**
	 * 转成字符串
	 * @param date
	 * @return
	 */
	public static String toString(Date date){
		Date now = new Date();
		long diff = (long) (Math.floor((now.getTime() - date.getTime()) / (60 * 1000)));
		if(diff < 1){
			return "刚刚";
		}
		else if(diff < 60){
			return diff + "分钟前";
		}
		else if(diff < 24 *60){
			int hours = (int) Math.floor(diff / 60);
			return hours + "小时前";
		}
		if(isSameYear(date, now)){
			formatDateByFormat(date, "MM-dd HH:mm");
		}
		return formatDateByFormat(date, "yy-MM-dd HH:mm");
	}
	

	
	/**
	 * Date 转字符串
	 * @param date
	 * @param pattern
	 * @param defaultValue
	 * @return
	 */
	public static String toString(Date date, String pattern, String defaultValue){
		if(date == null){
			return defaultValue;
		}
		SimpleDateFormat format = new SimpleDateFormat(pattern);
		return format.format(date);
	}
	
	public static Date isoToDate(String source) throws ParseException{
		String pattern = "yyyy-MM-dd'T'HH:mm:ss.SSSZ";
		if(source.endsWith("Z")){
			pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'";
		}
		return toDate(source, pattern);
	}
	
	public static String toIsoDate(Date date){
		String pattern = "yyyy-MM-dd'T'HH:mm:ss.SSSZ";
		return toString(date, pattern);
	}
	
	/**
	 * 获取当天日期
	 * @return
	 */
	public static String getTodayString(){
		return getDateString(new Date());
	}
	
	/**
	 * 获取日期字符串
	 * @param date
	 * @return
	 */
	public static String getDateString(Date date){
		SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
		return format.format(date);
	}
	
	public static String getLongDate(){
		SimpleDateFormat format = new SimpleDateFormat("yyyyMMddHHmmssSSS");
		return format.format(new Date());
	}
	
	/**
	 * 获取今天日期
	 */
	public static Date getToday(){
		Calendar calendar = Calendar.getInstance();
		calendar.set(Calendar.HOUR_OF_DAY, 0);
		calendar.set(Calendar.MINUTE, 0);
		calendar.set(Calendar.SECOND, 0);
		calendar.set(Calendar.MILLISECOND, 0);
		return calendar.getTime();
	}
	
	/**
	 * 获取今天日期
	 */
	public static int getDayOfWeek(){
		Calendar calendar = Calendar.getInstance();
		return calendar.get(Calendar.DAY_OF_WEEK);
	}
	
	
	/**
	 * 获取两个时间天数差
	 */
	public static int getDaysBetween(Date fromDate, Date toDate){
		return (int)((toDate.getTime() - fromDate.getTime())/(24 * 60 * 60 * 1000));
	}
	
	/**
	 * 获取向后时间
	 */
	public static Date getAfter(Date baseDate, int days){
		return after(baseDate, (24 * 60 * 60 * 1000L) * days);
	}
	
	/**
	 * 获取调整时间
	 * @param baseDate
	 * @param millsSeconds
	 * @return
	 */
	public static Date after(Date baseDate, long millsSeconds){
		Date date = new Date();
		date.setTime(baseDate.getTime() + millsSeconds);
		return date;
	}
	
	/**
	 * 根据小时数调整时间
	 * @param baseDate
	 * @param hours
	 * @return
	 */
	public static Date afterByHours(Date baseDate, int hours){
		return after(baseDate, hours * 60 * 60 * 1000);
	}
	
	/**
	 * 毫秒
	 * @param time
	 * @return
	 */
	public static long toMillsSeconds(String time){
		String[] arr = time.split(":");
		if(arr.length!=2){
			return 0L;
		}
		return (Integer.parseInt(arr[0]) * 60 +  Integer.parseInt(arr[1])) * 60 * 1000;
	}
	
	
	
	
	/**
	 * 获取当天运行毫秒数
	 */
	public static long getDayTime(){
		return new Date().getTime() - getToday().getTime();
	}
	
	/**
	 * 获取秒数
	 * @return
	 */
	public static int dateline(){
		return (int)(new Date().getTime()  / 1000); 
	}
	
	/**
	 * 获取短日期
	 * @return
	 */
	public static String getShortDate(){
		SimpleDateFormat sim=new SimpleDateFormat("yyyyMMdd");
		return sim.format(new Date());
	}
	
	/**
	 * 获取年龄
	 * @param birthday
	 * @return
	 */
	public static int getAge(String birthday){
		if(birthday != null && !"".equals(birthday)){
			try{
				Date date = DateUtils.toDate(birthday, "yyyy-MM-dd");
				Calendar calendar = Calendar.getInstance();
				calendar.setTime(date);
				int birthYear = calendar.get(Calendar.YEAR);
				int birthMonth = calendar.get(Calendar.MONTH);
				int birthDay = calendar.get(Calendar.DAY_OF_MONTH);
				calendar.setTime(new Date());
				int year = calendar.get(Calendar.YEAR);
				int month = calendar.get(Calendar.MONTH);
				int day = calendar.get(Calendar.DAY_OF_MONTH);
				
				int age = year - birthYear;
				if(month < birthMonth){
					age --;
				}
				else if(month == birthMonth && day < birthDay){
					age --;
				}
				if(age < 0){
					age = 0;
				}
				return age;
				
			}
			catch(Exception ex){
				ex.printStackTrace();
			}
		}
		return 0;
	}
	
	/**
	 * 获取时间段
	 * @param baseTime
	 * @param begin
	 * @param end
	 * @return
	 */
	public static TimeDuration getDuration(Date baseTime, String begin, String end){
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(baseTime);
		calendar.set(Calendar.SECOND, 0);
		calendar.set(Calendar.MILLISECOND, 0);
		
		String[] beginArr = begin.split(":");
		calendar.set(Calendar.HOUR_OF_DAY, Integer.parseInt(beginArr[0]));
		calendar.set(Calendar.MINUTE, Integer.parseInt(beginArr[1]));
		Date beginTime = calendar.getTime();
		
		String[] endArr = end.split(":");
		calendar.set(Calendar.HOUR_OF_DAY, Integer.parseInt(endArr[0]));
		calendar.set(Calendar.MINUTE, Integer.parseInt(endArr[1]));
		Date endTime = calendar.getTime();
		if(!endTime.after(beginTime)){
			endTime = getAfter(endTime, 1);
		}
		TimeDuration duration = new TimeDuration(beginTime, endTime);
		return duration;
	}
	
	/**
	 * 昨天日期
	 * @param pattern
	 * @return
	 */
	public static String getYesterdayString(String pattern){
		Date yesterday = getAfter(getToday(), -1);
		return toString(yesterday, pattern);
	}
	
	
	/**
	 * 是否同一年
	 * @param src
	 * @param dest
	 * @return
	 */
	public static boolean isSameYear(Date src, Date dest){
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(src);
		int srcYear = calendar.get(Calendar.YEAR);
		calendar.setTime(dest);
		int destYear = calendar.get(Calendar.YEAR);
		return srcYear == destYear;
	}

}
