package com.crawler.utils.http;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.ParseException;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.CookieStore;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.cookie.Cookie;
import org.apache.http.impl.client.BasicCookieStore;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;

import com.alibaba.fastjson.JSON;
import com.crawler.util.JsonMapper;
import com.crawler.utils.StringUtils;

/**
 * 简易http请求
 * @author rubekid
 * @date 2016年10月11日
 */
public class SimpleHttpClient {
	
	private static Logger logger = LoggerFactory.getLogger(SimpleHttpClient.class);

	private static CloseableHttpClient httpClient;
	
	private static CookieStore cookieStore;

	static {
		cookieStore  = new BasicCookieStore();

		// 将CookieStore设置到httpClient中
		httpClient = HttpClients.custom().setDefaultCookieStore(cookieStore).build();
	}
	
	public static String getCookie(String name){
		List<Cookie> cookies =  cookieStore.getCookies();
		for(Cookie cookie : cookies){
			if(cookie.getName().equalsIgnoreCase(name)){
				return cookie.getValue();
			}
		}
		return null;
		
	}
	
	public static String get(String url) throws ClientProtocolException, IOException{
		return get(url, null);
	}

	/**
	 * GET 请求
	 * @param url
	 * @return
	 * @throws ClientProtocolException
	 * @throws IOException
	 */
	public static String get(String url, Map<String, String> headers) throws ClientProtocolException, IOException {
		HttpGet httpGet = new HttpGet(url);
		if(headers != null){
			addHeaders(httpGet, headers);
		}
		HttpResponse httpResponse = httpClient.execute(httpGet);
		return EntityUtils.toString(httpResponse.getEntity());
	}

	/**
	 * POST 请求
	 * @param url
	 * @param params
	 * @param headers
	 * @return
	 * @throws ParseException
	 * @throws IOException
	 */
	public static String post(String url, Map<String, Object> params, Map<String, String> headers) throws ParseException, IOException{
		HttpPost httpPost = new HttpPost(url);
		UrlEncodedFormEntity postEntity = new UrlEncodedFormEntity(getParam(params), "UTF-8");
		httpPost.setEntity(postEntity);
		if(headers != null){
			addHeaders(httpPost, headers);
		}
		HttpResponse httpResponse = httpClient.execute(httpPost);
		return EntityUtils.toString(httpResponse.getEntity(), "UTF-8");
	}
	
	/**
	 * 获取数据并转成对象
	 * @param url
	 * @param clazz
	 * @return
	 * @throws IOException 
	 * @throws ClientProtocolException 
	 */
	public static <T> T getForObject(String url, Class<T> clazz){
		try{
			return toBean(get(url), clazz);
		}
		catch(IOException ex){
			logger.error(ex.getMessage(), ex);
		}
		return null;
	}

	/**
	 * 请求参数
	 * @param parameterMap
	 * @return
	 */
	private static List<NameValuePair> getParam(Map<String, Object> parameterMap) {
		List<NameValuePair> param = new ArrayList<NameValuePair>();
		for(Map.Entry<String, Object> entry : parameterMap.entrySet()){
			param.add(new BasicNameValuePair(entry.getKey(), String.valueOf(entry.getValue())));
		}
		return param;
	}
	
	/**
	 * 添加请求头部
	 * @param httpRequest
	 * @param headerMap
	 */
	private static void addHeaders(HttpRequestBase httpRequest, Map<String, String> headerMap){
		for(Map.Entry<String, String> entry : headerMap.entrySet()){
			httpRequest.addHeader(entry.getKey(), entry.getValue());
		}
	}
	
	/**
	 * 文本转换成BEAN
	 * @param response
	 * @param clazz
	 * @return
	 */
	private static <T> T toBean(String response, Class<T> clazz){
		MappingJackson2HttpMessageConverter messageConverter = new MappingJackson2HttpMessageConverter();
        messageConverter.setObjectMapper(JsonMapper.getMapper());
		return messageConverter.getObjectMapper().convertValue(JSON.parse(StringUtils.UrlDecode(response)), clazz);
	}

}
