package com.crawler.utils;

import com.crawler.beans.Coords;

/**
 * 坐标工具
 * WGS坐标系是国际通用的一种地心坐标系，WGS本身也有多个版本（不赘述了），GCJ-02是国内官方采用的一种坐标系，国内许多坐标系也是基于GCJ-02变种而来的，比如百度坐标系BD-09。
 * @author rubekid
 * @date 2016年10月27日
 */
public class CoordsUtils {
	
	/**
	 * 卫星椭球坐标投影到平面地图坐标系的投影因子。
	 */
	static double FACTOR = 6378245.0;
	/**
	 *  椭球的偏心率。
	 */
	static double EE = 0.00669342162296594323;
	
	/**
	 * 圆周率转换量。
	 */
	public final static double EXPI = Math.PI * 3000.0 / 180.0;
	
	/**
	 * 经度比率
	 */
	public final static double LONGITUDE_RATE = 102834.74258026089786013677476285;
	
	/**
	 * 维度比率
	 */
	public final static double LATITUDE_RATE = 111712.69150641055729984301412873;
	

	/**
	 * 计算距离
	 * @param coords1
	 * @param coords2
	 * @return
	 */
	public static long distance(Coords coords1, Coords coords2){
		if(coords1== null || coords2 == null){
			return 0;
		}
		return distance(coords1.getLongitude(), coords1.getLatitude(), coords2.getLongitude(), coords2.getLatitude());
	}
	
	/**
	 * 计算距离
	 * @param lon1
	 * @param lat1
	 * @param lon2
	 * @param lat2
	 * @return
	 */
	public static long distance(double lon1, double lat1, double lon2, double lat2){
        double b = Math.abs((lon1 - lon2) * LONGITUDE_RATE);
        double a = Math.abs((lat1 - lat2) * LATITUDE_RATE);
        return (long)Math.ceil(Math.sqrt(a * a + b * b));
	}
	
	


	public static double[] wgs2bd(double lat, double lon) {
	       double[] wgs2gcj = wgs2gcj(lat, lon);
	       double[] gcj2bd = gcj2bd(wgs2gcj[0], wgs2gcj[1]);
	       return gcj2bd;
	}

	public static double[] gcj2bd(double lat, double lon) {
	       double x = lon, y = lat;
	       double z = Math.sqrt(x * x + y * y) + 0.00002 * Math.sin(y * EXPI);
	       double theta = Math.atan2(y, x) + 0.000003 * Math.cos(x * EXPI);
	       double bd_lon = z * Math.cos(theta) + 0.0065;
	       double bd_lat = z * Math.sin(theta) + 0.006;
	       return new double[] { bd_lat, bd_lon };
	}

	public static double[] bd2gcj(double lat, double lon) {
	       double x = lon - 0.0065, y = lat - 0.006;
	       double z = Math.sqrt(x * x + y * y) - 0.00002 * Math.sin(y * EXPI);
	       double theta = Math.atan2(y, x) - 0.000003 * Math.cos(x * EXPI);
	       double gg_lon = z * Math.cos(theta);
	       double gg_lat = z * Math.sin(theta);
	       return new double[] { gg_lat, gg_lon };
	}

	public static double[] wgs2gcj(double lat, double lon) {
	       double dLat = transformLat(lon - 105.0, lat - 35.0);
	       double dLon = transformLon(lon - 105.0, lat - 35.0);
	       double radLat = lat / 180.0 * Math.PI;
	       double magic = Math.sin(radLat);
	       magic = 1 - EE * magic * magic;
	       double sqrtMagic = Math.sqrt(magic);
	       dLat = (dLat * 180.0) / ((FACTOR * (1 - EE)) / (magic * sqrtMagic) * Math.PI);
	       dLon = (dLon * 180.0) / (FACTOR / sqrtMagic * Math.cos(radLat) * Math.PI);
	       double mgLat = lat + dLat;
	       double mgLon = lon + dLon;
	       double[] loc = { mgLat, mgLon };
	       return loc;
	}

	private static double transformLat(double lat, double lon) {
	       double ret = -100.0 + 2.0 * lat + 3.0 * lon + 0.2 * lon * lon + 0.1 * lat * lon + 0.2 * Math.sqrt(Math.abs(lat));
	       ret += (20.0 * Math.sin(6.0 * lat * Math.PI) + 20.0 * Math.sin(2.0 * lat * Math.PI)) * 2.0 / 3.0;
	       ret += (20.0 * Math.sin(lon * Math.PI) + 40.0 * Math.sin(lon / 3.0 * Math.PI)) * 2.0 / 3.0;
	       ret += (160.0 * Math.sin(lon / 12.0 * Math.PI) + 320 * Math.sin(lon * Math.PI  / 30.0)) * 2.0 / 3.0;
	       return ret;
	}

	private static double transformLon(double lat, double lon) {
	       double ret = 300.0 + lat + 2.0 * lon + 0.1 * lat * lat + 0.1 * lat * lon + 0.1 * Math.sqrt(Math.abs(lat));
	       ret += (20.0 * Math.sin(6.0 * lat * Math.PI) + 20.0 * Math.sin(2.0 * lat * Math.PI)) * 2.0 / 3.0;
	       ret += (20.0 * Math.sin(lat * Math.PI) + 40.0 * Math.sin(lat / 3.0 * Math.PI)) * 2.0 / 3.0;
	       ret += (150.0 * Math.sin(lat / 12.0 * Math.PI) + 300.0 * Math.sin(lat / 30.0 * Math.PI)) * 2.0 / 3.0;
	       return ret;
	}
}