package com.crawler.utils;

import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Random;

import org.springframework.security.authentication.encoding.Md5PasswordEncoder;
import org.springframework.security.crypto.codec.Base64;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;

/**
 * 字符串工具
 * 
 * @author rubekid
 *
 */
public class StringUtils {

	private static final String CHARS = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";

	/**
	 * 获取指定长度随机字符串
	 * 
	 * @param length
	 * @return
	 */
	public static String getRandomString(int length) {
		StringBuffer sb = new StringBuffer();
		Random random = new Random();
		for (int i = 0; i < length; i++) {
			sb.append(CHARS.charAt(random.nextInt(CHARS.length())));
		}
		return sb.toString();
	}

	/**
	 * 获取随机数字字符串
	 * 
	 * @param length
	 * @return
	 */
	public static String getRandomNumString(int length) {
		if (length <= 0) {
			return "";
		}

		char[] numbersAndLetters = ("0123456789").toCharArray();
		Random random = new Random();
		char[] randomStr = new char[length];
		for (int i = 0; i < length; i++) {
			randomStr[i] = numbersAndLetters[random.nextInt(numbersAndLetters.length)];
		}
		return String.valueOf(randomStr);

	}

	/**
	 * Md5 加密
	 * 
	 * @param password
	 * @param salt
	 * @return
	 */
	public static String md5(String password, Object salt) {
		Md5PasswordEncoder encoder = new Md5PasswordEncoder();
		return encoder.encodePassword(password, salt);
	}
	
	/**
	 * Md5 加密
	 * 
	 * @param password
	 * @param salt
	 * @return
	 */
	public static String md5(String password) {
		return md5(password, null);
	}

	/**
	 * 16位MD5
	 * 
	 * @param password
	 * @return
	 */
	public static String shortMd5(String password) {
		String md5 = md5(password);
		return md5.substring(8, 24);
	}
	
	
	/**
	 * 生成Salt
	 * 
	 * @return
	 */
	public static String generateSalt() {
		return getRandomString(8);
	}
	
	/**
	 * 密码加密
	 * @param password
	 * @param salt
	 * @return
	 */
	public  static  String passwordMd5(String password, Object salt){
		return  md5(md5(password, salt), salt);
	}
	
	/**
	 * SHA1
	 * @param text
	 * @return
	 */
	public static String SHA1(String text) {
		try {
			MessageDigest digest = java.security.MessageDigest
					.getInstance("SHA-1");
			digest.update(text.getBytes());
			byte messageDigest[] = digest.digest();
			// Create Hex String
			StringBuffer hexString = new StringBuffer();
			// 字节数组转换为 十六进制 数
			for (int i = 0; i < messageDigest.length; i++) {
				String shaHex = Integer.toHexString(messageDigest[i] & 0xFF);
				if (shaHex.length() < 2) {
					hexString.append(0);
				}
				hexString.append(shaHex);
			}
			return hexString.toString();

		} catch (NoSuchAlgorithmException e) {
			e.printStackTrace();
		}
		return "";
	}


	/**
	 * 判断字符串是否为空
	 * 
	 * @param string
	 * @return
	 */
	public static boolean isNullOrEmpty(String string) {
		return string == null || "".equals(string);
	}
	
	/**
	 * 字符串组
	 * @param groups
	 * @return
	 */
	public static String groupString(String groups){
		if(groups == null){
			return "";
		}
		return "," + groups.replaceAll("^,|,$", "") + ",";
	}
	
	/**
	 * 获取非空字符串
	 */
	public static String firstNotNull(String ... strings){
		for(String string  : strings){
			if(!isNullOrEmpty(string)){
				return string;
			}
		}
		return "";
	}
	
	/**
	 * 获取首字母
	 */
	public static String  getInitial(String string){
		if(isNullOrEmpty(string)){
			return "#";
		}
		String initial = string.substring(0, 1).toUpperCase();
		if(!initial.matches("[A-Z]")){
			initial = "#";
		}
		return initial;
	}
	
	/**
	 * 获取基本文件名
	 * @param path
	 * @return
	 */
	public static String getBaseFileName(String path){
		int pos = path.lastIndexOf("/");
		if(pos > -1){
			path = path.substring(pos + 1);
		}
		pos = path.lastIndexOf(".");
		if(pos > -1){
			path = path.substring(0, pos);
		}
		return path;
	}
	
	/**
	 * base64
	 * @param str
	 * @return
	 */
	public static String base64(String str){
		new Base64();
		byte[] bytes = Base64.encode(str.getBytes());
		return new String(bytes);
	}

	/**
	 * 转码
	 * @param str
	 * @return
	 */
	public static String TextDecode(String str){

		if(str!=null && !Charset.forName("GBK").newEncoder().canEncode(str)){
			
			try {
				str = new String(str.getBytes("ISO-8859-1"), "utf-8");
			} catch (UnsupportedEncodingException e) {
				e.printStackTrace();
			}
		}
		
		return str;
	}
	
	/**
	 * 转码
	 * @param str
	 * @return
	 */
	public static String UrlDecode(String str){
		return TextDecode(str);
	}
	
	/**
	 * 解密方法函数
	 * 
	 * @param data
	 * @return
	 * @throws UnsupportedEncodingException
	 */
	public static String decodeFun(String data) throws UnsupportedEncodingException {
		return new String(Base64.decode(data.getBytes()), "UTF-8");
	}

	/**
	 * 加密方法函数
	 * 
	 * @param data
	 * @return
	 * @throws UnsupportedEncodingException 
	 */
	public static String encodeFun(byte[] data) throws UnsupportedEncodingException {
		return new String(Base64.encode(data), "UTF-8");
	}
	
	/**
	 * 解密用
	 * 
	 * @param str
	 * @return
	 * @throws Exception
	 */
	public static String secretDecode(String str){
		try{
			str = URLDecoder.decode(decodeFun(URLDecoder.decode(str, "UTF-8")), "UTF-8");
			char[] result = new char[str.length()];
			for (int i = 0; i < str.length(); i++) {
				int ascII = ((int) str.charAt(i));
				if (ascII < 12) {
					ascII += 128;
				}
				result[i] = (char) (ascII - 12);
			}
			return String.valueOf(result);
		
		}
		catch(UnsupportedEncodingException ex){
			return str;
		}
	}

	/**
	 * 加密
	 * 
	 * @param str
	 * @return
	 * @throws Exception
	 */
	public static String secretEncode(String str){
		try{
			char[] result = new char[str.length()];
			for (int i = 0; i < str.length(); i++) {
				int ascII = ((int) str.charAt(i));
				if (ascII > 116) {
					ascII -= 128;
				}
				result[i] = (char) (ascII + 12);
			}
			str = String.valueOf(result);
			return URLEncoder.encode(encodeFun(URLEncoder.encode(str, "UTF-8").getBytes("UTF-8")), "UTF-8");
		}
		catch(UnsupportedEncodingException ex){
			return str;
		}
	}
	
	/**
	 * 距离文本
	 * @param distance
	 * @return
	 */
	public static String getDistanceText(long distance){
		if(distance < 20){
			return "20m以内";
		}
		else if(distance < 1000){
			return distance + "m";
		}
		else{
			return ((double)Math.round((double)distance / 100)) / 10 + "km"; 
		}
	}
	
	/**
	 * 获取性别名称
	 * @param gender
	 * @return
	 */
	public static String getGenderText(int gender){
		if(gender == 1){
			return "男";
		}
		else{
			return "女";
		}
	}
	
	/**
	 * 字符串截取
	 * @param text
	 * @param len
	 * @return
	 */
	public static String substr(String text, int len){
		if(text.length() > len){
			text = text.substring(0, len - 2) + "...";
		}
		return text;
	}
	
	/**
	 * 格式化数据
	 * @param number
	 * @param length
	 * @param prefix
	 * @return
	 */
	public static String formatNumber(Number number, Integer length, String prefix){
		
		if(number == null){
			return "";
		}
		long no = number.longValue();
		int len = length == null ?  3 : length;

		double max = Math.pow(10D, (double)len);
		String result = String.valueOf(no);
		if(no < max){
			no = (long) (max + no);
			result = String.valueOf(no);
			result = result.substring(result.length() - len);
		}
		if(prefix !=null){
			result  = prefix + result;
		}
		return result;
	}
	
	/**
	 * 格式化数据
	 * @param number
	 * @param length
	 * @return
	 */
	public static String formatNumber(Number number, Integer length){
		return formatNumber(number, length, null);
	}
	
	/**
	 * 数字字符串
	 * @param string
	 * @return
	 */
	public static boolean isNumeric(String string){
		return string.matches("\\d*");
	}
	
	/**
	 * 列表转字符串数组
	 * @param list
	 * @return
	 */
	public static final <E> String[] toStrings(List<E> list){
		String[] items = new String[list.size()];
		for(int i=0; i<list.size(); i++){
			items[i] = JSONObject.toJSONString(list.get(i));
		}
		return items;
	}
	
	
	/**
	 * 拼接字符串
	 * @param list
	 * @param separator
	 * @return
	 */
	public static final String join(List<String> list, String separator){
		StringBuffer stringBuffer = new StringBuffer();
		for(String str : list){
			stringBuffer.append(str);
			stringBuffer.append(separator);
		}
		String result = stringBuffer.toString();
		if(!"".equals(separator)){
			result = result.replaceAll(separator + "$", "");
		}
		return result;
	}
	
	/**
	 * 字符串转对象
	 * @param value
	 * @param type
	 * @return
	 */
	public static final Object toObject(String value, String type){
		if("int".equalsIgnoreCase(type)){
			return Integer.valueOf(value);
		}
		else if("long".equalsIgnoreCase(type)){
			return Long.valueOf(value);
		}
		else if("float".equalsIgnoreCase(type)){
			return Float.valueOf(value);
		}
		else if("double".equalsIgnoreCase(type)){
			return Double.valueOf(value);
		}
		else if("boolean".equalsIgnoreCase(type)){
			return Boolean.valueOf(value);
		}
		else if("json".equalsIgnoreCase(type)){
			return JSONObject.parse(value);
		}
		else if("array".equalsIgnoreCase(type)){
			return JSONArray.parse(value);
		}
		else if("decimal".equalsIgnoreCase(value)){
			return new BigDecimal(value);
		}
		return value;
	}
	
	/**
	 * 字符串转
	 * @param str
	 * @param regex
	 * @return
	 */
	public static final List<String> split(String str, String regex){
		if(isNullOrEmpty(str)){
			return new ArrayList<String>();
		}
		String[] array = str.split(regex);
		return Arrays.asList(array);
	}
	
	/**
	 * 字符串转列表
	 * @param str
	 * @param regex
	 * @return
	 */
	public static final List<Integer> split2Ingeter(String str, String regex){
		List<Integer> items = new ArrayList<Integer>();
		if(!isNullOrEmpty(str)){
			String[] array = str.split(regex);
			for(String string : array){
				items.add(Integer.valueOf(string));
			}
		}
		return items;
	}
	
	
	/**
	 * 生成订单号
	 * @return
	 */
	public static String uniqueNo(String prev){
		return prev + DateUtils.getLongDate() + StringUtils.getRandomNumString(3);
	}
	
	/**
	 * 生成唯一单号
	 * @return
	 */
	public static String uniqueId(){
		return uniqueNo("");
	}

}
