package com.crawler.uc.utils;


import java.util.Date;
import java.util.Random;

import com.crawler.waf.config.WafProperties;
import com.crawler.waf.security.authens.OauthAccessToken;
import com.crawler.waf.support.Constants;
import com.crawler.waf.utils.Encrypt;
import com.crawler.waf.utils.StringUtil;

/**
 * 用户中心模拟器
 * @author Rubekid
 *
 * 2017年5月15日 上午11:02:21
 */
public class UcSimulator {

	/**
	 * token保存前缀
	 */
	private static final String UC_TOKEN = "uc:token";
	
	/**
	 * 用户前缀
	 */
	private static final String UC_USER = "uc:user";
	
	/**
	 * ACCESS_TOKEN 前缀
	 */
	private static final String ACCESS_TOKEN = "access";
	
	/**
	 * 映射 前缀
	 */
	private static final String MAPPING = "mapping";
	
	/**
	 * REFRESH_TOKEN 前缀
	 */
	private static final String REFRESH_TOKEN = "refresh";
	
	
	/**
	 * 授权码
	 */
	private static final String AUTH_CODE = "code";
	
	/**
	 * 授权码有效期
	 */
	public static final int CODE_EXPIRE_MINUTE = WafProperties.getPropertyForInteger("code_expire_minute", "5");
	
	/**
	 * 过期时间 分钟
	 */
	private static final int TOKEN_TIMEOUT = WafProperties.getPropertyForInteger("token.timeout", "7200");
	
	/**
	 * 保存授权码
	 * @param code
	 */
	public static void saveCode(String code, Long userId){
		int seconds = 60 * CODE_EXPIRE_MINUTE;
		UcRedisManager.set(codeKey(code), userId, seconds);
	}
	
	/**
	 * 获取授权码
	 * @param code
	 */
	public static Long findCode(String code){
		return UcRedisManager.get(codeKey(code), Long.class);
	}
	
	/**
	 * 移除授权码
	 * @param code
	 */
	public static void removeCode(String code){
		UcRedisManager.del(codeKey(code));
	}
	
	/**
	 * 生成token
	 * @param userId
	 * @param type
	 * @return
	 */
	public static OauthAccessToken generateBearerToken(){
		String tokenUserId = String.valueOf(0L);
		OauthAccessToken oauthAccessToken = new OauthAccessToken();
		String accessToken = generateToken(tokenUserId);
		String refreshToken = generateToken(tokenUserId);
		Date expiresAt = new Date(System.currentTimeMillis() + TOKEN_TIMEOUT * 60 * 1000);
		oauthAccessToken.setAccessToken(accessToken);
		oauthAccessToken.setRefreshToken(refreshToken);
		oauthAccessToken.setExpiresAt(expiresAt);
		oauthAccessToken.setUserId(tokenUserId);
		oauthAccessToken.setServerTime(new Date());
		UcSimulator.set(Constants.AUTHORIZATION_TYPE_BEARER, oauthAccessToken);
		return oauthAccessToken;
	}
	

	
	/**
	 * 设置token
	 * @param type
	 * @param oauthAccessToken
	 */
	public static void set(String type, OauthAccessToken oauthAccessToken){
		Long userId = Long.parseLong(oauthAccessToken.getUserId());
		int seconds = (int) ((oauthAccessToken.getExpiresAt().getTime() - System.currentTimeMillis()) / 1000);
		
		//永不过期返回前端过期时间为100年后
		if(seconds == 0){
			oauthAccessToken.setExpiresAt(new Date(System.currentTimeMillis() + 100L * 365 * 24 * 60 * 60 * 1000));
		}
		UcRedisManager.set(accessTokenKey(type, oauthAccessToken.getAccessToken()), oauthAccessToken, seconds);
		UcRedisManager.set(refreshTokenKey(type, oauthAccessToken.getRefreshToken()), oauthAccessToken.getAccessToken(), seconds);
		UcRedisManager.set(mappingKey(type, userId), oauthAccessToken.getAccessToken(), seconds);
	}
	
	/**
	 * 获取token
	 * @param type
	 * @param accessToken
	 * @return
	 */
	public static OauthAccessToken get(String type, String accessToken){
		return UcRedisManager.get(accessTokenKey(type, accessToken), OauthAccessToken.class);
	}
	
	/**
	 * 通过refresh获取Token
	 * @param type
	 * @param refreshToken
	 * @return
	 */
	public static OauthAccessToken findByRefreshToken(String type, String refreshToken){
		String accessToken = UcRedisManager.get(refreshTokenKey(type, refreshToken));
		if(StringUtil.isNullOrEmpty(accessToken)){
			return null;
		}
		return get(type, accessToken);
	}
	
	/**
	 * 删除
	 * @param type
	 * @param accessToken
	 */
	public static void remove(String type, String accessToken){
		OauthAccessToken oauthAccessToken = get(type, accessToken);
		UcRedisManager.del(accessTokenKey(type, accessToken));
		if(oauthAccessToken != null){
			UcRedisManager.del(refreshTokenKey(type, oauthAccessToken.getRefreshToken()));			
		}
	}

	
	/**
	 * 用户 key
	 * @param type
	 * @param userId
	 * @return
	 */
	private static String userKey(Long userId){
		return String.format("%s:%s", UC_USER, userId);
	}
	
	/**
	 * Mapping key
	 * @param type
	 * @param userId
	 * @return
	 */
	private static String mappingKey(String type, Long userId){
		return String.format("%s:%s:%s:%s", UC_TOKEN, type, MAPPING, userId);
	}
	
	/**
	 * Access Token key
	 * @param type
	 * @param accessToken
	 * @return
	 */
	private static String accessTokenKey(String type, String accessToken){
		return String.format("%s:%s:%s:%s", UC_TOKEN, type, ACCESS_TOKEN, accessToken);
	}
	
	/**
	 * Refresh Token Key
	 * @param type
	 * @param refreshToken
	 * @return
	 */
	private static String refreshTokenKey(String type, String refreshToken){
		return String.format("%s:%s:%s:%s", UC_TOKEN, type, REFRESH_TOKEN , refreshToken);
	}
	
	/**
	 * 授权码 key	
	 * @param code
	 * @return
	 */
	private static String codeKey(String code){
		return String.format("%s:%s:%s", UC_TOKEN, AUTH_CODE, code);
	}
	

	/**
	 * 生成token
	 * @return
	 */
	private static String generateToken(String userId){
		return (Encrypt.shortMd5(userId) + Encrypt.SHA256(getRandomString(8))).toUpperCase();
	}

	/**
	 * 获取指定长度随机字符串
	 * 
	 * @param length
	 * @return
	 */
	private static String getRandomString(int length) {
		String CHARS = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
		StringBuffer sb = new StringBuffer();
		Random random = new Random();
		for (int i = 0; i < length; i++) {
			sb.append(CHARS.charAt(random.nextInt(CHARS.length())));
		}
		return sb.toString();
	}
}
