package com.liuwa.unicloud.http;


import com.liuwa.http.HttpRequest;
import com.liuwa.http.HttpResponse;
import com.liuwa.unicloud.common.UniCloudConfig;
import com.liuwa.unicloud.common.UniCloudProperties;
import com.liuwa.unicloud.exception.UniCloudException;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import java.net.HttpURLConnection;
import java.util.*;

/**
 * 阿里云
 */
public class AliyunHttpClient extends CloudHttpClient{

    private static final String API_URL  = "https://api.bspapp.com/client";
    private static final String KEY_MAC = "HmacMD5";
    private static final String CACHE_KEY = ":aliyn:accesstoken:";


    /**
     * 客户端密钥
     */
    private String clientSecret;

    public AliyunHttpClient(){
        this(UniCloudConfig.APPID, UniCloudConfig.ALIYUN_SPACE_ID, UniCloudConfig.ALIYUN_CLIENT_SECRET);
    }

    public AliyunHttpClient(String appId, String spaceId, String clientSecret){
        super();
        this.userAgent = USER_AGENT;
        this.os = OS;
        this.appId = appId;
        this.spaceId = spaceId;
        this.clientSecret = clientSecret;
    }



    /**
     * 请求
     * @param url
     * @param config
     * @return
     */
    @Override
    public JSONObject request(String url, LinkedHashMap<String, Object> config){
        String accessToken = getAccessToken();
        if(accessToken == null || "".equals(accessToken)){
            throw new UniCloudException("access_token 为空");
        }
        String uuid = UUID.randomUUID().toString().replaceAll("-", "");

        LinkedHashMap<String, Object> clientInfo = new LinkedHashMap<String, Object>();
        clientInfo.put("PLATFORM", "h5");
        clientInfo.put("OS", os);
        clientInfo.put("APPID", appId);
        clientInfo.put("CLIENT_SDK_VERSION", "1.0.0");

        LinkedHashMap<String, Object> uniCloudClientInfo = new LinkedHashMap<String, Object>();
        uniCloudClientInfo.put("ak", appId);
        uniCloudClientInfo.put("p", "i");
        uniCloudClientInfo.put("ut", "h5");
        uniCloudClientInfo.put("uuid", uuid);
        uniCloudClientInfo.put("fn", "http");
        uniCloudClientInfo.put("sid", spaceId);
        uniCloudClientInfo.put("pvd", "a");

        LinkedHashMap<String, Object> functionArgs = new LinkedHashMap<String, Object>();
        functionArgs.put("url", url);
        functionArgs.put("config", config);
        functionArgs.put("clientInfo", clientInfo);
        functionArgs.put("uniCloudClientInfo", urlEncode(JSONObject.fromObject(uniCloudClientInfo).toString()));

        LinkedHashMap<String, Object> params = new LinkedHashMap<String, Object>();
        params.put("functionTarget", "http");
        params.put("functionArgs", functionArgs);


        SortedMap<String, Object> form = new TreeMap<String, Object>();
        form.put("method", "serverless.function.runtime.invoke");
        form.put("params", JSONObject.fromObject(params).toString());
        form.put("spaceId", spaceId);
        form.put("timestamp", System.currentTimeMillis() );
        form.put("token", accessToken);

        String sign = sign(form);


        Map<String, String> headers = new HashMap<String, String>();
        headers.put("user-agent", userAgent);
        headers.put("content-type", "application/json");
        headers.put("x-basement-token", accessToken);
        headers.put("x-serverless-sign", sign);

        HttpRequest httpRequest = new HttpRequest();
        HttpResponse response = httpRequest.post(API_URL, JSONObject.fromObject(form).toString(), headers);
        JSONObject json = JSONObject.fromObject(response.getContent());
        json.put("url", url);
        return json;
    }


    /**
     * 签名
     * @param form
     * @return
     */
    public String sign(SortedMap<String, Object> form){
        StringBuffer sb = new StringBuffer();
        for(Map.Entry<String, Object> entry: form.entrySet()){
            sb.append(entry.getKey()).append("=").append(String.valueOf(entry.getValue())).append("&");
        }
        String data = sb.toString().replaceAll("&$", "");
        return HmacMD5(data, clientSecret);
    }

    /**
     * 解析成 HttpResponse
     * @param json
     * @return
     */
    @Override
    public HttpResponse parse(JSONObject json){
        logger.debug("响应内容：{}", json.toString());
        HttpResponse httpResponse = new HttpResponse();
        httpResponse.setUrl(json.getString("url"));
        if(json.getBoolean("success")){
            JSONObject data = json.getJSONObject("data");
            String content = data.getString("data");
            int status = data.getInt("status");
            JSONObject respHeaders = data.getJSONObject("headers");
            httpResponse.setStatusCode(status);
            httpResponse.setContent(content);
            httpResponse.setBytes(toBytes(content));


            Map<String, List<String>> headers = new HashMap<String, List<String>>();
            if(respHeaders != null){

                if(respHeaders.containsKey("location")) {
                    httpResponse.setUrl(respHeaders.getString("location"));
                    logger.debug("{} 跳转：{}", status, httpResponse.getUrl());
                }
                Iterator<String> iterator = respHeaders.keys();
                while(iterator.hasNext()){
                    String key = iterator.next();
                    Object value = respHeaders.get(key);
                    List<String> list = new ArrayList<String>();
                    if(value instanceof JSONArray){
                        JSONArray array = (JSONArray) value;
                        for(int i=0; i<array.size(); i++){
                            list.add(array.getString(i));
                        }
                    }
                    else{
                        list.add(respHeaders.getString(key));
                    }
                    headers.put(key, list);
                }
            }



            httpResponse.setHeaders(headers);
        }
        else{
            httpResponse.setStatusCode(HttpURLConnection.HTTP_INTERNAL_ERROR);
        }
        return httpResponse;
    }

    /**
     * 获取token
     * @return
     */
    public String getAccessToken(){
        String cacheKey = CACHE_KEY + uid;
        String accessToken = getCache(cacheKey);
        if(accessToken != null && !"".equals(accessToken)){
            return accessToken;
        }
        SortedMap<String, Object> form = new TreeMap<String, Object>();
        form.put("method", "serverless.auth.user.anonymousAuthorize");
        form.put("params", "{}");
        form.put("spaceId", spaceId);
        form.put("timestamp", System.currentTimeMillis() );

        String sign = sign(form);

        Map<String, String> headers = new HashMap<String, String>();
        headers.put("user-agent", userAgent);
        headers.put("content-type", "application/json");
        headers.put("x-serverless-sign", sign);

        HttpRequest httpRequest = new HttpRequest();
        HttpResponse response = httpRequest.post(API_URL, JSONObject.fromObject(form).toString(), headers);
        JSONObject json = JSONObject.fromObject(response.getContent());
        if(json.getBoolean("success")){
            accessToken = json.getJSONObject("data").getString("accessToken");
            int expiresInSecond = json.getJSONObject("data").getInt("expiresInSecond");
            setCache(cacheKey, accessToken, expiresInSecond);

        }
        return accessToken;
    }


    /**
     * HmacMD5
     * @param data
     * @param key
     * @return
     */
    private String HmacMD5(String data, String key){
        byte[] inputData = data.getBytes();
        SecretKey secretKey = new SecretKeySpec(key.getBytes(), KEY_MAC);
        Mac mac;
        try {
            mac = Mac.getInstance(secretKey.getAlgorithm());
            mac.init(secretKey);
            return byteArrayToHexString(mac.doFinal(inputData));
        } catch (Exception e) {
            logger.error("HmacMD5算法加密失败",e);
        }
        return "";
    }

    /*byte数组转换为HexString*/
    private String byteArrayToHexString(byte[] b) {
        StringBuffer sb = new StringBuffer(b.length * 2);
        for (int i = 0; i < b.length; i++) {
            int v = b[i] & 0xff;
            if (v < 16) {
                sb.append('0');
            }
            sb.append(Integer.toHexString(v));
        }
        return sb.toString();
    }


}