package com.liuwa.unicloud.http;

import com.liuwa.http.HttpResponse;
import com.liuwa.unicloud.cache.LocalStorage;
import net.sf.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.UUID;

/**
 * 云请求
 */
public abstract class CloudHttpClient {

    protected Logger logger = LoggerFactory.getLogger(getClass());

    public static final String USER_AGENT = "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/86.0.4240.183 Safari/537.36";

    public static final String OS = "other";

    /**
     * 响应数据格式
     */
    public static final String DATA_TYPE_JSON = "json";
    public static final String DATA_TYPE_TEXT = "text";

    /**
     * 请求方法
     */
    public static final String METHOD_GET = "GET";
    public static final String METHOD_POST = "POST";
    public static final String METHOD_PUT = "PUT";
    public static final String METHOD_DELETE = "DELETE";

    /**
     * 用户ID （用来映射登录信息）
     */
    protected String uid;

    /**
     * Uni APP ID
     */
    protected String appId;

    /**
     * User-Agent
     */
    protected String userAgent;

    /**
     * 系统
     */
    protected String os;

    /**
     * 空间ID
     */
    protected String spaceId;

    public CloudHttpClient(){
        uid = UUID.randomUUID().toString();
    }

    /**
     * 获取缓存
     * @param cacheKey
     * @return
     */
    public String getCache(String cacheKey){
        return LocalStorage.get(cacheKey);
    }

    /**
     * 设置缓存
     * @param cacheKey
     * @param value
     * @param expiresInSecond
     */
    public void setCache(String cacheKey, String value, int expiresInSecond){
        LocalStorage.set(cacheKey, value, expiresInSecond);
    }

    /**
     * get 请求
     * @param url
     * @param headers
     * @return
     */
    public JSONObject getForJSON(String url, Map<String, String> headers, String dataType){
        return request(METHOD_GET, url, headers,null, dataType);
    }

    /**
     * get 请求
     * @param url
     * @param headers
     * @return
     */
    public JSONObject getForJSON(String url, Map<String, String> headers, Map<String, Object> data,String dataType){
        return request(METHOD_GET, url, headers, data, dataType);
    }

    /**
     * post 请求
     * @param url
     * @param headers
     * @return
     */
    public JSONObject postForJSON(String url, Map<String, String> headers, Map<String, Object> data,String dataType){
        return request(METHOD_POST, url, headers, data, dataType);
    }


    /**
     * get 请求
     * @param url
     * @return
     */
    public JSONObject getForJSON(String url){
        return request(METHOD_GET, url, null, null, DATA_TYPE_TEXT);
    }



    /**
     * get 请求
     * @param url
     * @return
     */
    public HttpResponse get(String url){
        return get(url, null);
    }

    /**
     * get请求
     * @param url
     * @param headers
     * @return
     */
    public HttpResponse get(String url, Map<String, String> headers){
        JSONObject json = getForJSON(url, headers, DATA_TYPE_TEXT);
        return parse(json);
    }


    /**
     * get请求
     * @param url
     * @param
     * @param headers
     * @return
     */
    public HttpResponse get(String url, Map<String, String> headers, Map<String, Object> data){
        JSONObject json = getForJSON(url, headers, data, DATA_TYPE_TEXT);
        return parse(json);
    }

    /**
     * post请求
     * @param url
     * @param
     * @param headers
     * @return
     */
    public HttpResponse post(String url, Map<String, String> headers, Map<String, Object> data){
        JSONObject json = postForJSON(url, headers, data, DATA_TYPE_JSON);
        return parse(json);
    }

    /**
     * 请求
     * @param method
     * @param url
     * @param data
     * @param dataType
     * @return
     */
    public JSONObject request(String method, String url, Map<String, Object> data, String dataType){
        return request(method, url, null, data, dataType);
    }




    /**
     * 请求
     * @param method
     * @param url
     * @param headers
     * @param data
     * @param dataType
     * @return
     */
    public JSONObject request(String method, String url, Map<String, String> headers, Map<String, Object> data, String dataType){
        LinkedHashMap<String, Object> config = new LinkedHashMap<String, Object>();
        config.put("method", method);
        if(headers != null){
            config.put("headers", headers);
        }
        if(data != null){
            config.put("data", data);
        }
        config.put("dataType", dataType);

        return request(url, config);
    }

    /**
     * 请求
     * @param url
     * @param config
     * @return
     */
    public abstract JSONObject request(String url, LinkedHashMap<String, Object> config);
    /**
     * 解析数据
     * @param json
     * @return
     */
    public abstract HttpResponse parse(JSONObject json);

    /**
     * URL 编码
     * @param text
     * @return
     */
    public String urlEncode(String text){
        return urlEncode(text, "utf-8");
    }

    /**
     * URL 编码
     * @param text
     * @return
     */
    public String urlEncode(String text, String charset){
        try{
            return URLEncoder.encode(text, charset);
        }
        catch (UnsupportedEncodingException ex){
            logger.error(ex.getMessage(), ex);
        }
        return text;
    }

    /**
     * 转换成bytes 失败
     * @param text
     * @return
     */
    protected byte[] toBytes(String text){
        try{
            return text.getBytes("UTF-8");
        }
        catch (IOException ex){
            logger.error(ex.getMessage(), ex);
        }
        return null;
    }

    public String getUid() {
        return uid;
    }

    public void setUid(String uid) {
        this.uid = uid;
    }

    public String getAppId() {
        return appId;
    }

    public void setAppId(String appId) {
        this.appId = appId;
    }

    public String getUserAgent() {
        return userAgent;
    }

    public void setUserAgent(String userAgent) {
        this.userAgent = userAgent;
    }

    public String getOs() {
        return os;
    }

    public void setOs(String os) {
        this.os = os;
    }

    public String getSpaceId() {
        return spaceId;
    }

    public void setSpaceId(String spaceId) {
        this.spaceId = spaceId;
    }
}
