package com.liuwa.unicloud.http;


import com.liuwa.http.HttpRequest;
import com.liuwa.http.HttpResponse;
import com.liuwa.unicloud.common.UniCloudConfig;
import com.liuwa.unicloud.common.UniCloudProperties;
import com.liuwa.unicloud.exception.UniCloudException;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

import java.net.HttpURLConnection;
import java.util.*;

/**
 * 腾讯云
 */
public class TencentCloudHttpClient extends CloudHttpClient{

    private static final String API_BASE_URL  = "http://tcb-api.tencentcloudapi.com/web";
    private static final String CACHE_KEY = ":tencentcloud:accesstoken:";

    /**
     * 客户端sdk版本
     */
    private static final String CLIENT_SDK_VERSION = "1.0.0";


    /**
     * 请求地址
     */
    private String apiUrl;

    /**
     * 客户端数据版本
     */
    private String dataVersion;


    /**
     * 构造函数
     */
    public TencentCloudHttpClient(){
        this(UniCloudConfig.APPID, UniCloudConfig.TENCENT_SPACE_ID, UniCloudConfig.TENCENT_DATA_VERSION);
    }

    /**
     * 构造函数
     * @param appId
     * @param spaceId
     * @param dataVersion
     */
    public TencentCloudHttpClient(String appId, String spaceId, String dataVersion){
        super();
        this.apiUrl = API_BASE_URL + "?env=" + spaceId;
        this.userAgent = USER_AGENT;
        this.os = OS;
        this.appId = appId;
        this.spaceId = spaceId;
        this.dataVersion = dataVersion;
    }


    /**
     * 请求
     * @param url
     * @param config
     * @return
     */
    @Override
    public JSONObject request(String url, LinkedHashMap<String, Object> config){
        String accessToken = getAccessToken();
        if(accessToken == null || "".equals(accessToken)){
            throw new UniCloudException("access_token 为空");
        }
        String uuid = UUID.randomUUID().toString().replaceAll("-", "");

        String seqId = generateSeqId();

        LinkedHashMap<String, Object> clientInfo = new LinkedHashMap<String, Object>();
        clientInfo.put("PLATFORM", "h5");
        clientInfo.put("OS", os);
        clientInfo.put("APPID", appId);
        clientInfo.put("CLIENT_SDK_VERSION", CLIENT_SDK_VERSION);

        LinkedHashMap<String, Object> uniCloudClientInfo = new LinkedHashMap<String, Object>();
        uniCloudClientInfo.put("ak", appId);
        uniCloudClientInfo.put("p", "i");
        uniCloudClientInfo.put("ut", "h5");
        uniCloudClientInfo.put("uuid", uuid);
        uniCloudClientInfo.put("fn", "http");
        uniCloudClientInfo.put("sid", spaceId);
        uniCloudClientInfo.put("pvd", "t");

        LinkedHashMap<String, Object> functionArgs = new LinkedHashMap<String, Object>();
        functionArgs.put("url", url);
        functionArgs.put("config", config);
        functionArgs.put("clientInfo", clientInfo);
        functionArgs.put("uniCloudClientInfo", urlEncode(JSONObject.fromObject(uniCloudClientInfo).toString()));



        SortedMap<String, Object> form = new TreeMap<String, Object>();
        form.put("access_token", accessToken);
        form.put("action", "functions.invokeFunction");
        form.put("dataVersion", dataVersion);
        form.put("env", spaceId);
        form.put("function_name", "http");
        form.put("request_data", JSONObject.fromObject(functionArgs).toString() );
        form.put("seqId", seqId);

        Map<String, String> headers = new HashMap<String, String>();
        headers.put("user-agent", userAgent);
        headers.put("content-type", "application/json;charset=UTF-8");
        headers.put("X-SDK-Version", "1.3.5");
        headers.put("x-seqid", seqId);

        HttpRequest httpRequest = new HttpRequest();
        HttpResponse response = httpRequest.post(apiUrl, JSONObject.fromObject(form).toString(), headers);
        JSONObject json = JSONObject.fromObject(response.getContent());
        json.put("url", url);
        return json;
    }




    /**
     * 解析成 HttpResponse
     * @param json
     * @return
     */
    @Override
    public HttpResponse parse(JSONObject json){
        logger.debug("响应内容：{}", json.toString());
        HttpResponse httpResponse = new HttpResponse();
        httpResponse.setUrl(json.getString("url"));
        if(json.containsKey("data")){
            JSONObject data = json.getJSONObject("data").getJSONObject("response_data");
            String content = data.getString("data");
            int status = data.getInt("status");
            JSONObject respHeaders = data.getJSONObject("headers");
            httpResponse.setStatusCode(status);
            httpResponse.setContent(content);
            httpResponse.setBytes(toBytes(content));


            Map<String, List<String>> headers = new HashMap<String, List<String>>();
            if(respHeaders != null){

                if(respHeaders.containsKey("location")) {
                    httpResponse.setUrl(respHeaders.getString("location"));
                    logger.debug("{} 跳转：{}", status, httpResponse.getUrl());
                }
                Iterator<String> iterator = respHeaders.keys();
                while(iterator.hasNext()){
                    String key = iterator.next();
                    Object value = respHeaders.get(key);
                    List<String> list = new ArrayList<String>();
                    if(value instanceof JSONArray){
                        JSONArray array = (JSONArray) value;
                        for(int i=0; i<array.size(); i++){
                            list.add(array.getString(i));
                        }
                    }
                    else{
                        list.add(respHeaders.getString(key));
                    }
                    headers.put(key, list);
                }
            }

            httpResponse.setHeaders(headers);
        }
        else{
            httpResponse.setStatusCode(HttpURLConnection.HTTP_INTERNAL_ERROR);
        }
        return httpResponse;
    }

    /**
     * 获取token
     * @return
     */
    public String getAccessToken(){
        String cacheKey = CACHE_KEY + uid;
        String accessToken = getCache(cacheKey);
        if(accessToken != null && !"".equals(accessToken)){
            return accessToken;
        }

        String seqId = generateSeqId();

        SortedMap<String, Object> form = new TreeMap<String, Object>();
        form.put("action", "auth.signInAnonymously");
        form.put("dataVersion", dataVersion);
        form.put("env", spaceId);
        form.put("seqId", seqId);

        Map<String, String> headers = new HashMap<String, String>();
        headers.put("user-agent", USER_AGENT);
        headers.put("content-type", "application/json;charset=UTF-8");
        headers.put("X-SDK-Version", "1.3.5");
        headers.put("x-seqid", seqId);

        HttpRequest httpRequest = new HttpRequest();
        HttpResponse response = httpRequest.post(apiUrl, JSONObject.fromObject(form).toString(), headers);
        JSONObject json = JSONObject.fromObject(response.getContent());
        if(!json.containsKey("refresh_token")){
            return null;
        }
        String refreshToken = json.getString("refresh_token");
        String uuid = json.getString("uuid");
        seqId = generateSeqId();
        form = new TreeMap<String, Object>();
        form.put("action", "auth.getJwt");
        form.put("anonymous_uuid", uuid);
        form.put("dataVersion", dataVersion);
        form.put("env", spaceId);
        form.put("refresh_token", refreshToken);
        form.put("seqId", seqId );

        response = httpRequest.post(apiUrl, JSONObject.fromObject(form).toString(), headers);
        json = JSONObject.fromObject(response.getContent());
        if(!json.containsKey("access_token")){
            return null;
        }
        accessToken = json.getString("access_token");
        int expiresInSecond = json.getInt("access_token_expire");
        setCache(cacheKey, accessToken, expiresInSecond);
        return accessToken;
    }


    /**
     * 生成 13 随机字符串 seq ID
     * @return
     */
    private String generateSeqId(){
        return UUID.randomUUID().toString().replaceAll("-", "").substring(0, 13);
    }
}