package com.crawler.push.common;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.UUID;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.crawler.push.bean.PushBody;
import com.crawler.push.config.PushProperties;
import com.gexin.rp.sdk.base.IPushResult;
import com.gexin.rp.sdk.base.ITemplate;
import com.gexin.rp.sdk.base.impl.AppMessage;
import com.gexin.rp.sdk.base.impl.SingleMessage;
import com.gexin.rp.sdk.base.impl.Target;
import com.gexin.rp.sdk.base.payload.APNPayload;
import com.gexin.rp.sdk.base.uitls.AppConditions;
import com.gexin.rp.sdk.exceptions.RequestException;
import com.gexin.rp.sdk.http.IGtPush;
import com.gexin.rp.sdk.template.LinkTemplate;
import com.gexin.rp.sdk.template.NotificationTemplate;
import com.gexin.rp.sdk.template.TransmissionTemplate;
import com.gexin.rp.sdk.template.style.Style0;

/**
 *  个推推送类
 * @author rubekid
 *
 */
public class GePusher implements Pushable{
    
	
	protected static final Logger LOG = LoggerFactory.getLogger(GePusher.class);
	
    private static String APP_ID = PushProperties.getProperty("getui.appId");
    private static String APP_KEY = PushProperties.getProperty("getui.appKey");
    private static String MASTER_SECRET = PushProperties.getProperty("getui.masterSecret");
	
	static String HOST = PushProperties.getProperty("getui.apiUrl", "http://sdk.open.api.igexin.com/apiex.htm");

	private String appId;
	
	private String appKey;
	
	private String masterSecret;
    
	/**
	 * 构造方法
	 */
	public GePusher(){
		this.appId = APP_ID;
		this.appKey = APP_KEY;
		this.masterSecret = MASTER_SECRET;
	}
	
	/**
	 * 构造方法
	 * @param appKey
	 * @param masterSecret
	 */
	public GePusher(String appId,String appKey, String masterSecret){
		this.appId = appId;
		this.appKey = appKey;
		this.masterSecret = masterSecret;
	}

    
	
	/**
	 * 别名推送
	 * @param alias
	 * @param pushBody
	 */
	@Override
	public void pushToAlias(String alias, PushBody pushBody){
		List<Object> aliasList = new ArrayList<Object>();
		aliasList.add(alias);
		pushToAlias(aliasList, pushBody);
	}
	
	/**
	 * 别名推送
	 * @param alias
	 * @param pushBody
	 */
	@Override
	public void pushToAlias(List<Object> alias, PushBody pushBody){
		List<String> aliasList = new ArrayList<String>();
		for(Object o : alias){
			aliasList.add(String.valueOf(o));
		}
		pushBody.setMsgId(UUID.randomUUID().toString());
		pushBody.setDate(System.currentTimeMillis());
		if(pushBody.getTitle() == null){
			pushBody.setTitle("系统通知");
		}
		if(pushBody.getIcon() == null){			
			pushBody.setIcon(PushBody.PUSH_ICON);
		}
		if(pushBody.getExtra() == null){
			pushBody.setExtra(new HashMap<String, Object>());
		}
		
		sendPushByAlias(aliasList, pushBody);
	}
	
	
	/**
	 * 发送给别名列表
	 * @param alias
	 * @param title
	 * @param content
	 */
	@Override
	public void sendPushByAlias(List<String> aliasList, PushBody pushBody){
		
		NotificationTemplate template = buildNotificationTemplate(pushBody);
		for(String cid : aliasList) {
			pushMessageToSingle(cid, template);			
		}

		
		
	}
	
	/**
	 * 发送给tag分组
	 * @param alias
	 * @param pushBody
	 */
	@Override
	public void pushToTags(List<Object> tags, PushBody pushBody){
		List<String> tagList = new ArrayList<String>();
		for(Object o : tags){
			tagList.add(String.valueOf(o));
		}
		pushBody.setMsgId(UUID.randomUUID().toString());
		pushBody.setDate(System.currentTimeMillis());
		if(pushBody.getTitle() == null){
			pushBody.setTitle("系统通知");
		}
		if(pushBody.getIcon() == null){			
			pushBody.setIcon(PushBody.PUSH_ICON);
		}
		if(pushBody.getExtra() == null){
			pushBody.setExtra(new HashMap<String, Object>());
		}
		
		sendPushByTags(tagList, pushBody);
	}
	
	/**
	 * 发送给tag分组列表
	 * @param tags
	 * @param title
	 * @param content
	 */
	@Override
	public void sendPushByTags(List<String> tags, PushBody pushBody){
		//TODO
	}
	
	/**
	 * 发送通知
	 * @param pushMsg
	 */
    @Override
	public void notification(PushBody pushBody) {
    	NotificationTemplate template = buildNotificationTemplate(pushBody);
    	pushMessageToApp(template);
	}
	
    /**
     * 创建通知模板
     * @param pushBody
     * @return
     */
    public NotificationTemplate buildNotificationTemplate(PushBody pushBody) {
    	NotificationTemplate  template = new NotificationTemplate();
        // 设置APPID与APPKEY
        template.setAppId(appId);
        template.setAppkey(appKey);

        Style0 style = new Style0();
        // 设置通知栏标题与内容
        style.setTitle(pushBody.getTitle());
        style.setText(pushBody.getContent());
        // 配置通知栏图标
        style.setLogo(pushBody.getIcon());
        // 配置通知栏网络图标
        style.setLogoUrl("");
        // 设置通知是否响铃，震动，或者可清除
        style.setRing(true);
        style.setVibrate(true);
        style.setClearable(true);
        template.setStyle(style);
        template.setTransmissionContent(pushBody.toString());
        return template;
    }
    
    /**
     * 创建连接推送
     * @param pushBody
     * @return
     */
    public LinkTemplate buildLinkTemplate(PushBody pushBody) {
    	LinkTemplate  template = new LinkTemplate();
        // 设置APPID与APPKEY
        template.setAppId(appId);
        template.setAppkey(appKey);

        Style0 style = new Style0();
        // 设置通知栏标题与内容
        style.setTitle(pushBody.getTitle());
        style.setText(pushBody.getContent());
        // 配置通知栏图标
        style.setLogo(pushBody.getIcon());
        // 配置通知栏网络图标
        style.setLogoUrl("");
        // 设置通知是否响铃，震动，或者可清除
        style.setRing(true);
        style.setVibrate(true);
        style.setClearable(true);
        template.setStyle(style);
        
        template.setUrl(pushBody.getLink());
        return template;
    }
    
    /**
     * 透传消息
     * @param pushBody
     * @return
     */
    public TransmissionTemplate buildTransmissionTemplate(PushBody pushBody) {
        TransmissionTemplate template = new TransmissionTemplate();
        template.setAppId(appId);
        template.setAppkey(appKey);
        template.setTransmissionContent(pushBody.toString());
        template.setTransmissionType(2);
        APNPayload payload = new APNPayload();
        //在已有数字基础上加1显示，设置为-1时，在已有数字上减1显示，设置为数字时，显示指定数字
        payload.setAutoBadge("+1");
        payload.setContentAvailable(1);
        //ios 12.0 以上可以使用 Dictionary 类型的 sound
        payload.addCustomMsg("payload", pushBody.toString());

        //简单模式APNPayload.SimpleMsg
        payload.setAlertMsg(new APNPayload.SimpleAlertMsg(pushBody.getTitle()));

        //字典模式使用APNPayload.DictionaryAlertMsg
        payload.setAlertMsg(getDictionaryAlertMsg(pushBody));

        //设置语音播报类型，int类型，0.不可用 1.播放body 2.播放自定义文本
        payload.setVoicePlayType(0);
        //设置语音播报内容，String类型，非必须参数，用户自定义播放内容，仅在voicePlayMessage=2时生效
        //注：当"定义类型"=2, "定义内容"为空时则忽略不播放
        payload.setVoicePlayMessage(pushBody.getTitle());

      
        template.setAPNInfo(payload);
        return template;
    }
    
    private static APNPayload.DictionaryAlertMsg getDictionaryAlertMsg(PushBody pushBody){
        APNPayload.DictionaryAlertMsg alertMsg = new APNPayload.DictionaryAlertMsg();
        alertMsg.setBody(pushBody.getContent());
        alertMsg.setActionLocKey("ActionLockey");
        alertMsg.setLocKey("LocKey");
        alertMsg.addLocArg("loc-args");
        alertMsg.setLaunchImage("launch-image");
        // iOS8.2以上版本支持
        alertMsg.setTitle(pushBody.getTitle());
        alertMsg.setTitleLocKey("TitleLocKey");
        alertMsg.addTitleLocArg("TitleLocArg");
        return alertMsg;
    }
    
    /**
     * 个人推送
     * @param cid
     * @param template
     */
    private void pushMessageToSingle(String cid, ITemplate template) {
    	IGtPush push = new IGtPush(HOST, appKey, masterSecret);
        
        SingleMessage message = new SingleMessage();
        template.getPushInfo();
        message.setOffline(true);
        // 离线有效时间，单位为毫秒，可选
        message.setOfflineExpireTime(24 * 3600 * 1000);
        message.setData(template);
        // 可选，1为wifi，0为不限制网络环境。根据手机处于的网络情况，决定是否下发
        message.setPushNetWorkType(0);
        Target target = new Target();
        target.setAppId(appId);
        target.setClientId(cid);
        //target.setAlias(Alias);
        IPushResult ret = null;
        
        try {
            ret = push.pushMessageToSingle(message, target);
        } catch (RequestException e) {
            e.printStackTrace();
            ret = push.pushMessageToSingle(message, target, e.getRequestId());
        }
        if (ret != null) {
            System.out.println(ret.getResponse().toString());
        } else {
            System.out.println("服务器响应异常");
        }
    }
    
    /**
     * 广播
     * @param cid
     * @param template
     */
    private void pushMessageToApp(ITemplate template) {
    	IGtPush push = new IGtPush(HOST, appKey, masterSecret);
        
    	AppMessage message = new AppMessage();
        message.setData(template);

        message.setOffline(true);
        //离线有效时间，单位为毫秒，可选
        message.setOfflineExpireTime(24 * 1000 * 3600);
        //推送给App的目标用户需要满足的条件
        AppConditions cdt = new AppConditions();
        List<String> appIdList = new ArrayList<String>();
        appIdList.add(appId);
        message.setAppIdList(appIdList);
        //手机类型
        List<String> phoneTypeList = new ArrayList<String>();
        //省份
        List<String> provinceList = new ArrayList<String>();
        //自定义tag
        List<String> tagList = new ArrayList<String>();

        cdt.addCondition(AppConditions.PHONE_TYPE, phoneTypeList);
        cdt.addCondition(AppConditions.REGION, provinceList);
        cdt.addCondition(AppConditions.TAG,tagList);
        message.setConditions(cdt);
        
        IPushResult ret = push.pushMessageToApp(message);
        System.out.println(ret.getResponse().toString());
    }
}

