package com.crawler.push.common;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.UUID;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.crawler.push.bean.PushBody;
import com.crawler.push.config.PushProperties;

import cn.jpush.api.JPushClient;
import cn.jpush.api.common.resp.APIConnectionException;
import cn.jpush.api.common.resp.APIRequestException;
import cn.jpush.api.push.PushResult;
import cn.jpush.api.push.model.Message;
import cn.jpush.api.push.model.Platform;
import cn.jpush.api.push.model.PushPayload;
import cn.jpush.api.push.model.audience.Audience;

/**
 * 极光推送类
 * @author rubekid
 *
 */
public class JPusher implements Pushable{
    
	
	protected static final Logger LOG = LoggerFactory.getLogger(JPusher.class);
	
	/**
	 * 默认秘钥
	 */
	private static final String APP_KEY = PushProperties.getProperty("jpush.appKey");
	private static final String MASTER_SECRET = PushProperties.getProperty("jpush.masterSecret");
	

	
	
	private String appKey;
	
	private String masterSecret;
    
	/**
	 * 构造方法
	 */
	public JPusher(){
		this.appKey = APP_KEY;
		this.masterSecret = MASTER_SECRET;
	}
	
	/**
	 * 构造方法
	 * @param appKey
	 * @param masterSecret
	 */
	public JPusher(String appKey, String masterSecret){
		this.appKey = appKey;
		this.masterSecret = masterSecret;
	}

    
	
	/**
	 * 别名推送
	 * @param alias
	 * @param pushBody
	 */
	@Override
	public void pushToAlias(String alias, PushBody pushBody){
		List<Object> aliasList = new ArrayList<Object>();
		aliasList.add(alias);
		pushToAlias(aliasList, pushBody);
	}
	
	/**
	 * 别名推送
	 * @param alias
	 * @param pushBody
	 */
	@Override
	public void pushToAlias(List<Object> alias, PushBody pushBody){
		List<String> aliasList = new ArrayList<String>();
		for(Object o : alias){
			aliasList.add(String.valueOf(o));
		}
		pushBody.setMsgId(UUID.randomUUID().toString());
		pushBody.setDate(System.currentTimeMillis());
		if(pushBody.getTitle() == null){
			pushBody.setTitle("系统通知");
		}
		if(pushBody.getIcon() == null){			
			pushBody.setIcon(PushBody.PUSH_ICON);
		}
		if(pushBody.getExtra() == null){
			pushBody.setExtra(new HashMap<String, Object>());
		}
		
		sendPushByAlias(aliasList, pushBody);
	}
	
	
	/**
	 * 发送给别名列表
	 * @param alias
	 * @param title
	 * @param content
	 */
	@Override
	public void sendPushByAlias(List<String> aliasList, PushBody pushBody){
		JPushClient jpushClient = new JPushClient(masterSecret, appKey, 5);
		//指定推送对象
		
        PushPayload payload = buildPushObjectByAlias(aliasList, pushBody.toString());
        
        try {
        	PushResult result = jpushClient.sendPush(payload);
            
            LOG.info("Got result - " + result);
            
        } catch (APIConnectionException e) {
            LOG.error("Connection error. Should retry later. ", e);
            
        } catch (APIRequestException e) {
            LOG.error("Error response from JPush server. Should review and fix it. ", e);
            LOG.info("HTTP Status: " + e.getStatus());
            LOG.info("Error Code: " + e.getErrorCode());
            LOG.info("Error Message: " + e.getErrorMessage());
            LOG.info("Msg ID: " + e.getMsgId());
        } 
	}
	
	/**
	 * 发送给tag分组
	 * @param alias
	 * @param pushBody
	 */
	@Override
	public void pushToTags(List<Object> tags, PushBody pushBody){
		List<String> tagList = new ArrayList<String>();
		for(Object o : tags){
			tagList.add(String.valueOf(o));
		}
		pushBody.setMsgId(UUID.randomUUID().toString());
		pushBody.setDate(System.currentTimeMillis());
		if(pushBody.getTitle() == null){
			pushBody.setTitle("系统通知");
		}
		if(pushBody.getIcon() == null){			
			pushBody.setIcon(PushBody.PUSH_ICON);
		}
		if(pushBody.getExtra() == null){
			pushBody.setExtra(new HashMap<String, Object>());
		}
		
		sendPushByTags(tagList, pushBody);
	}
	
	/**
	 * 发送给tag分组列表
	 * @param tags
	 * @param title
	 * @param content
	 */
	@Override
	public void sendPushByTags(List<String> tags, PushBody pushBody){
		JPushClient jpushClient = new JPushClient(masterSecret, appKey, 5);
		//指定推送对象
		 PushPayload payload = buildPushObjectByAlias(tags, pushBody.toString());
        
        try {
            PushResult result = jpushClient.sendPush(payload);
            LOG.info("Got result - " + result);
            
        } catch (APIConnectionException e) {
            LOG.error("Connection error. Should retry later. ", e);
            
        } catch (APIRequestException e) {
            LOG.error("Error response from JPush server. Should review and fix it. ", e);
            LOG.info("HTTP Status: " + e.getStatus());
            LOG.info("Error Code: " + e.getErrorCode());
            LOG.info("Error Message: " + e.getErrorMessage());
            LOG.info("Msg ID: " + e.getMsgId());
        } 
	}
	
	/**
	 * 发送通知
	 * @param pushMsg
	 */
    @Override
	public void notification(PushBody pushBody) {
		JPushClient jpushClient = new JPushClient(masterSecret, appKey, 5);

		PushPayload payload = bulidNotification(pushBody.toString());
		try {
			PushResult result = jpushClient.sendPush(payload);

			LOG.info("Got result - " + result);
		} catch (APIConnectionException e) {
			LOG.error("Connection error. Should retry later. ", e);
		} catch (APIRequestException e) {
			LOG.error("Error response from JPush server. Should review and fix it. ", e);
			LOG.info("HTTP Status: " + e.getStatus());
			LOG.info("Error Code: " + e.getErrorCode());
			LOG.info("Error Message: " + e.getErrorMessage());
			LOG.info("Msg ID: " + e.getMsgId());
		}
	}
	

	/**
	 * 所有平台，指定目标别名推送
	 */
    private PushPayload buildPushObjectByAlias(List<String> alias,String msgContent) {
        return PushPayload.newBuilder()
        		.setPlatform(Platform.all())//设置接受的平台
                .setAudience(Audience.alias(alias))
                .setMessage(Message.content(msgContent)) //自定义消息
                .build();
    }
    

    /**
     * 双平台，指定tag设备，通知为msg
     */
    private PushPayload buildPushObjectByTags(List<String> tags, String msgContent) {
        return PushPayload.newBuilder()
                .setPlatform(Platform.android_ios())
                .setAudience(Audience.tag(tags))
                .setMessage(Message.content(msgContent)) //自定义消息
                .build();
    }
    
    
	
	/**
	 * 创建通知
	 * @param msgContent
	 * @return
	 */
	private PushPayload bulidNotification(String msgContent) {
		return PushPayload.newBuilder().setPlatform(Platform.android_ios()).setAudience(Audience.all())
				.setMessage(Message.newBuilder().setMsgContent(msgContent).build()).build();
	}
}

