package com.crawler.redis.utils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.poi.ss.formula.functions.T;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.crawler.redis.config.JacksonObjectMapper;
import com.crawler.redis.exception.SimpleRuntimeException;
import com.crawler.redis.vo.RedisPage;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;
import redis.clients.jedis.JedisPubSub;
import redis.clients.jedis.exceptions.JedisException;

/**
 * redis 管理器
 * @author rubekid
 *
 * 2016年12月17日 下午10:19:25
 */
public class RedisManager {
	
	/**
	 * 日志
	 */
	private static Logger logger = LoggerFactory.getLogger(RedisManager.class);
	
	/**
	 * 等待超时毫秒数
	 */
	private static int WAIT_TIMEOUT_MSECS = 60000;
	
	/**
	 * 锁自动过期毫秒数
	 */
	private static int LOCK_EXPIRE_MSECS = 60000;
	
	/**
	 * 
	 * jedis池
	 */
	private static JedisPool pool;
	
	/**
	 * 数据库
	 */
	private static int DB_INDEX = 0;


	 /**
	  * 静态代码初始化池配置
	  */
	static {
		try {
		
			Properties props = new Properties();
			props.load(RedisManager.class.getClassLoader().getResourceAsStream("redis.properties"));
			JedisPoolConfig config = new JedisPoolConfig();
			config.setMaxTotal(Integer.valueOf(props.getProperty("redis.pool.maxTotal")));
			config.setMaxIdle(Integer.valueOf(props.getProperty("redis.pool.maxIdle")));
			config.setMaxWaitMillis(Long.valueOf(props.getProperty("redis.pool.maxWaitMillis")));
			config.setTestOnBorrow(Boolean.valueOf(props.getProperty("redis.pool.testOnBorrow")));
			config.setTestOnReturn(Boolean.valueOf(props.getProperty("redis.pool.testOnReturn")));
			config.setTestWhileIdle(Boolean.valueOf(props.getProperty("redis.pool.testWhileIdle")));
			
			String host = props.getProperty("redis.host");
			Integer port = Integer.valueOf(props.getProperty("redis.port"));
			Integer timeout = Integer.valueOf(props.getProperty("redis.timeout"));
			String password = props.getProperty("redis.password");
			
			DB_INDEX = Integer.valueOf(props.getProperty("redis.dbindex", "0"));
			
			System.out.println("---------------- REDIS CONFIG ------------ "
					+ "\nHOST:" + host 
					+ "\nPORT:" + port 
					+ "\nPASSWORD:" + password
					+ "\nINDEX:" + DB_INDEX);
			
			if(password !=null){
				pool = new JedisPool(config, host, port, timeout, password);
			}
			else{
				pool = new JedisPool(config, host, port, timeout);
			}
			System.out.println("=====================REDIS 初始化=======================\nHOST:" + host + "\nINDEX:" + DB_INDEX);

		} catch (IOException ex) {
			logger.error(ex.getMessage(), ex);
		}
	}

	
	/**
	 * 获取线程池信息
	 * @return
	 */
	public static Map<String, Object> getPoolInfo(){
		Map<String, Object> map = new HashMap<String, Object>();
		map.put("maxBorrowWaitTimeMillis", pool.getMaxBorrowWaitTimeMillis());
		map.put("meanBorrowWaitTimeMillis", pool.getMeanBorrowWaitTimeMillis());
		map.put("active", pool.getNumActive());
		map.put("idle", pool.getNumIdle());
		map.put("waiters", pool.getNumWaiters());
		return map;
	}
	
	/**
	 * 获取db index
	 * @return
	 */
	public static int getDbIndex(){
		return DB_INDEX;
	}
	
	/**
	 * 同步锁
	 * @param lockKey
	 * @param waitTimeout
	 * @return
	 */
	public static boolean lock(String lockKey, Integer waitTimeout){
		long start = System.currentTimeMillis();
		Jedis jedis = null;
		try{
			jedis = getJedis();
			if(waitTimeout == null){
				waitTimeout = WAIT_TIMEOUT_MSECS;
			}
	        while (waitTimeout >= 0) {
	            long time = System.currentTimeMillis() + LOCK_EXPIRE_MSECS + 1;
	            String expireAt = String.valueOf(time);
	            if (jedis.setnx(lockKey, expireAt) == 1){
	                return true;
	            }

	            String currentValue = jedis.get(lockKey); 
	            if (currentValue != null && Long.parseLong(currentValue) < System.currentTimeMillis()) {

	                String oldValue = jedis.getSet(lockKey, expireAt);
	                //获取上一个锁到期时间，并设置现在的锁到期时间，
	                //只有一个线程才能获取上一个线上的设置时间，因为jedis.getSet是同步的
	                if(oldValue != null && oldValue.equals(currentValue)) {
	                    //如过这个时候，多个线程恰好都到了这里，但是只有一个线程的设置值和当前值相同，他才有权利获取锁
	                    return true;
	                }
	            }
	            waitTimeout -= 100;
	            try {
	            	if(waitTimeout >= 0){	            		
	            		Thread.sleep(100);
	            	}
				} catch (InterruptedException ex) {
					logger.error(ex.getMessage(), ex);
				}
	        }
	        return false;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
			logger.info("获取redis锁时间：" + (System.currentTimeMillis() - start));
		}
		
	}
	
	/**
	 * 同步锁
	 * @param key
	 */
	public static boolean lock(String lockKey){
		return lock(lockKey, null);
	}
	
	/**
	 * 解除锁定
	 * @param key
	 */
	public static void unlock(String lockKey){
		Jedis jedis = null;
		try{
			jedis = getJedis();
	        jedis.del(lockKey);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 尝试获取锁
	 * @param lockKey
	 * @param seconds
	 * @return
	 */
	public static boolean tryLock(String lockKey, int seconds){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			boolean bool = jedis.setnx(lockKey, ".lock") == 1;
			// 获取成功设置过期时间
			if(bool) {
				jedis.expire(lockKey, seconds);
			}
			return bool;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 设置过期时间
	 * @param key
	 * @param seconds
	 * @return
	 */
	public static boolean expire(String key, int seconds) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.expire(key, seconds) == 1;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 设置过期时间
	 * @param key
	 * @param seconds
	 * @return
	 */
	public static boolean expireAt(String key, long timestamp) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.expireAt(key, timestamp) == 1;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 获取剩余过期时间 -2 不存在 -1 没有设置过期时间 
	 * @param key
	 * @param seconds
	 * @return
	 */
	public static long ttl(String key) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.ttl(key);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}

	/**
	 * 移除过期设置
	 * @param key
	 * @param seconds
	 * @return
	 */
	public static boolean persist(String key) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.persist(key)  == 1;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 批量删除
	 * @param jedis
	 * @param prefix
	 */
	public static void batchDeleteByKey(String prefix){
		Jedis jedis = null;
		try{
			jedis = getJedis();
	        Set<String> set = jedis.keys(prefix + "*");  
	        Iterator<String> it = set.iterator();
	        List<String> keys = new ArrayList<String>();
	        while(it.hasNext()){
	        	keys.add(it.next());
	        	if(keys.size() > 0 && keys.size() % 100 == 0){
	        		jedis.del(keys.toArray(new String[keys.size()]));
	        		keys.clear();
	        	}
	        	
	        }   
	        if(keys.size() > 0){
	        	jedis.del(keys.toArray(new String[keys.size()]));  
	        }
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}

	/**
     * 获取Jedis实例
     * 
     * @return
     */
    public synchronized static Jedis getJedis() {
    	Jedis resource = null;
        try {
            if (pool != null) {
                resource = pool.getResource();
                resource.select(DB_INDEX);
            }
        } catch (RuntimeException ex) {
        	 if(resource != null ) {
        		 System.out.println("=========== Jedis 获取 resource 失败 ============");
                 resource.close();
             }
            logger.error(ex.getMessage(), ex);
        }
        if(resource== null){        	
        	throw new NullPointerException("jedis is null, please check the redis server.");
        }
        return resource;
    }
	
	/**
	 * 关闭
	 * @param jedis
	 */
	public static void close(Jedis jedis) {
		jedis.close();
	}
	
	/**
	 * 通过key获取
	 * @param key
	 * @return
	 */
	public static String get(String key, String defaultValue){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			String value = jedis.get(key);
			return value == null ? defaultValue : value;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 通过key获取
	 * @param key
	 * @return
	 */
	public static String get(String key){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			String value = jedis.get(key);
			return value;
		}
		catch(JedisException ex){
			logger.error(ex.getMessage(), ex);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
		return null;
	}
	
	/**
	 * 保存
	 * @param key
	 * @param value
	 */
	public static void set(String key, String value){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.set(key, value);
		}
		catch(JedisException ex){
			logger.error(ex.getMessage(), ex);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	

    /**
     * 设置键的字符串值(指定超时)
     * @param key
     * @param data
     * @param seconds
     */
    public static void set(String key, Object data, int seconds){
    	String value = null;
    	if(data instanceof String){
    		value = String.valueOf(data);
    	}
    	else{
    		value = toJSONString(data);
    	}
    	Jedis jedis = null;
		try{
			jedis  = getJedis();
			jedis.setex(key, seconds, value);
		}
		catch(JedisException ex){
			logger.error(ex.getMessage(), ex);
		}
		finally{
			close(jedis);
		}
    }
	
	/**
	 * 根据key 获取对象
	 * @param key
	 * @param clazz
	 * @return
	 */
	public static <T> T get(String key, Class<T> clazz){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			String value = jedis.get(key);
			if(isNullOrEmpty(value)){
				return null;
			}
			return parse(value, clazz);
		}
		catch(JedisException ex){
			logger.error(ex.getMessage(), ex);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
		return null;
		 
	}
	
	/**
	 * 根据key 获取对象
	 * @param index
	 * @param key
	 * @param clazz
	 * @return
	 */
	public static <T> T get(int index, String key, Class<T> clazz){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			jedis.select(index);
			String value = jedis.get(key);
			if(isNullOrEmpty(value)){
				return null;
			}
			return parse(value, clazz);
		}
		catch(JedisException ex){
			logger.error(ex.getMessage(), ex);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
		return null;
	}
	
	/**
	 * 根据key 获取对象
	 * @param index
	 * @param key
	 * @return
	 */
	public static String get(int index, String key){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			jedis.select(index);
			return jedis.get(key);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 根据key 获取列表
	 * @param key
	 * @param clazz
	 * @return
	 */
	public static <T> List<T> findList(String key, Class<T> clazz, String ... exceptKeys){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			List<String> data = new ArrayList<String>();
			if(key.contains("*")){
				Set<String> keys = jedis.keys(key);
				List<String> exceptKeyList = Arrays.asList(exceptKeys);
				for(String k : keys){
					if(exceptKeyList.contains(k)){
						continue;
					}
					data.addAll(jedis.lrange(k, 0, -1));
				}
			}
			else{
				data = jedis.lrange(key, 0, -1);
			}

			List<T> list = new ArrayList<T>();
			for(String string : data){
				list.add(parse(string, clazz));
			}
			return list;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
		
	}
	
	/**
	 * 获取列表
	 * @param key
	 * @return
	 */
	public static List<String> findList(String key){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			List<String> data = new ArrayList<String>();
			data = jedis.lrange(key, 0, -1);
			return data;
			
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
		
	}
	
	/**
	 * 分页
	 * @param page
	 * @param key
	 * @return
	 */
	public static RedisPage<T> findPage(RedisPage<T> page, String key){
		Jedis jedis = null;
		try{
			int start = page.getFirst();
			int end = start + page.getPageSize();
			jedis  = getJedis();
			List<String> data = jedis.lrange(key, start, end);
			List<T> list = new ArrayList<T>();
			for(String string : data){
				list.add(parse(string, T.class));
			}
			page.setItems(list);
			return page;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 设置值
	 * @param key
	 * @param data
	 */
	public static void put(String key, Object data){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			jedis.set(key, toJSONString(data));
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 设置列表
	 * @param key
	 * @param list
	 */
	public static <E> void setList(String key, List<E> list){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.del(key);
			String[] strings = toStrings(list);
			if(strings.length > 0){				
				jedis.rpush(key, strings);
			}
			else{
				throw new SimpleRuntimeException("数据错误");
			}
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 新增
	 * @param key
	 * @param item
	 */
	public static void insert(String key, Object item){
		rpush(key, item);
	}
	
	/**
	 * rpush
	 * @param key
	 * @param item
	 */
	public static void rpush(String key, Object item){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.rpush(key, toJSONString(item));
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * lpush
	 * @param key
	 * @param item
	 */
	public static void lpush(String key, Object item) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.lpush(key, toJSONString(item));
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 获取键值列表
	 * @param pattern
	 * @return
	 */
	public static Set<String> keys(String pattern){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			Set<String> keys =  jedis.keys(pattern);
			return keys;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 设置值
	 * @param key
	 * @param obj
	 */
	public static void set(String key, Object obj){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.set(key, toJSONString(obj));
		}
		catch(JedisException ex){
			logger.error(ex.getMessage(), ex);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 设置值
	 * @param index
	 * @param key
	 * @param obj
	 */
	public static void set(int index, String key, Object obj){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.select(index);
			jedis.set(key, toJSONString(obj));
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 自增
	 * @param index
	 * @param key
	 * @param integer
	 */
	public static long incrBy(int index, String key, long value) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.select(index);
			return jedis.incrBy(key, value);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 自增
	 * @param key
	 * @param integer
	 */
	public static long incrBy(String key, long value) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.incrBy(key, value);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 *   自减
	 * @param index
	 * @param key
	 * @param integer
	 */
	public static long decrBy(int index, String key, long value) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.select(index);
			return jedis.decrBy(key, value);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 *  自减
	 * @param key
	 * @param integer
	 */
	public static long decrBy(String key, long value) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.decrBy(key, value);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 自增加 浮点型
	 * @param index
	 * @param key
	 * @param integer
	 */
	public static void incrByFloat(int index, String key, float value) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.select(index);
			jedis.incrByFloat(key, value);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 自增加 浮点型
	 * @param key
	 * @param integer
	 */
	public static void incrByFloat(String key, float value) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.incrByFloat(key, value);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 哈希表字段自增
	 * @param index
	 * @param key
	 * @param field
	 * @param integer
	 */
	public static long hincrBy(int index, String key, String field, long value) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.select(index);
			return jedis.hincrBy(key, field, value);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 哈希表字段自增
	 * @param key
	 * @param field
	 * @param integer
	 */
	public static long hincrBy(String key, String field, long value) {
		Jedis jedis = null;
		try{
			jedis = getJedis();
			return jedis.hincrBy(key, field, value);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	
	/**
	 * lset
	 * @param key
	 * @param index
	 * @param obj
	 */
	public static void lset(String key, int index, Object obj){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.lset(key, index, toJSONString(obj));
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 删除
	 * @param key
	 */
	public static void remove(String key){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.del(key);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	
	/**
	 * 发布
	 * @param channel
	 * @param obj
	 */
	public static void publish(String channel, Object obj){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			String data = toJSONString(obj);
			System.out.println("================== 发布消息 ========================");
			System.out.println(channel + " " + data);
			jedis.publish(channel, data);
			System.out.println("================== 发布消息结束 ========================");
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 订阅
	 * @param jedisPubSub
	 * @param channels
	 */
	public static void subscribe(final JedisPubSub jedisPubSub, final String... channels){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.subscribe(jedisPubSub, channels);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 根据key pop对象
	 * @param key
	 * @param clazz
	 * @return
	 */
	public static <T> T pop(String key, Class<T> clazz){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			String value = jedis.lpop(key);
			if(isNullOrEmpty(value)){
				return null;
			}

			return parse(value, clazz);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 根据key pop 字符串
	 * @param key
	 * @return
	 */
	public static String pop(String key){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			return jedis.lpop(key);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 根据key rpop 字符串
	 * @param key
	 * @return
	 */
	public static String rpop(String key){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			return jedis.rpop(key);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 判断key 是否存在
	 * @param key
	 * @return
	 */
	public static boolean exists(String key){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			return jedis.exists(key);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}

	
	/**
	 * 转字符串
	 * @param object
	 * @return
	 */
	public static String toJSONString(Object object) {
		if(object instanceof String){
			return (String) object;
		}
		ObjectMapper objectMapper = new JacksonObjectMapper();
		try {
			return objectMapper.writeValueAsString(object);
		} catch (JsonProcessingException ex) {
			logger.error(ex.getMessage(), ex);
		}
		return "";
	}
	
	/**
	 * 转对象
	 * @param value
	 * @param clazz
	 * @return
	 * @throws IOException
	 */
	@SuppressWarnings({ "unchecked", "hiding" })
	public static <T> T parse(String value, Class<T> clazz){
		if(clazz == null || clazz.equals(String.class)){
			return (T) value;
		}
		ObjectMapper objectMapper = new JacksonObjectMapper();
		try {
			return objectMapper.readValue(value, clazz);
		} catch (IOException ex) {
			logger.error(ex.getMessage(), ex);
		}
		return null;
	}
	
	/**
	 * 列表转字符串数组
	 * @param list
	 * @return
	 */
	public static final <E> String[] toStrings(List<E> list){
		String[] items = new String[list.size()];
		for(int i=0; i<list.size(); i++){
			items[i] = toJSONString(list.get(i));
		}
		return items;
	}
	

	/**
	 * 判断字符串是否为空
	 * 
	 * @param string
	 * @return
	 */
	public static boolean isNullOrEmpty(String string) {
		return string == null || "".equals(string);
	}

	


}
