package com.crawler.res.common;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.UUID;

import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.multipart.commons.CommonsMultipartFile;

import com.crawler.ffmpeg.bean.Video;
import com.crawler.ffmpeg.utils.FFMpegUtils;
import com.crawler.res.image.Image;
import com.crawler.res.qiniu.QiniuClient;
import com.crawler.res.utils.FileUtils;
import com.crawler.res.utils.QiNiuUtils;
import com.qiniu.common.QiniuException;
import com.qiniu.http.Client;

import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

/**
 * 上传基类
 * @author Rubekid
 *
 * 2017年4月15日 下午9:57:43
 */
public abstract class UploaderBaseController {
	
	
	/**
	 * 获取上传token
	 * @return
	 */
	@RequestMapping(value = "/token", method = RequestMethod.GET)
	public Object token(){
		String token = QiNiuUtils.getUpToken();
		JSONObject json = new JSONObject();
		json.put("value", token);
		return json;
	}
	
	/**
	 * 文件上传
	 * @param multipartFiles
	 * @return
	 * @throws IOException 
	 * @throws Exception
	 */
	@ResponseBody
	@RequestMapping(method = RequestMethod.POST, headers = "content-type=multipart/form-data")
	public JSONObject upload( @RequestParam("file") CommonsMultipartFile multipartFile) throws IOException{
		return doFileUpload(multipartFile);
		
	}
	
	/**
	 * 多文件上传
	 * @param multipartFiles1
	 * @param multipartFiles2
	 * @return
	 * @throws IOException
	 */
	@ResponseBody
	@RequestMapping(value = "/files", method = RequestMethod.POST, headers = "content-type=multipart/form-data")
	public JSONArray multiUpload( @RequestParam("files") List<CommonsMultipartFile> multipartFiles1, @RequestParam("files[]") List<CommonsMultipartFile> multipartFiles2) throws IOException{
		List<CommonsMultipartFile> multipartFiles = multipartFiles1 == null ? multipartFiles2 : multipartFiles1;
		JSONArray items = new JSONArray();
		for(CommonsMultipartFile multipartFile : multipartFiles){
			items.add(doFileUpload(multipartFile));
		}
		return items;
	}
	
	/**
	 * 图片上传
	 * @param multipartFile
	 * @return
	 * @throws Exception 
	 */
	@ResponseBody
	@RequestMapping(value = "/image", method = RequestMethod.POST, headers = "content-type=multipart/form-data")
	public JSONObject imageUpload( @RequestParam("image") CommonsMultipartFile multipartFile,
			@RequestParam(value = "showInfo", required = false, defaultValue="false") Boolean showInfo,
			@RequestParam(value = "width", required = false) Integer width,
			@RequestParam(value = "height", required = false) Integer height) throws Exception{
		return doImageUpload(multipartFile, showInfo, width, height);
	}
	
	/**
	 * 图片批量上传
	 * @param multipartFile
	 * @return
	 * @throws Exception 
	 */
	@ResponseBody
	@RequestMapping(value = "/images", method = RequestMethod.POST, headers = "content-type=multipart/form-data")
	public JSONArray batchImageUpload( @RequestParam(value = "images", required = false) List<CommonsMultipartFile> multipartFiles1, 
			@RequestParam(value = "images[]", required = false) List<CommonsMultipartFile> multipartFiles2,
			@RequestParam(value = "showInfo", required = false, defaultValue="false") Boolean showInfo,
			@RequestParam(value = "width", required = false) Integer width,
			@RequestParam(value = "height", required = false) Integer height) throws Exception{
		List<CommonsMultipartFile> multipartFiles = multipartFiles1 == null ? multipartFiles2 : multipartFiles1;
		JSONArray items = new JSONArray();
		for(CommonsMultipartFile multipartFile : multipartFiles){
			items.add(doImageUpload(multipartFile, showInfo, width, height));
		}
		return items;
	}
	
	/**
	 * 音频上传
	 * @param multipartFiles
	 * @return
	 * @throws IOException 
	 * @throws Exception
	 */
	@ResponseBody
	@RequestMapping(value = "/audio", method = RequestMethod.POST, headers = "content-type=multipart/form-data")
	public JSONObject audioUpload( @RequestParam("audio") CommonsMultipartFile multipartFile) throws IOException{
		return doFileUpload(multipartFile, ".mp3");
		
	}
	
	/**
	 * 视频上传
	 * @param multipartFile
	 * @return
	 * @throws IOException
	 */
	@ResponseBody
	@RequestMapping(value = "/video", method = RequestMethod.POST, headers = "content-type=multipart/form-data")
	public JSONObject videoUpload( @RequestParam("video") CommonsMultipartFile multipartFile,
			@RequestParam(value = "privately", defaultValue="false") Boolean privately,
			@RequestParam(value = "forceResize", defaultValue="false") Boolean forceResize,
			@RequestParam(value = "resolution", defaultValue="640x360") String resolution) throws IOException{
		String oFileName = multipartFile.getOriginalFilename();
		String ext = ".mp4";
		if(oFileName.contains(".")){
			ext = oFileName.substring(oFileName.lastIndexOf("."));
		}
		String baseFileName = FileUtils.createFileName();
		String fileName =  baseFileName + ext;
		String newFileName = baseFileName + ".mp4";
		JSONObject result = new JSONObject();
		
		// 不是Mp4类型或者强制压缩
		if(!newFileName.equals(fileName) || forceResize){
			fileName = "tmp/" + fileName;
			String url =  QiNiuUtils.uploadVideo(multipartFile.getBytes(), fileName);
			String persistentId = QiNiuUtils.toMp4(fileName, newFileName,resolution);
			result.put("persistentId", persistentId);
			result.put("url", url );
			result.put("preview", privately ? QiNiuUtils.getPrivateDownloadUrl(url) : url);
			return  result;
		}
		else{
			String url = QiNiuUtils.uploadVideo(multipartFile.getBytes(), newFileName);
			JSONObject info = QiNiuUtils.avinfo(url);
			result.put("url",  url);
			result.put("preview", privately ? QiNiuUtils.getPrivateDownloadUrl(url) : url);
			result.put("cover", QiNiuUtils.getVideoCover(url, info.getInt("width"), info.getInt("height")));
			result.put("size", info.get("size"));
			result.put("width", info.get("width"));
			result.put("height", info.get("height"));
			result.put("duration", info.get("duration"));
			return  result;
		}
	}
	
	/**
	 * 视频上传并转成m3u8
	 * @param multipartFile
	 * @return
	 * @throws IOException 
	 * @throws InterruptedException 
	 * @throws Exception 
	 */
	@ResponseBody
	@RequestMapping(value = "/m3u8", method = RequestMethod.POST, headers = "content-type=multipart/form-data")
	public JSONObject m3u8( @RequestParam("video") CommonsMultipartFile multipartFile, 
			@RequestParam(value = "privately", defaultValue="false") Boolean privately,
			@RequestParam(value = "forceResize", defaultValue="false") Boolean forceResize) throws IOException, InterruptedException{
		String oFileName = multipartFile.getOriginalFilename();
		String ext = ".tmp";
		if(oFileName.contains(".")){
			ext = oFileName.substring(oFileName.lastIndexOf("."));
		}
		String baseFileName = FileUtils.createFileName();
		String fileName =  "video/mp4/" + baseFileName + ".mp4";
		JSONObject result = new JSONObject();
		String url = "";
		
		// 不是Mp4类型或者强制压缩
		if(!".mp4".equalsIgnoreCase(ext)){
			String srcFilePath = ResConstant.TEMP_PATH + baseFileName + ext;
			String destFilePath = ResConstant.TEMP_PATH + baseFileName + ".mp4";
			File srcFile = new File(srcFilePath);
			if(!srcFile.exists()){
				srcFile.createNewFile();
			}
			multipartFile.transferTo(srcFile);
			Video video = FFMpegUtils.parseVideo(srcFilePath);
			FFMpegUtils.videoTransfer(srcFilePath, destFilePath, video.getDestResolution(), video.getTranspose());
			
			File destFile = new File(destFilePath);
			url = QiNiuUtils.uploadVideo(destFilePath, fileName);
			srcFile.delete();
			destFile.delete();
		}
		else if(forceResize){
			String srcFilePath = ResConstant.TEMP_PATH + baseFileName + ext;
			String destFilePath = ResConstant.TEMP_PATH + baseFileName + "_resize.mp4";
			File srcFile = new File(srcFilePath);
			if(!srcFile.exists()){
				srcFile.createNewFile();
			}
			multipartFile.transferTo(srcFile);
			Video video = FFMpegUtils.parseVideo(srcFilePath);
			if(video.getBitRate() > 900){
				FFMpegUtils.resize(srcFilePath, destFilePath, "800k", video.getDestResolution());
				File destFile = new File(destFilePath);
				url = QiNiuUtils.uploadVideo(destFilePath, fileName);
				srcFile.delete();
				destFile.delete();
			}
			else{
				url = QiNiuUtils.uploadVideo(multipartFile.getBytes(), fileName);
			}
		}
		else{
			url = QiNiuUtils.uploadVideo(multipartFile.getBytes(), fileName);
		}
		
		String persistentId = QiNiuUtils.m3u8(fileName);
		result.put("persistentId", persistentId);
		result.put("url", url );
		result.put("preview", privately ? QiNiuUtils.getPrivateDownloadUrl(url) : url);
		result.put("m3u8", QiNiuUtils.getM3u8FullPath(fileName));
		return  result;
	}
	
	
	/**
	 * 获取视频封面
	 * @param url
	 * @param width
	 * @param height
	 * @return
	 */
	@ResponseBody
	@RequestMapping(value = "/cover", method = RequestMethod.GET)
	public JSONObject getCover(@RequestParam("url") String url, 
			@RequestParam("width") Integer width, @RequestParam("height") Integer height){
		JSONObject response = new JSONObject();
		response.put("cover", QiNiuUtils.uploadVideoCover(url, width, height));
		return response;
	}
	
	/**
	 * 查询视频状态
	 * @return
	 * @throws QiniuException 
	 */
	@ResponseBody
	@RequestMapping(value = "/video/query/status/{persistentId:.+}", method = RequestMethod.GET)
	public JSONObject queryVideoStatus(@PathVariable("persistentId") String persistentId) throws QiniuException{
		Client client = new Client();
		return JSONObject.fromObject(client.get("http://api.qiniu.com/status/get/prefop?id=" + persistentId).bodyString());
	}
	
	/**
	 * 获取私有下载地址
	 * @return
	 * @throws QiniuException 
	 */
	@ResponseBody
	@RequestMapping(value = "/download", method = RequestMethod.GET)
	public JSONObject download(@RequestParam("url") String baseUrl, @RequestParam("expires") Long expires) throws QiniuException{
		JSONObject json = new JSONObject();
		json.put("url", QiNiuUtils.getPrivateDownloadUrl(baseUrl, expires));
		return json;
	}
	

	
	/**
	 * 图片上传操作
	 * @param multipartFile
	 * @return
	 * @throws Exception 
	 */
	protected JSONObject doImageUpload(CommonsMultipartFile multipartFile, Boolean showInfo, Integer width, Integer height) throws Exception{
		if(width == null){
			width = ResConstant.MAX_WIDTH;
		}
		if(height == null){
			height = 0;
		}
		Image image = new Image(multipartFile);
		byte[] bytes;
		if(width > 0){
			ByteArrayOutputStream outputStream = image.resize(width, 0, ResConstant.LIMIT_SIZE);
			image.removeTemp();
			bytes = outputStream.toByteArray();
		}
		else{
			bytes = multipartFile.getBytes();
		}
		String url = "";
		QiniuClient client = getQiniuClient();
		if(client!= null){
			url = client.upload(bytes, image.getFullName());
		}
		else{
			url = QiNiuUtils.upload(bytes, image.getFullName());
		}
		JSONObject result = new JSONObject();
		result.put("url", url );
		if(showInfo!= null && showInfo){
			JSONObject info = QiNiuUtils.imageInfo(url);
			result.put("size", info.get("size"));
			result.put("width", info.get("width"));
			result.put("height", info.get("height"));
		}
		return  result;
	}
	
	
	/**
	 * 普通文件上传
	 * @param multipartFile
	 * @return
	 * @throws IOException
	 */
	protected JSONObject doFileUpload(CommonsMultipartFile multipartFile, String ext) throws IOException{
		if(ext == null){
			ext = ".tmp";
		}
		String oFileName = multipartFile.getOriginalFilename();
		if(oFileName.contains(".")){
			ext = oFileName.substring(oFileName.lastIndexOf("."));
		}
		String fileName =  UUID.randomUUID() + ext;
		byte[] bytes = multipartFile.getBytes();
		QiniuClient client = getQiniuClient();
		String url = "";
		if(client!= null){
			url = client.upload(bytes, fileName);
		}
		else{
			url = QiNiuUtils.upload(bytes, fileName);
		}
		JSONObject result = new JSONObject();
		result.put("url", url );
		return  result;
	}
	
	/**
	 * 普通文件上传
	 * @param multipartFile
	 * @return
	 * @throws IOException
	 */
	protected JSONObject doFileUpload(CommonsMultipartFile multipartFile) throws IOException{
		return doFileUpload(multipartFile, ".tmp");
	}
	
	
	
	/**
	 * 获取七牛客户端
	 * @param request
	 * @return
	 */
	protected QiniuClient getQiniuClient(){
		return null;
		
	}
	
}
