package com.crawler.utils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import org.apache.poi.ss.formula.functions.T;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.crawler.beans.Page;
import com.crawler.rest.utils.Warner;

import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;
import redis.clients.jedis.JedisPubSub;

/**
 * redis 管理器
 * @author rubekid
 *
 * 2016年12月17日 下午10:19:25
 */
public class RedisManager {
	
	/**
	 * 日志
	 */
	private static Logger logger = LoggerFactory.getLogger(RedisManager.class);
	
	/**
	 * 等待超时毫秒数
	 */
	private static int TIMEOUT_MSECS = 60000;
	
	/**
	 * 锁自动过期毫秒数
	 */
	private static int LOCK_EXPIRE_MSECS = 60000;
	
	/**
	 * 
	 * jedis池
	 */
	private static JedisPool pool;


	 /**
	  * 静态代码初始化池配置
	  */
	static {
		try {
		
			Properties props = new Properties();
			props.load(RedisManager.class.getClassLoader().getResourceAsStream("redis.properties"));
			JedisPoolConfig config = new JedisPoolConfig();
			config.setMaxTotal(Integer.valueOf(props.getProperty("jedis.pool.maxTotal")));
			config.setMaxIdle(Integer.valueOf(props.getProperty("jedis.pool.maxIdle")));
			config.setMaxWaitMillis(Long.valueOf(props.getProperty("jedis.pool.maxWaitMillis")));
			config.setTestOnBorrow(Boolean.valueOf(props.getProperty("jedis.pool.testOnBorrow")));
			config.setTestOnReturn(Boolean.valueOf(props.getProperty("jedis.pool.testOnReturn")));
			
			String ip = props.getProperty("redis.ip");
			Integer port = Integer.valueOf(props.getProperty("redis.port"));
			Integer expire = Integer.valueOf(props.getProperty("redis.expire"));
			String password = props.getProperty("redis.password");
			if(password !=null){
				pool = new JedisPool(config, ip, port, expire, password);
			}
			else{
				pool = new JedisPool(config, ip, port, expire);
			}
			

		} catch (IOException ex) {
			logger.error(ex.getMessage(), ex);
		}
	}
	
	/**
	 * 同步锁
	 * @param key
	 * @throws InterruptedException 
	 */
	public static boolean lock(String lockKey){
		long start = System.currentTimeMillis();
		Jedis jedis = null;
		try{
			jedis = getJedis();
			
			int timeout = TIMEOUT_MSECS;
	        while (timeout >= 0) {
	            long time = System.currentTimeMillis() + LOCK_EXPIRE_MSECS + 1;
	            String expireAt = String.valueOf(time);
	            if (jedis.setnx(lockKey, expireAt) == 1){
	                return true;
	            }

	            String currentValue = jedis.get(lockKey); 
	            if (currentValue != null && Long.parseLong(currentValue) < System.currentTimeMillis()) {

	                String oldValue = jedis.getSet(lockKey, expireAt);
	                //获取上一个锁到期时间，并设置现在的锁到期时间，
	                //只有一个线程才能获取上一个线上的设置时间，因为jedis.getSet是同步的
	                if(oldValue != null && oldValue.equals(currentValue)) {
	                    //如过这个时候，多个线程恰好都到了这里，但是只有一个线程的设置值和当前值相同，他才有权利获取锁
	                    return true;
	                }
	            }
	            timeout -= 100;
	            try {
					Thread.sleep(100);
				} catch (InterruptedException ex) {
					logger.error(ex.getMessage(), ex);
					Warner.send("RedisManager.lock Thread.sleep 异常：" + ex.getMessage());
				}
	        }
	        return false;
		}
		finally{
			
			if(null != jedis){				
				jedis.close();
			}
			System.out.println("获取redis锁时间：" + (System.currentTimeMillis() - start));
		}
		
	}
	
	/**
	 * 解除锁定
	 * @param key
	 */
	public static void unlock(String lockKey){
		Jedis jedis = null;
		try{
			jedis = getJedis();
	        jedis.del(lockKey);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}

	/**
	 * 批量删除
	 * @param jedis
	 * @param prefix
	 */
	public static void batchDeleteByKey(String prefix){
		Jedis jedis = null;
		try{
			jedis = getJedis();
	        Set<String> set = jedis.keys(prefix + "*");  
	        Iterator<String> it = set.iterator();
	        List<String> keys = new ArrayList<String>();
	        while(it.hasNext()){
	        	keys.add(it.next());
	        	if(keys.size() > 0 && keys.size() % 100 == 0){
	        		jedis.del(keys.toArray(new String[keys.size()]));
	        		keys.clear();
	        	}
	        	
	        }   
	        if(keys.size() > 0){
	        	jedis.del(keys.toArray(new String[keys.size()]));  
	        }
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}

	/**
	 * 获取Jedis
	 * @return
	 */
	public static Jedis getJedis() {
		return pool.getResource();

	}
	
	/**
	 * 关闭
	 * @param jedis
	 */
	public static void close(Jedis jedis) {
		jedis.close();
	}
	
	/**
	 * 通过key获取
	 * @param key
	 * @return
	 */
	public static String get(String key, String defaultValue){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			String value = jedis.get(key);
			return value == null ? defaultValue : value;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 保存
	 * @param key
	 * @param value
	 */
	public static void set(String key, String value){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.set(key, value);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 根据key 获取对象
	 * @param key
	 * @param clazz
	 * @return
	 */
	public static <T> T get(String key, Class<T> clazz){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			JSONObject json = JSONObject.parseObject(jedis.get(key));
			return json.toJavaObject(clazz);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
		 
	}
	
	/**
	 * 根据key 获取列表
	 * @param key
	 * @param clazz
	 * @return
	 */
	public static <T> List<T> findList(String key, Class<T> clazz, String ... exceptKeys){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			List<String> data = new ArrayList<String>();
			if(key.contains("*")){
				Set<String> keys = jedis.keys(key);
				List<String> exceptKeyList = Arrays.asList(exceptKeys);
				for(String k : keys){
					if(exceptKeyList.contains(k)){
						continue;
					}
					data.addAll(jedis.lrange(k, 0, -1));
				}
			}
			else{
				data = jedis.lrange(key, 0, -1);
			}

			List<T> list = new ArrayList<T>();
			for(String string : data){
				list.add(JSONObject.parseObject(string).toJavaObject(clazz));
			}
			return list;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
		
	}
	
	/**
	 * 分页
	 * @param page
	 * @param key
	 * @return
	 */
	public static Page<T> findPage(Page<T> page, String key){
		Jedis jedis = null;
		try{
			int start = page.getFirst();
			int end = start + page.getPageSize();
			jedis  = getJedis();
			List<String> data = jedis.lrange(key, start, end);
			List<T> list = new ArrayList<T>();
			for(String string : data){
				list.add(JSONObject.parseObject(string).toJavaObject(T.class));
			}
			page.setItems(list);
			return page;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 设置值
	 * @param key
	 * @param data
	 */
	public static void put(String key, Object data){
		Jedis jedis = null;
		try{
			jedis  = getJedis();
			jedis.set(key, JSONObject.toJSONString(data));
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 设置列表
	 * @param key
	 * @param list
	 */
	public static <E> void setList(String key, List<E> list){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.del(key);
			String[] strings = StringUtils.toStrings(list);
			if(strings.length > 0){				
				jedis.rpush(key, strings);
			}
			else{
				Warner.send("redis 参数有误" + JSON.toJSONString(list));
			}
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 新增
	 * @param key
	 * @param item
	 */
	public static void insert(String key, Object item){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.rpush(key, JSONObject.toJSONString(item));
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 获取键值列表
	 * @param pattern
	 * @return
	 */
	public static Set<String> keys(String pattern){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			Set<String> keys =  jedis.keys(pattern);
			return keys;
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 设置值
	 * @param key
	 * @param obj
	 */
	public static void set(String key, Object obj){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.set(key, JSONObject.toJSONString(obj));
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	
	public static void lset(String key, int index, Object obj){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.lset(key, index, JSONObject.toJSONString(obj));
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 删除
	 * @param key
	 */
	public static void remove(String key){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.del(key);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	
	/**
	 * 发布
	 * @param channel
	 * @param obj
	 */
	public static void publish(String channel, Object obj){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			String data = JSONObject.toJSONString(obj);
			System.out.println("================== 发布消息 ========================");
			System.out.println(channel + " " + data);
			jedis.publish(channel, data);
			System.out.println("================== 发布消息结束 ========================");
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}
	
	/**
	 * 订阅
	 * @param jedisPubSub
	 * @param channels
	 */
	public static void subscribe(final JedisPubSub jedisPubSub, final String... channels){
		Jedis jedis = null;
		try{
			jedis = getJedis();
			jedis.subscribe(jedisPubSub, channels);
		}
		finally{
			if(null != jedis){				
				jedis.close();
			}
		}
	}

	

}
