package com.crawler.weixin.web;

import java.io.IOException;
import java.util.Date;

import javax.servlet.http.HttpServletResponse;

import org.apache.commons.codec.digest.DigestUtils;

import com.crawler.common.config.UtilConstant;
import com.crawler.utils.EhCacheManager;
import com.crawler.utils.StringUtils;
import com.crawler.utils.http.SimpleHttpClient;


public class WeixinWebUtils {
	
	private static final String WECHAT_CACHE_NAME = "wechat-cache";
	
	/**
	 * 获取公众号token
	 */
	private static final String ACCESS_TOKEN_URL = "https://api.weixin.qq.com/cgi-bin/token?grant_type=client_credential&appid=%s&secret=%s";
	
	/**
	 * 获取公众号票据
	 */
	private static final String TICKET_URL = "https://api.weixin.qq.com/cgi-bin/ticket/getticket?access_token=%s&type=jsapi";
	
	/**
	 * 获取tokenURL
	 */
	private static final String USER_ACCESS_TOKEN_URL = "https://api.weixin.qq.com/sns/oauth2/access_token?appid=%s&secret=%s&code=%s&grant_type=authorization_code";
	
	/**
	 * 获取用户信息URL
	 */
	private static final String USERINFO_URL = "https://api.weixin.qq.com/sns/userinfo?access_token=%s&openid=%s&lang=zh_CN";
	
	/**
	 * 授权登录
	 */
	private static final String AUTHORIZE_URL  = "https://open.weixin.qq.com/connect/oauth2/authorize?appid=%s&redirect_uri=%s&response_type=code&scope=snsapi_userinfo&state=1#wechat_redirect";
	
	/**
	 * 通过code换取网页授权access_token
	 * @param appId
	 * @param appSecret
	 * @param code
	 */
	public static UserAccessToken getUserAccessToken(String appId, String appSecret, String code){
		String url = String.format(USER_ACCESS_TOKEN_URL, appId, appSecret, code);
		return SimpleHttpClient.getForObject(url, UserAccessToken.class);		
	}
	
	/**
	 * 获取用户信息
	 * @param accessToken
	 * @param openid
	 */
	public static UserInfo getUserInfo(String accessToken, String openid){
		String url = String.format(USERINFO_URL, accessToken, openid);
		return SimpleHttpClient.getForObject(url, UserInfo.class);
	}
	
	
	/**
	 * 获取Token
	 * @param appId
	 * @param appSecret
	 * @param code
	 * @return
	 */
	public static AccessToken getAccessToken(String appId, String appSecret){
		EhCacheManager ehCacheManager = new EhCacheManager(WECHAT_CACHE_NAME, "acccess_token");
		String key = String.format("%s_%s", appId, appSecret);
		AccessToken accessToken = ehCacheManager.get(key, AccessToken.class);
		if(accessToken == null || accessToken.getExpireAt().before(new Date())){
			String url = String.format(ACCESS_TOKEN_URL, appId, appSecret);
			accessToken =  SimpleHttpClient.getForObject(url, AccessToken.class);
			accessToken.setExpireAt(new Date(System.currentTimeMillis() + accessToken.getExpiresIn()));
			ehCacheManager.put(key, accessToken);
		}
		return accessToken;
	}
	
	/**
	 * 获取票据
	 * @param accessToken
	 * @return
	 */
	public static Ticket getTicket(String accessToken){
		EhCacheManager ehCacheManager = new EhCacheManager(WECHAT_CACHE_NAME, "ticket");
		Ticket ticket = ehCacheManager.get(accessToken, Ticket.class);
		if(ticket == null || ticket.getExpireAt().before(new Date())){
			String url = String.format(TICKET_URL, accessToken);
			ticket =  SimpleHttpClient.getForObject(url, Ticket.class);
			ticket.setExpireAt(new Date(System.currentTimeMillis() + ticket.getExpiresIn()));
			ehCacheManager.put(accessToken, ticket);
		}
		return ticket;
	}
	
	public static WxJsConfig signature(String url){
		return signature(UtilConstant.WECHAT_APPID, UtilConstant.WECHAT_APPSECRET, url);
	}
	
	/**
	 * 获取微信签名
	 * @param appId
	 * @param secret
	 * @param url
	 * @return
	 */
	public static WxJsConfig signature(String appId, String secret, String url){
		WxJsConfig wxJsConfig = new WxJsConfig();
		wxJsConfig.setAppId(appId);
		wxJsConfig.setNonceStr(StringUtils.getRandomString(16));
		wxJsConfig.setTimestamp(System.currentTimeMillis() / 1000);
		AccessToken accessToken = getAccessToken(appId, secret);
		Ticket ticket = getTicket(accessToken.getAccessToken());
		String str = String.format("jsapi_ticket=%s&noncestr=%s&timestamp=%s&url=%s", ticket.getTicket(), wxJsConfig.getNonceStr(), wxJsConfig.getTimestamp(),url);
		String signature = DigestUtils.sha1Hex(str.getBytes());
		wxJsConfig.setSignature(signature);
		return wxJsConfig;
	}
	
	/**
	 * 授权登录
	 * @param response
	 * @param redirectUrl
	 * @throws IOException
	 */
	public static void authorize(HttpServletResponse response, String redirectUrl) throws IOException{
		String location = String.format(AUTHORIZE_URL, UtilConstant.WECHAT_APPID, redirectUrl);
		response.sendRedirect(location);
	}
}
