package com.crawler.client.http;


import java.io.IOException;
import java.lang.reflect.Type;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import org.apache.http.HeaderElement;
import org.apache.http.HeaderElementIterator;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.config.ConnectionConfig;
import org.apache.http.config.SocketConfig;
import org.apache.http.conn.ConnectionKeepAliveStrategy;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicHeaderElementIterator;
import org.apache.http.protocol.HTTP;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.http.client.ClientHttpRequest;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.util.StopWatch;
import org.springframework.web.client.HttpMessageConverterExtractor;
import org.springframework.web.client.RequestCallback;
import org.springframework.web.client.ResponseExtractor;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriTemplate;

import com.crawler.client.util.JsonMapper;
import com.crawler.waf.config.WafProperties;

/**
 * http请求工具类
 */
public class RestHttpClient {
    public static final String CLIENT_CONNECT_TIMEOUT = "client.connectTimeout";
    public static final String CLIENT_SOCKET_TIMEOUT = "client.socketTimeout";

    private static final String CLIENT_CONNECT_TIMEOUT_VALUE = "30000";
    private static final String CLIENT_SOCKET_TIMEOUT_VALUE = "10000";

    protected final Logger log = LoggerFactory.getLogger(this.getClass());
    private RestRestTemplate restTemplate;

    static CloseableHttpClient httpClient;
    static IdleConnectionMonitorThread monitorThread;

    static {
        final ConnectionConfig connectionConfig = ConnectionConfig.custom()
                .setBufferSize(8 * 1024)
                .setFragmentSizeHint(8 * 1024)
                .build();
        final SocketConfig socketConfig = SocketConfig.custom()
                .setSoTimeout(Integer.parseInt(CLIENT_SOCKET_TIMEOUT_VALUE))
                .build();

        ConnectionKeepAliveStrategy keepAliveStrategy = new ConnectionKeepAliveStrategy() {
            @Override
            public long getKeepAliveDuration(HttpResponse response, HttpContext context) {
                HeaderElementIterator it = new BasicHeaderElementIterator(
                        response.headerIterator(HTTP.CONN_KEEP_ALIVE));
                while (it.hasNext()) {
                    HeaderElement he = it.nextElement();
                    String param = he.getName();
                    String value = he.getValue();
                    if (value != null && param.equalsIgnoreCase("timeout")) {
                        try {
                            return Long.parseLong(value) * 1000;
                        } catch (NumberFormatException ignore) {
                        }
                    }
                }
                // keep alive for 30 seconds
                return 30 * 1000;
            }
        };

        PoolingHttpClientConnectionManager mgr = new PoolingHttpClientConnectionManager();
        mgr.setDefaultSocketConfig(socketConfig);
        mgr.setDefaultConnectionConfig(connectionConfig);
        mgr.setMaxTotal(1000);
        mgr.setDefaultMaxPerRoute(500);
        httpClient = HttpClients.custom()
                .setKeepAliveStrategy(keepAliveStrategy)
                .setConnectionManager(mgr)
                .build();

        // http://hc.apache.org/httpcomponents-client-dev/tutorial/html/connmgmt.html#d4e631
        // http://codereview.stackexchange.com/questions/49151/http-client-requests-done-right
        monitorThread = new IdleConnectionMonitorThread(mgr);
        // Don't stop quitting.
        monitorThread.setDaemon(true);
        monitorThread.start();
    }

    public static class IdleConnectionMonitorThread extends Thread {
        private final HttpClientConnectionManager connMgr;
        private volatile boolean shutdown;

        public IdleConnectionMonitorThread(HttpClientConnectionManager connMgr) {
            super();
            this.connMgr = connMgr;
        }

        @Override
        public void run() {
            try {
                while (!shutdown) {
                    synchronized (this) {
                        wait(5000);
                        // Close expired connections
                        connMgr.closeExpiredConnections();
                        // Optionally, close connections
                        // that have been idle longer than 30 sec
                        connMgr.closeIdleConnections(30, TimeUnit.SECONDS);
                    }
                }
            } catch (InterruptedException ex) {
                // terminate
            }
        }

        public void shutdown() {
            shutdown = true;
            synchronized (this) {
                notifyAll();
            }
        }
    }

    public RestHttpClient() {
        this(WafProperties.getPropertyForInteger(CLIENT_CONNECT_TIMEOUT, CLIENT_CONNECT_TIMEOUT_VALUE),
        		WafProperties.getPropertyForInteger(CLIENT_SOCKET_TIMEOUT, CLIENT_SOCKET_TIMEOUT_VALUE));
    }

    /**
     * 初始化RestHttpClient对象
     *
     * @param connectTimeout 连接超时时间（毫秒），默认 5000 ms
     * @param socketTimeout  socket读写数据超时时间（毫秒），默认 10000 ms
     */
    public RestHttpClient(int connectTimeout, int socketTimeout) {
        if (connectTimeout < 0)
            throw new IllegalArgumentException("Connect timeout value is illegal, must be >=0");
        if (socketTimeout < 0)
            throw new IllegalArgumentException("Socket timeout value is illegal, must be >=0");

        HttpComponentsClientHttpRequestFactory requestFactory = new HttpComponentsClientHttpRequestFactory(httpClient);

        requestFactory.setConnectTimeout(connectTimeout);//设置连接主机超时
        requestFactory.setReadTimeout(socketTimeout);//设置从主机读取数据超时

        restTemplate = new RestRestTemplate(requestFactory);

        MappingJackson2HttpMessageConverter messageConverter = new MappingJackson2HttpMessageConverter();
        messageConverter.setObjectMapper(JsonMapper.getMapper());

        List<HttpMessageConverter<?>> mcs = new ArrayList<HttpMessageConverter<?>>();
        mcs.add(messageConverter);

        restTemplate.setMessageConverters(mcs);
        restTemplate.setErrorHandler(new WafApiErrorHandler());
    }

    public RestTemplate getRestTemplate() {
        return restTemplate;
    }
    
    public HttpClient getHttpClient(){
    	return httpClient;
    }

    protected HttpHeaders mergerHeaders(HttpHeaders headers) {
        return headers;
    }

    //region post

    /**
     * 发送 post 请求
     *
     * @param url
     * @param requestBody
     * @param responseType
     * @param uriVariables
     * @return
     * @since 0.9.5
     */
    public <T> T postForObject(String url, Object requestBody, Class<T> responseType, Object... uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(requestBody);
        return executeForObject(url, HttpMethod.POST, httpEntity, responseType, uriVariables);
    }

    /**
     * 发送 post 请求
     *
     * @param url
     * @param requestBody
     * @param responseType
     * @param uriVariables
     * @return
     */
    public <T> ResponseEntity<T> postForEntity(String url, Object requestBody, Class<T> responseType, Object... uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(requestBody);
        return executeForEntity(url, HttpMethod.POST, httpEntity, responseType, uriVariables);
    }

    /**
     * 发送 post 请求
     *
     * @param url
     * @param requestBody
     * @param uriVariables
     */
    public void post(String url, Object requestBody, Object... uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(requestBody);
        execute(url, HttpMethod.POST, httpEntity, uriVariables);
    }

    /**
     * 发送 post 请求
     *
     * @param url
     * @param requestBody
     * @param responseType
     * @param uriVariables
     * @return
     */
    public <T> T postForObject(String url, Object requestBody, Class<T> responseType, Map<String, ?> uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(requestBody);
        return executeForObject(url, HttpMethod.POST, httpEntity, responseType, uriVariables);
    }

    /**
     * 发送 post 请求
     *
     * @param url
     * @param requestBody
     * @param responseType
     * @param uriVariables
     * @return
     */
    public <T> ResponseEntity<T> postForEntity(String url, Object requestBody, Class<T> responseType, Map<String, ?> uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(requestBody);
        return executeForEntity(url, HttpMethod.POST, httpEntity, responseType, uriVariables);
    }

    /**
     * 发送 post 请求
     *
     * @param url
     * @param requestBody
     * @param uriVariables
     */
    public void post(String url, Object requestBody, Map<String, ?> uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(requestBody);
        execute(url, HttpMethod.POST, httpEntity, uriVariables);
    }
    //endregion

    //region get

    /**
     * 发送 get 请求
     *
     * @param url
     * @param responseType
     * @param uriVariables
     * @return
     */
    public <T> T getForObject(String url, Class<T> responseType, Object... uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(null);
        return executeForObject(url, HttpMethod.GET, httpEntity, responseType, uriVariables);
    }

    /**
     * 发送 get 请求
     *
     * @param url
     * @param responseType
     * @param uriVariables
     * @return
     */
    public <T> ResponseEntity<T> getForEntity(String url, Class<T> responseType, Object... uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(null);
        return executeForEntity(url, HttpMethod.GET, httpEntity, responseType, uriVariables);
    }

    /**
     * 发送 get 请求
     *
     * @param url
     * @param responseType
     * @param uriVariables
     * @return
     */
    public <T> T getForObject(String url, Class<T> responseType, Map<String, ?> uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(null);
        return executeForObject(url, HttpMethod.GET, httpEntity, responseType, uriVariables);
    }

    /**
     * 发送 get 请求
     *
     * @param url
     * @param responseType
     * @param uriVariables
     * @return
     */
    public <T> ResponseEntity<T> getForEntity(String url, Class<T> responseType, Map<String, ?> uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(null);
        return executeForEntity(url, HttpMethod.GET, httpEntity, responseType, uriVariables);
    }
    //endregion

    //region put

    /**
     * 发送 put 请求
     *
     * @param url
     * @param requestBody
     * @param uriVariables
     */
    public void put(String url, Object requestBody, Object... uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(requestBody);
        execute(url, HttpMethod.PUT, httpEntity, uriVariables);
    }

    /**
     * 发送 put 请求
     *
     * @param url
     * @param requestBody
     * @param responseType
     * @param uriVariables
     * @return
     */
    public <T> T putForObject(String url, Object requestBody, Class<T> responseType, Object... uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(requestBody);
        return executeForObject(url, HttpMethod.PUT, httpEntity, responseType, uriVariables);
    }

    /**
     * 发送 put 请求
     *
     * @param url
     * @param requestBody
     * @param responseType
     * @param uriVariables
     * @return
     */
    public <T> ResponseEntity<T> putForEntity(String url, Object requestBody, Class<T> responseType, Object... uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(requestBody);
        return executeForEntity(url, HttpMethod.PUT, httpEntity, responseType, uriVariables);
    }

    /**
     * 发送 put 请求
     *
     * @param url
     * @param requestBody
     * @param uriVariables
     */
    public void put(String url, Object requestBody, Map<String, ?> uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(requestBody);
        execute(url, HttpMethod.PUT, httpEntity, uriVariables);
    }

    /**
     * 发送 put 请求
     *
     * @param url
     * @param requestBody
     * @param responseType
     * @param uriVariables
     * @return
     */
    public <T> T putForObject(String url, Object requestBody, Class<T> responseType, Map<String, ?> uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(requestBody);
        return executeForObject(url, HttpMethod.PUT, httpEntity, responseType, uriVariables);
    }

    /**
     * 发送 put 请求
     *
     * @param url
     * @param requestBody
     * @param responseType
     * @param uriVariables
     * @return
     */
    public <T> ResponseEntity<T> putForEntity(String url, Object requestBody, Class<T> responseType, Map<String, ?> uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(requestBody);
        return executeForEntity(url, HttpMethod.PUT, httpEntity, responseType, uriVariables);
    }
    //endregion

    //region delete

    /**
     * 发送 delete 请求
     *
     * @param url
     * @param uriVariables
     */
    public void delete(String url, Object... uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(null);
        execute(url, HttpMethod.DELETE, httpEntity, uriVariables);
    }

    /**
     * 发送 delete 请求
     *
     * @param url
     * @param responseType
     * @param uriVariables
     * @return
     */
    public <T> T deleteForObject(String url, Class<T> responseType, Object... uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(null);
        return executeForObject(url, HttpMethod.DELETE, httpEntity, responseType, uriVariables);
    }

    /**
     * 发送 delete 请求
     *
     * @param url
     * @param responseType
     * @param uriVariables
     * @return
     */
    public <T> ResponseEntity<T> deleteForEntity(String url, Class<T> responseType, Object... uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(null);
        return executeForEntity(url, HttpMethod.DELETE, httpEntity, responseType, uriVariables);
    }

    /**
     * 发送 delete 请求
     *
     * @param url
     * @param uriVariables
     */
    public void delete(String url, Map<String, ?> uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(null);
        execute(url, HttpMethod.DELETE, httpEntity, uriVariables);
    }

    /**
     * 发送 delete 请求
     *
     * @param url
     * @param responseType
     * @param uriVariables
     * @return
     */
    public <T> T deleteForObject(String url, Class<T> responseType, Map<String, ?> uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(null);
        return executeForObject(url, HttpMethod.DELETE, httpEntity, responseType, uriVariables);
    }

    /**
     * 发送 delete 请求
     *
     * @param url
     * @param responseType
     * @param uriVariables
     * @return
     */
    public <T> ResponseEntity<T> deleteForEntity(String url, Class<T> responseType, Map<String, ?> uriVariables) {
        HttpEntity<Object> httpEntity = getHttpEntity(null);
        return executeForEntity(url, HttpMethod.DELETE, httpEntity, responseType, uriVariables);
    }
    //endregion

    //region execute

    /**
     * 根据 method 执行rest请求，返回对象
     *
     * @param url
     * @param method
     * @param requestEntity
     * @param responseType
     * @param uriVariables
     * @return
     */
    public <T> T executeForObject(String url, HttpMethod method, HttpEntity<?> requestEntity, Class<T> responseType, Object... uriVariables) {
        URI uri = new UriTemplate(url).expand(uriVariables);
        HttpEntity<?> mergeRequestEntity = merge(requestEntity);
        RequestCallback requestCallback = restTemplate.httpEntityCallback(mergeRequestEntity, responseType);
        ResponseExtractor<T> responseExtractor = restTemplate.httpMessageConverterExtractor(responseType);
        return doExecute(uri, method, requestCallback, responseExtractor);
    }

    /**
     * 根据 method 发送 rest 请求，返回带头信息的对象
     *
     * @param url
     * @param method
     * @param requestEntity
     * @param responseType
     * @param uriVariables
     * @return
     */
    public <T> ResponseEntity<T> executeForEntity(String url, HttpMethod method, HttpEntity<?> requestEntity, Class<T> responseType, Object... uriVariables) {
        URI uri = new UriTemplate(url).expand(uriVariables);
        HttpEntity<?> mergeRequestEntity = merge(requestEntity);
        RequestCallback requestCallback = restTemplate.httpEntityCallback(mergeRequestEntity, responseType);
        ResponseExtractor<ResponseEntity<T>> responseExtractor = restTemplate.responseEntityExtractor(responseType);

        return doExecute(uri, method, requestCallback, responseExtractor);
    }

    /**
     * 根据 method 执行 rest 请求，不返回数据
     *
     * @param url
     * @param method
     * @param requestEntity
     * @param uriVariables
     */
    public void execute(String url, HttpMethod method, HttpEntity<?> requestEntity, Object... uriVariables) {
        URI uri = new UriTemplate(url).expand(uriVariables);
        HttpEntity<?> mergeRequestEntity = merge(requestEntity);
        RequestCallback requestCallback = restTemplate.httpEntityCallback(mergeRequestEntity);
        doExecute(uri, method, requestCallback, null);
    }

    /**
     * 根据 method 执行 rest 请求，返回对象
     *
     * @param url
     * @param method
     * @param requestEntity
     * @param responseType
     * @param uriVariables
     * @return
     */
    public <T> T executeForObject(String url, HttpMethod method, HttpEntity<?> requestEntity, Class<T> responseType, Map<String, ?> uriVariables) {
        URI uri = new UriTemplate(url).expand(uriVariables);
        HttpEntity<?> mergeRequestEntity = merge(requestEntity);
        RequestCallback requestCallback = restTemplate.httpEntityCallback(mergeRequestEntity, responseType);
        ResponseExtractor<T> responseExtractor = restTemplate.httpMessageConverterExtractor(responseType);
        return doExecute(uri, method, requestCallback, responseExtractor);
    }

    /**
     * 根据 method 发送 rest 请求，返回 {@link org.springframework.http.ResponseEntity} 对象
     *
     * @param url
     * @param method
     * @param requestEntity
     * @param responseType
     * @param uriVariables
     * @return
     */
    public <T> ResponseEntity<T> executeForEntity(String url, HttpMethod method, HttpEntity<?> requestEntity, Class<T> responseType, Map<String, ?> uriVariables) {
        URI uri = new UriTemplate(url).expand(uriVariables);
        HttpEntity<?> mergeRequestEntity = merge(requestEntity);
        RequestCallback requestCallback = restTemplate.httpEntityCallback(mergeRequestEntity, responseType);
        ResponseExtractor<ResponseEntity<T>> responseExtractor = restTemplate.responseEntityExtractor(responseType);

        return doExecute(uri, method, requestCallback, responseExtractor);
    }

    /**
     * 根据 method 执行 rest 请求，不返回数据
     *
     * @param url
     * @param method
     * @param requestEntity
     * @param uriVariables
     */
    public void execute(String url, HttpMethod method, HttpEntity<?> requestEntity, Map<String, ?> uriVariables) {
        URI uri = new UriTemplate(url).expand(uriVariables);
        HttpEntity<?> mergeRequestEntity = merge(requestEntity);
        RequestCallback requestCallback = restTemplate.httpEntityCallback(mergeRequestEntity);
        doExecute(uri, method, requestCallback, null);
    }
    //endregion

    protected HttpEntity<Object> getHttpEntity(Object requestBody) {
        return new HttpEntity<>(requestBody);
    }

    private HttpEntity<?> merge(HttpEntity<?> httpEntity) {
        HttpHeaders httpHeaders = mergerHeaders(httpEntity.getHeaders());
        return new HttpEntity<>(httpEntity.getBody(), httpHeaders);
    }

    /**
     * 基于此方法真正发送rest请求
     *
     * @param url
     * @param method
     * @param requestCallback
     * @param responseExtractor
     * @return
     */
    protected <T> T doExecute(URI url, HttpMethod method, RequestCallback requestCallback, ResponseExtractor<T> responseExtractor) {
        log.info("Http request begin. " + method + ": " + url);
        StopWatch sw = new StopWatch();
        sw.start();
        T result = restTemplate.execute(url, method, requestCallback, responseExtractor);
        sw.stop();
        log.info("Http request end. Total millis: " + sw.getTotalTimeMillis() + " " + method + ": " + url);

        return result;
    }

    private class RestRestTemplate extends RestTemplate {

        public RestRestTemplate(ClientHttpRequestFactory requestFactory) {
            super(requestFactory);
        }

        @Override
        protected <T> RequestCallback acceptHeaderRequestCallback(Class<T> responseType) {
            return super.acceptHeaderRequestCallback(responseType);
        }

        @Override
        protected <T> RequestCallback httpEntityCallback(Object requestBody) {
            return super.httpEntityCallback(requestBody);
        }

        @Override
        protected <T> RequestCallback httpEntityCallback(Object requestBody, Type responseType) {
            return super.httpEntityCallback(requestBody, responseType);
        }

        @Override
        protected <T> ResponseExtractor<ResponseEntity<T>> responseEntityExtractor(Type responseType) {
            return super.responseEntityExtractor(responseType);
        }

        protected <T> ResponseExtractor<T> httpMessageConverterExtractor(Type responseType) {
            return new HttpMessageConverterExtractor(responseType, getMessageConverters());
        }

        @Override
        protected ClientHttpRequest createRequest(URI url, HttpMethod method) throws IOException {
            return new WafClientHttpRequest(super.createRequest(url, method));
        }
    }
}

