package com.crawler.waf.exceptions;

import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.web.accept.ContentNegotiationManager;

import com.crawler.waf.exceptions.HandlerExceptionResolver;
import com.crawler.waf.exceptions.HandlerExceptionResolverBuilder;
import com.crawler.waf.exceptions.handlers.AbstractWafExceptionHandler;
import com.crawler.waf.exceptions.handlers.CustomExceptionHandler;
import com.crawler.waf.exceptions.handlers.ExceptionHandler;
import com.crawler.waf.exceptions.support.ErrorMessageMapper;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.crawler.waf.exceptions.MapUtils.putAllIfAbsent;
import static org.springframework.util.StringUtils.hasText;
/**
 * 当前对象实现Waf中特定异常的Handler的映射
 *
 */
public class HandlerExceptionResolverBuilder {


	private final Map<Class, ExceptionHandler> exceptionHandlers = new HashMap<>();

	private MediaType defaultContentType;

	private ContentNegotiationManager contentNegotiationManager;

	/**
	 * 类型转换器
	 */
	private List<HttpMessageConverter<?>> httpMessageConverters;


	/**
	 * 使用默认的handler
	 */
	private boolean withDefaultHandlers = true;



	public HandlerExceptionResolver build() {

		if (withDefaultHandlers) {
			// 增加默认的handler
			putAllIfAbsent(exceptionHandlers, getDefaultHandlers()); 
		}


		HandlerExceptionResolver resolver = new HandlerExceptionResolver();
		resolver.setExceptionHandlers((Map) exceptionHandlers);

		if (httpMessageConverters != null) {
			resolver.setMessageConverters(httpMessageConverters); 
		}
		if (contentNegotiationManager != null) {
			resolver.setContentNegotiationManager(contentNegotiationManager);
		}
		if (defaultContentType != null) {
			resolver.setDefaultContentType(defaultContentType);
		}
		resolver.afterPropertiesSet();

		return resolver;
	}

	/**
	 * 默认的媒体类型支持
	 */
	public HandlerExceptionResolverBuilder defaultContentType(
			MediaType mediaType) {
		this.defaultContentType = mediaType;
		return this;
	}

	/**
	 * 默认的媒体类型支持
	 */
	public HandlerExceptionResolverBuilder defaultContentType(
			String mediaType) {
		defaultContentType(hasText(mediaType) ? MediaType
				.parseMediaType(mediaType) : null);
		return this;
	}

	/**
	 * 注册给定的异常类
	 */
	public HandlerExceptionResolverBuilder addHandler(
			Class<?> exceptionClass,
			ExceptionHandler exceptionHandler) {

		exceptionHandlers.put(exceptionClass, exceptionHandler);
		return this;
	}

	/**
	 * 增加异常和异常捕获类到Map中
	 */
	public  HandlerExceptionResolverBuilder addHandler(
			AbstractWafExceptionHandler exceptionHandler) {

		return addHandler(exceptionHandler.getExceptionClass(),
				exceptionHandler);
	}

	/**
	 * 注册异常和异常捕获类
	 *
	 * @param exceptionClass
	 *            异常类型
	 * @param status
	 *            http状态码.
	 */
	public HandlerExceptionResolverBuilder addErrorMessageHandler(
			Class<?> exceptionClass, HttpStatus status) {

		return addHandler(new CustomExceptionHandler(
				exceptionClass, status));
	}

	@SuppressWarnings("rawtypes")
	private Map<Class, ExceptionHandler> getDefaultHandlers() {
		Map<Class, ExceptionHandler> map = new HashMap<>();
		
		for (ErrorMessageMapper item : ErrorMessageMapper.values()) {
			addHandler(item.getExClass(), item.getHandler());
		}

		return map;
	}

	@SuppressWarnings("unused")
	private void addHandlerTo(Map<Class<?>, ExceptionHandler> map,
			Class<?> exceptionClass, HttpStatus status) {
		map.put(exceptionClass, new CustomExceptionHandler(
				exceptionClass, status));
	}

	public MediaType getDefaultContentType() {
		return defaultContentType;
	}

	public void setDefaultContentType(MediaType defaultContentType) {
		this.defaultContentType = defaultContentType;
	}

	public ContentNegotiationManager getContentNegotiationManager() {
		return contentNegotiationManager;
	}

	public void setContentNegotiationManager(
			ContentNegotiationManager contentNegotiationManager) {
		this.contentNegotiationManager = contentNegotiationManager;
	}

	public List<HttpMessageConverter<?>> getHttpMessageConverters() {
		return httpMessageConverters;
	}

	public void setHttpMessageConverters(
			List<HttpMessageConverter<?>> httpMessageConverters) {
		this.httpMessageConverters = httpMessageConverters;
	}


	public boolean isWithDefaultHandlers() {
		return withDefaultHandlers;
	}

	public void setWithDefaultHandlers(boolean withDefaultHandlers) {
		this.withDefaultHandlers = withDefaultHandlers;
	}

	public Map<Class, ExceptionHandler> getExceptionHandlers() {
		return exceptionHandlers;
	}

}
