package com.crawler.uc.support;

import com.crawler.uc.authens.BaseUserInfo;
import com.crawler.uc.authens.UserRole;
import com.crawler.uc.exception.AuthenticationException;
import com.crawler.uc.security.MacTokenAuthentication;
import com.crawler.uc.utils.UcSimulator;
import com.crawler.waf.config.WafProperties;
import com.crawler.waf.security.authens.OauthAccessToken;
import com.crawler.waf.support.Constants;
import com.crawler.waf.utils.Assert;
import com.crawler.waf.utils.StringUtil;
import com.crawler.waqf.common.utils.RequestUtils;
import com.crawler.waqf.common.utils.StringUtils;
import org.apache.shiro.codec.Base64;
import org.apache.shiro.codec.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import javax.servlet.http.HttpServletRequest;
import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.List;

/**
 * 提供给系统的全局访问认证用户的信息。
 *
 */
public final class WafContext {

	private static final Logger logger = LoggerFactory.getLogger(WafContext.class);


	public static MacTokenAuthentication getAuthentication(){
		HttpServletRequest request = RequestUtils.getRequest();
		String authorization = request.getHeader("Authorization");
		if(authorization != null && authorization.startsWith("MAC")){
			String authenticationValue = authorization.replaceAll("^MAC", "").trim();
			return getMacTokenAuthentication(request, authenticationValue);
		}

		return null;
	}

	public static MacTokenAuthentication getMacTokenAuthentication(HttpServletRequest request, String authenticationValue){
		String host = request.getHeader("Host");
		if (StringUtils.isNullOrEmpty(host)) {
			try {
				host = InetAddress.getLocalHost().getHostAddress();
			} catch (UnknownHostException e) {
				throw new AuthenticationException("Host缺失");
			}
		}
		String requestURI = StringUtil.urlHandler(request);
		requestURI = StringUtil.getURI(host, request.getServerPort(), requestURI);

		String[] tokenSlips = authenticationValue.split(",");
		if(tokenSlips.length<3){
			throw new AuthenticationException("Authorization 异常");
		}
		String id = tokenSlips[0].substring(tokenSlips[0].indexOf("=") + 1).replace("\"", "").trim();
		String nonce = tokenSlips[1].substring(tokenSlips[1].indexOf("=") + 1).replace("\"", "").trim();
		String mac = tokenSlips[2].substring(tokenSlips[2].indexOf("=") + 1).replace("\"", "").trim();
		return new MacTokenAuthentication(id,mac,nonce,request.getMethod(),requestURI, host);
	}

	/**
	 * 获取当前用户ID
	 * @return
	 */
	public static String getUserId(){
		OauthAccessToken accessToken = getCurrentToken();
		if(accessToken == null){
			return null;
		}
		return accessToken.getUserId();
	}

	/**
	 * 获取当前token信息
	 * @return
	 */
	public static OauthAccessToken getCurrentToken(){
		MacTokenAuthentication authentication = getAuthentication();
		if(authentication == null){
			return null;
		}
		OauthAccessToken accessToken = UcSimulator.get(Constants.AUTHORIZATION_TYPE_MAC, authentication.getId());
		if(accessToken == null){
			return null;
		}
		checkMac(accessToken, authentication);
		return accessToken;
	}

	/**
	 * 获得当前登录用户的信息
	 * 
	 * @return
	 */
	public static BaseUserInfo getCurrertUserInfo() {
		return UcSimulator.getUserInfo(getUserId());
	}



	/**
	 * 获得当前用户的登录角色信息
	 * 
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public static List<UserRole> getCurrentUserRoles() {
		return UcSimulator.getRoleList(getUserId());
	}


	public static OauthAccessToken checkMac(OauthAccessToken ucCheckToken, MacTokenAuthentication authRequest) {
		Assert.notNull(ucCheckToken, "ucCheckToken");
		Assert.notNull(authRequest, "authRequest");
		StringBuilder sbRawMac = new StringBuilder();
		sbRawMac.append(authRequest.getNonce());
		sbRawMac.append("\n");
		sbRawMac.append(authRequest.getHttpMethod().toUpperCase());
		sbRawMac.append("\n");
		sbRawMac.append(authRequest.getRequestUri());
		sbRawMac.append("\n");
		sbRawMac.append(authRequest.getHost());
		sbRawMac.append("\n");
		String hmac256 = encryptHMac256(sbRawMac.toString(), ucCheckToken.getMacKey());
		String md5 = md5(sbRawMac.toString(), ucCheckToken.getMacKey());
		String mac = authRequest.getMac();
		String algorithm = WafProperties.getProperty("token.algorithm");
		boolean verify = true;
		if("md5".equalsIgnoreCase(algorithm) && !mac.equalsIgnoreCase(md5)){
			verify = false;
		}
		else if("hmac256".equalsIgnoreCase(algorithm) && !mac.equalsIgnoreCase(hmac256)){
			verify = false;
		}
		else if (!mac.equalsIgnoreCase(hmac256) && !mac.equalsIgnoreCase(md5)) {
			verify = false;
		}
		if(!verify){
			logger.error(authRequest.getMac() + ":" + sbRawMac.toString());
			throw new AuthenticationException("授权校验失败");
		}
		return ucCheckToken;
	}

	public static String encryptHMac256(String content, String key) {
		Assert.notNull(content, "content");
		Assert.notNull(key, "key");
		// 还原密钥
		SecretKey secretKey = new SecretKeySpec(key.getBytes(), "HmacSHA256");
		// 实例化Mac
		Mac mac = null;
		try {
			mac = Mac.getInstance(secretKey.getAlgorithm());
			mac.init(secretKey);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		// 执行消息摘要
		byte[] digest = mac.doFinal(content.getBytes());
		return new String(Base64.encode(digest));
	}

	/**
	 * MD5加密
	 * @param content
	 * @param key
	 * @return
	 */
	public static String md5(String content, String key) {

		try {
			MessageDigest messageDigest = MessageDigest.getInstance("MD5");
			if (key != null && !"".equals(key)) {
				content =  content + "{" + key.toString() + "}";
			}
			byte[] digest = messageDigest.digest(content.getBytes("utf-8"));
			return new String(Hex.encode(digest));
		} catch (NoSuchAlgorithmException | UnsupportedEncodingException e) {
			logger.error(e.getMessage(), e);
		}
		return null;

	}


}