package com.crawler.uc.utils;


import com.crawler.uc.authens.BaseUserInfo;
import com.crawler.uc.authens.UserRole;
import com.crawler.uc.config.UcConstant;
import com.crawler.uc.push.EventPushBody;
import com.crawler.uc.push.UcPusher;
import com.crawler.uc.push.event.KickEvent;
import com.crawler.waf.config.WafProperties;
import com.crawler.waf.security.authens.OauthAccessToken;
import com.crawler.waf.support.Constants;
import com.crawler.waf.utils.Encrypt;
import com.crawler.waf.utils.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Random;

/**
 * 用户中心模拟器
 * @author Rubekid
 *
 * 2017年5月15日 上午11:02:21
 */
public class UcSimulator {
	
	private static final Logger logger = LoggerFactory.getLogger(UcSimulator.class);
	
	private static final String APP_ID = WafProperties.getProperty("app.id", "app");

	/**
	 * token保存前缀
	 */
	private static final String UC_TOKEN =  APP_ID + ":uc:token";
	
	/**
	 * 用户前缀
	 */
	private static final String UC_USER = APP_ID + ":uc:user";
	
	/**
	 * ACCESS_TOKEN 前缀
	 */
	private static final String ACCESS_TOKEN = "access";
	
	/**
	 * 映射 前缀
	 */
	private static final String MAPPING = "mapping";
	
	/**
	 * REFRESH_TOKEN 前缀
	 */
	private static final String REFRESH_TOKEN = "refresh";
	
	
	/**
	 * 授权码
	 */
	private static final String AUTH_CODE = "code";
	
	/**
	 * 授权码有效期
	 */
	public static final int CODE_EXPIRE_MINUTE = WafProperties.getPropertyForInteger("code_expire_minute", "5");
	
	/**
	 * 过期时间 分钟
	 */
	private static final int TOKEN_TIMEOUT = WafProperties.getPropertyForInteger("token.timeout", "7200");
	
	/**
	 *  管理员过期时间 分钟
	 */
	private static final int ADMIN_TOKEN_TIMEOUT = WafProperties.getPropertyForInteger("token.admin.timeout", "604800");
	
	/**
	 * 保存授权码
	 * @param code
	 */
	public static void saveCode(String code, Long userId){
		int seconds = 60 * CODE_EXPIRE_MINUTE;
		UcRedisManager.set(codeKey(code), userId, seconds);
	}
	
	/**
	 * 获取授权码
	 * @param code
	 */
	public static Long findCode(String code){
		return UcRedisManager.get(codeKey(code), Long.class);
	}
	
	/**
	 * 移除授权码
	 * @param code
	 */
	public static void removeCode(String code){
		UcRedisManager.del(codeKey(code));
	}


	/**
	 * bearerToken
	 * @return
	 */
	public static OauthAccessToken generateBearerToken(){
		String tokenUserId = String.valueOf(0L);
		OauthAccessToken oauthAccessToken = new OauthAccessToken();
		String accessToken = generateToken(tokenUserId);
		String refreshToken = generateToken(tokenUserId);
		Date expiresAt = new Date(System.currentTimeMillis() + TOKEN_TIMEOUT * 60 * 1000);
		oauthAccessToken.setAccessToken(accessToken);
		oauthAccessToken.setRefreshToken(refreshToken);
		oauthAccessToken.setExpiresAt(expiresAt);
		oauthAccessToken.setUserId(tokenUserId);
		oauthAccessToken.setServerTime(new Date());
		UcSimulator.set(Constants.AUTHORIZATION_TYPE_BEARER, oauthAccessToken);
		return oauthAccessToken;
	}

	/**
	 * 生成token
	 * @param userId
	 * @param type
	 * @return
	 */
	public static OauthAccessToken generate(Long userId, String type){
		String tokenUserId = String.valueOf(userId);
		OauthAccessToken oauthAccessToken = new OauthAccessToken();
		String accessToken = generateToken(tokenUserId);
		String refreshToken = generateToken(tokenUserId);
		Date expiresAt = new Date(System.currentTimeMillis() + TOKEN_TIMEOUT * 60 * 1000);
		oauthAccessToken.setAccessToken(accessToken);
		oauthAccessToken.setRefreshToken(refreshToken);
		oauthAccessToken.setExpiresAt(expiresAt);
		oauthAccessToken.setUserId(tokenUserId);
		oauthAccessToken.setServerTime(new Date());
		if(Constants.AUTHORIZATION_TYPE_MAC.equals(type)){
			String macKey = getRandomString(10);
			oauthAccessToken.setMacKey(macKey);
		}
		UcSimulator.set(type, oauthAccessToken);
		return oauthAccessToken;
	}
	
	/**
	 * 踢掉
	 * @param userId
	 */
	public static void kick(String type, Long userId){
		String accessToken = UcRedisManager.get(mappingKey(type, userId));
		if(!StringUtil.isNullOrEmpty(accessToken)){
			remove(type, accessToken);
			KickEvent event = new KickEvent(type, accessToken);
			EventPushBody body = new EventPushBody(UcConstant.PUSH_EVENT_KICK, event);
			UcPusher.send(body);
		}
	}
	
	/**
	 * 设置token
	 * @param type
	 * @param oauthAccessToken
	 */
	public static void set(String type, OauthAccessToken oauthAccessToken){
		Long userId = Long.parseLong(oauthAccessToken.getUserId());
		boolean needKick = !UcConstant.MULTI_DEVICE;
		
		if(!needKick){
			List<UserRole> userRoles = getRoleList(oauthAccessToken.getUserId());
			for(UserRole userRole : userRoles){
				if(UcConstant.ADMIN_ROLES.contains(userRole.getRoleName())){
					needKick = true;
					oauthAccessToken.setExpiresAt(new Date(System.currentTimeMillis() + ADMIN_TOKEN_TIMEOUT * 1000));
					break;
				}
			}
		}
		
		if(needKick){
			kick(type, userId);
		}
		
		int seconds = (int) ((oauthAccessToken.getExpiresAt().getTime() - System.currentTimeMillis()) / 1000);
		//永不过期返回前端过期时间为100年后
		if(seconds == 0){
			oauthAccessToken.setExpiresAt(new Date(System.currentTimeMillis() + 100L * 365 * 24 * 60 * 60 * 1000));
		}
		UcRedisManager.set(accessTokenKey(type, oauthAccessToken.getAccessToken()), oauthAccessToken, seconds);
		UcRedisManager.set(refreshTokenKey(type, oauthAccessToken.getRefreshToken()), oauthAccessToken.getAccessToken(), seconds);
		UcRedisManager.set(mappingKey(type, userId), oauthAccessToken.getAccessToken(), seconds);
	}
	
	/**
	 * 获取token
	 * @param type
	 * @param accessToken
	 * @return
	 */
	public static OauthAccessToken get(String type, String accessToken){
		return UcRedisManager.get(accessTokenKey(type, accessToken), OauthAccessToken.class);
	}
	
	/**
	 * 通过refresh获取Token
	 * @param type
	 * @param refreshToken
	 * @return
	 */
	public static OauthAccessToken findByRefreshToken(String type, String refreshToken){
		String accessToken = UcRedisManager.get(refreshTokenKey(type, refreshToken));
		if(StringUtil.isNullOrEmpty(accessToken)){
			return null;
		}
		return get(type, accessToken);
	}
	
	/**
	 * 删除
	 * @param type
	 * @param accessToken
	 */
	public static void remove(String type, String accessToken){
		OauthAccessToken oauthAccessToken = get(type, accessToken);
		UcRedisManager.del(accessTokenKey(type, accessToken));
		if(oauthAccessToken != null){
			UcRedisManager.del(refreshTokenKey(type, oauthAccessToken.getRefreshToken()));			
		}
		// 删除用户信息
		//UcRedisManager.del(userKey(oauthAccessToken.getUserId()));
	}
	
	/**
	 * 注册用户
	 * @param userInfo
	 */
	public static void register(BaseUserInfo userInfo){
		setUser(userInfo);
	}
	
	/**
	 * 保存用户
	 * @param userInfo
	 */
	public static void setUser(BaseUserInfo userInfo){
		String key = userKey(userInfo.getUserId());
		UcRedisManager.set(key, userInfo);
	}
	
	public static void updateUser(BaseUserInfo userInfo){
		setUser(userInfo);
	}
	
	
	/**
	 * 获取用户信息
	 * @param userIdStr
	 * @return
	 */
	public static BaseUserInfo getUserInfo(String userIdStr){
		Long userId = Long.parseLong(userIdStr);
		String key = userKey(userId);
		BaseUserInfo userInfo = UcRedisManager.get(key, BaseUserInfo.class);
		logger.info("\n用户信息{}: {}", userId,  UcRedisManager.toJSONString(userInfo));
		if(userInfo == null){
			userInfo = new BaseUserInfo();
			userInfo.setUserId(userId);
		}
		return userInfo;
	}
	
	/**
	 * 获取用户信息
	 * @param userId
	 * @param clazz
	 * @return
	 */
	public static <T extends BaseUserInfo> T getUserInfo(Long userId, Class<T> clazz){
		String key = userKey(userId);
		T userInfo = UcRedisManager.get(key, clazz);
		return userInfo;
	}
	
	/**
	 * 获取用户角色
	 * @param userId
	 * @return
	 */
	public static List<UserRole> getRoleList(String userId){
		List<UserRole> roles = new ArrayList<UserRole>();
		BaseUserInfo userInfo = getUserInfo(userId);
		if(userInfo.getRole() != null){
			UserRole userRole = new UserRole();
			userRole.setRoleName(userInfo.getRole());
			roles.add(userRole);
		}
		return roles;
	}
	
	/**
	 * 用户 key
	 * @param userId
	 * @return
	 */
	private static String userKey(Object userId){
		return String.format("%s:%s", UC_USER, userId);
	}
	
	/**
	 * Mapping key
	 * @param type
	 * @param userId
	 * @return
	 */
	private static String mappingKey(String type, Long userId){
		return String.format("%s:%s:%s:%s", UC_TOKEN, type, MAPPING, userId);
	}
	
	/**
	 * Access Token key
	 * @param type
	 * @param accessToken
	 * @return
	 */
	private static String accessTokenKey(String type, String accessToken){
		return String.format("%s:%s:%s:%s", UC_TOKEN, type, ACCESS_TOKEN, accessToken);
	}
	
	/**
	 * Refresh Token Key
	 * @param type
	 * @param refreshToken
	 * @return
	 */
	private static String refreshTokenKey(String type, String refreshToken){
		return String.format("%s:%s:%s:%s", UC_TOKEN, type, REFRESH_TOKEN , refreshToken);
	}
	
	/**
	 * 授权码 key	
	 * @param code
	 * @return
	 */
	private static String codeKey(String code){
		return String.format("%s:%s:%s", UC_TOKEN, AUTH_CODE, code);
	}
	

	/**
	 * 生成token
	 * @return
	 */
	private static String generateToken(String userId){
		return (Encrypt.shortMd5(userId) + Encrypt.SHA256(getRandomString(8))).toUpperCase();
	}

	/**
	 * 获取指定长度随机字符串
	 * 
	 * @param length
	 * @return
	 */
	private static String getRandomString(int length) {
		String CHARS = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
		StringBuffer sb = new StringBuffer();
		Random random = new Random();
		for (int i = 0; i < length; i++) {
			sb.append(CHARS.charAt(random.nextInt(CHARS.length())));
		}
		return sb.toString();
	}

}
