package utils;

import client.DadaApiResponse;
import client.DadaRequestClient;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import config.AppConfig;
import domain.merchant.MerchantAddModel;
import domain.merchant.ShopAddModel;
import domain.merchant.ShopQueryModel;
import domain.merchant.ShopUpdateModel;
import domain.order.OrderAddModel;
import domain.order.OrderQueryDeliverFeeModel;
import domain.order.OrderQueryModel;
import service.CityCodeService;
import service.merchant.MerchantAddService;
import service.merchant.ShopAddService;
import service.merchant.ShopQueryService;
import service.merchant.ShopUpdateService;
import service.order.*;

import java.util.*;

public class DaDaApiUtils {

    /**
     * 注册商户
     * @return
     */
    public static DadaApiResponse registerMerchant(MerchantAddModel model){
        // 1.初始化配置(isOnline表示是否测试环境)
        AppConfig appConfig = new AppConfig();

        // 3.初始化service
        MerchantAddService merchantAddService = new MerchantAddService(JSONUtil.toJson(model));

        // 4.初始化客户端
        DadaRequestClient dadaClient = new DadaRequestClient(merchantAddService, appConfig);
        return dadaClient.callRpc();
    }

    /**
     * 查询城市信息
     * @return
     */
    public static DadaApiResponse queryCityCode(Integer sourceId){
        // 1.初始化配置(isOnline表示是否测试环境)
        AppConfig appConfig = new AppConfig(sourceId);


        // 2.初始化service
        CityCodeService cityCodeService = new CityCodeService("");

        // 3.初始化客户端
        DadaRequestClient dadaClient = new DadaRequestClient(cityCodeService, appConfig);
        return dadaClient.callRpc();
    }

    /**
     * 城市映射
     * @param sourceId
     * @return
     */
    public static Map<String, String> getCityCodeMap(Integer sourceId){
        Map<String, String> map = new HashMap<String, String>();
        DadaApiResponse response = queryCityCode(sourceId);
        JSONArray array = (JSONArray)response.getResult();
        for(int i=0; i<array.size(); i++){
            JSONObject item = array.getJSONObject(i);
            String cityName = item.getString("cityName");
            String cityCode = item.getString("cityCode");
            map.put(cityName, cityCode);
        }
        return map;
    }


    /**
     * 新增门店
     * @param sourceId
     * @param shopAddModel
     * @return
     */
    public static DadaApiResponse addShop(Integer sourceId, ShopAddModel shopAddModel){
        // 1.初始化配置(isOnline表示是否测试环境)
        AppConfig appConfig = new AppConfig(sourceId);

        // 3.初始化service (门店新增比较特殊,是一个批量新增接口)
        List<ShopAddModel> shopAddList = new ArrayList<ShopAddModel>();
        shopAddList.add(shopAddModel);
        ShopAddService shopAddService = new ShopAddService(JSONUtil.toJson(shopAddList));


        // 4.初始化客户端
        DadaRequestClient dadaClient = new DadaRequestClient(shopAddService, appConfig);
        return dadaClient.callRpc();
    }

    /**
     * 更新门店
     * @param sourceId
     * @param model
     * @return
     */
    public static DadaApiResponse updateShop(Integer sourceId, ShopUpdateModel model){
        AppConfig appConfig = new AppConfig(sourceId);
        ShopUpdateService shopUpdateService = new ShopUpdateService(JSONUtil.toJson(model));
        DadaRequestClient dadaClient = new DadaRequestClient(shopUpdateService, appConfig);
        return dadaClient.callRpc();
    }


    /**
     * 查询门店
     * @param sourceId
     * @param originShopId
     * @return
     */
    public static DadaApiResponse queryShop(Integer sourceId, String originShopId){
        AppConfig appConfig = new AppConfig(sourceId);
        ShopQueryModel model = new ShopQueryModel();
        model.setOriginShopId(originShopId);
        ShopQueryService shopQueryService = new ShopQueryService(JSONUtil.toJson(model));
        DadaRequestClient dadaClient = new DadaRequestClient(shopQueryService, appConfig);
        return dadaClient.callRpc();
    }






    /**
     * 创建订单
     * @param sourceId
     * @param orderAddModel
     * @return
     *
     * 同步响应：{"code":0,"msg":"成功","result":{"insuranceFee":0.0,"distance":3135259.6686146017,"fee":12.0,"deliverFee":12.0,"tips":0.0},"status":"success"}
     * 异步数据：{"cancel_from":"0","cancel_reason":"","client_id":"977435770693085349","dm_id":"0","dm_mobile":"","dm_name":"","order_id":"1619751945130","order_status":"1","signature":"3e6fe7c78956176c6b8204236eb6f0ce","update_time":"1619751987"}
     */
    public static DadaApiResponse addOrder(Integer sourceId, OrderAddModel orderAddModel) {
        // 1.初始化配置(isOnline表示是否测试环境)
        AppConfig appConfig = new AppConfig(sourceId);

        // 2.初始化service
        OrderAddService orderService = new OrderAddService(orderAddModel.toJson());

        // 3.初始化客户端
        DadaRequestClient dadaClient = new DadaRequestClient(orderService, appConfig);
        return dadaClient.callRpc();


    }

    /**
     * 重发订单
     * @param sourceId
     * @param  orderAddModel
     * @return
     */
    public static DadaApiResponse reAddOrder(Integer sourceId, OrderAddModel orderAddModel) {
        // 1.初始化配置(isOnline表示是否测试环境)
        AppConfig appConfig = new AppConfig(sourceId);

        // 2.初始化service
        OrderReAddService orderService = new OrderReAddService(orderAddModel.toJson());

        // 3.初始化客户端
        DadaRequestClient dadaClient = new DadaRequestClient(orderService, appConfig);
        return dadaClient.callRpc();


    }


    /**
     * 查询运费
     * @param sourceId
     * @param queryModel
     * @return
     */
    public static DadaApiResponse queryDeliverFee(Integer sourceId, OrderQueryDeliverFeeModel queryModel) {
        // 1.初始化配置(isOnline表示是否测试环境)
        AppConfig appConfig = new AppConfig(sourceId);

        // 2.初始化service
        OrderQueryDeliverFeeService orderService = new OrderQueryDeliverFeeService(queryModel.toJson());

        // 4.初始化客户端
        DadaRequestClient dadaClient = new DadaRequestClient(orderService, appConfig);
        return dadaClient.callRpc();

    }

    /**
     * 模拟接单
     * @param sourceId
     * @param orderId
     * @return
     * 异步通知 {"cancel_from":"0","cancel_reason":"","client_id":"977435770694374042","dm_id":"666","dm_mobile":"13546670420","dm_name":"达达骑手","order_id":"DD202104301126442244","order_status":"2","signature":"fa12ecee3720a74504b1eb59f66b4b39","update_time":"1619753346"}
     */
    public static DadaApiResponse  acceptOrder(Integer sourceId, String orderId){
        // 1.初始化配置(isOnline表示是否测试环境)
        AppConfig appConfig = new AppConfig(sourceId);

        // 2.初始化model
        OrderQueryModel model = new OrderQueryModel();
        model.setOrderId(orderId);

        // 3.初始化service
        OrderAcceptService orderService = new OrderAcceptService(model.toJson());

        // 4.初始化客户端
        DadaRequestClient dadaClient = new DadaRequestClient(orderService, appConfig);
        return dadaClient.callRpc();
    }

    /**
     * 模拟取货
     * @param sourceId
     * @param orderId
     * @return
     */
    public static DadaApiResponse  fetchOrder(Integer sourceId, String orderId){
        // 1.初始化配置(isOnline表示是否测试环境)
        AppConfig appConfig = new AppConfig(sourceId);

        // 2.初始化model
        OrderQueryModel model = new OrderQueryModel();
        model.setOrderId(orderId);

        // 3.初始化service
        OrderFetchService orderService = new OrderFetchService(model.toJson());

        // 4.初始化客户端
        DadaRequestClient dadaClient = new DadaRequestClient(orderService, appConfig);
        return dadaClient.callRpc();
    }


    /**
     * 查询订单
     * @param orderId
     * @return
     * {"code":0,"msg":"成功","result":{"supplierName":"票务","supplierLat":"31.228624","supplierAddress":"厦门市软件园二期44号1808","finishTime":"","distance":3135259,"orderId":"DD202104301126442244","supplierLng":"121.587173","tips":0.0,"supplierPhone":"17759227187","transporterName":"","transporterLat":"","deliveryFee":12.0,"insuranceFee":0.0,"statusMsg":"待接单","transporterLng":"","createTime":"2021-04-30 11:43:03","cancelTime":"","transporterPhone":"","fetchTime":"","receiptUrl":"","actualFee":12.0,"orderFinishCode":"0","statusCode":1},"status":"success"}
     */
    public static DadaApiResponse  queryOrder(Integer sourceId, String orderId){
        // 1.初始化配置(isOnline表示是否测试环境)
        AppConfig appConfig = new AppConfig(sourceId);

        // 2.初始化model
        OrderQueryModel model = new OrderQueryModel();
        model.setOrderId(orderId);

        // 3.初始化service
        OrderQueryService orderService = new OrderQueryService(model.toJson());

        // 4.初始化客户端
        DadaRequestClient dadaClient = new DadaRequestClient(orderService, appConfig);
        return dadaClient.callRpc();
    }


    /**
     * 校验签名
     * @param json
     * @return
     */
    public static boolean checkSign(JSONObject json){
        List<String> list = new ArrayList<String>();
        list.add(json.getString("client_id"));
        list.add(json.getString("order_id"));
        list.add(json.getString("update_time"));
        Collections.sort(list);
        StringBuffer stringBuffer = new StringBuffer();
        for(String str : list) {
            stringBuffer.append(str);
        }
        String md5Str = Md5.encrypt(stringBuffer.toString());
        return md5Str.equalsIgnoreCase(json.getString("signature"));
    }

}
