/*
 * SPDX-License-Identifier: Apache-2.0
 * Copyright Blazebit
 */

package com.blazebit.persistence.view.metamodel;

import java.util.Comparator;
import java.util.Map;

/**
 * Instances of the type {@linkplain PluralAttribute} represent collection-valued attributes.
 *
 * @param <X> The type of the declaring entity view
 * @param <C> The type of the represented collection
 * @param <E> The element type of the represented collection
 * @author Christian Beikov
 * @since 1.0.0
 */
public interface PluralAttribute<X, C, E> extends Attribute<X, C> {

    /**
     * Returns the collection type.
     *
     * @return The collection type
     */
    public CollectionType getCollectionType();

    /**
     * Returns the element collection type.
     *
     * @return The element collection type
     * @since 1.5.0
     */
    public ElementCollectionType getElementCollectionType();

    /**
     * Returns the type representing the element type of the collection.
     *
     * @return The element type
     * @since 1.2.0
     */
    public Type<E> getElementType();

    /**
     * Returns the inheritance subtypes that should be considered for the elements of this plural attribute.
     * When the element type of the attribute is not a subview, this returns an empty set.
     *
     * @return The inheritance subtypes or an empty set
     * @since 1.2.0
     */
    public Map<ManagedViewType<? extends E>, String> getElementInheritanceSubtypeMappings();
    
    /**
     * Returns whether this collection is indexed or not.
     * 
     * @return true if the collection is indexed, false otherwise
     */
    public boolean isIndexed();
    
    /**
     * Returns whether this collection is ordered or not.
     * 
     * @return true if the collection is ordered, false otherwise
     */
    public boolean isOrdered();
    
    /**
     * Returns whether this collection is sorted or not.
     * 
     * @return true if the collection is sorted, false otherwise
     */
    public boolean isSorted();

    /**
     * Returns whether this collection is forcefully deduplicated on load or not.
     *
     * @return true if the collection is forcefully deduplicated on load, false otherwise
     * @since 1.3.0
     */
    public boolean isForcedUnique();
    
    /**
     * Returns the comparator that should be used for sorting.
     * Returns null if no sorting is used or the natural sort order should be used.
     * 
     * @return the comparator that should be used for sorting
     */
    public Comparator<?> getComparator();
    
    /**
     * Returns the comparator class that should be used for sorting.
     * Returns null if no sorting is used or the natural sort order should be used.
     * 
     * @return the comparator class that should be used for sorting
     */
    public Class<Comparator<?>> getComparatorClass();

    /**
     * Returns whether the element collection is ordered or not.
     *
     * @return true if the element collection is ordered, false otherwise
     * @since 1.6.0
     */
    public boolean isElementCollectionOrdered();

    /**
     * Returns whether the element collection is sorted or not.
     *
     * @return true if the element collection is sorted, false otherwise
     * @since 1.6.0
     */
    public boolean isElementCollectionSorted();

    /**
     * Returns whether the element collection is forcefully deduplicated on load or not.
     *
     * @return true if the element collection is forcefully deduplicated on load, false otherwise
     * @since 1.6.0
     */
    public boolean isElementCollectionForcedUnique();

    /**
     * Returns the comparator that should be used for sorting in the element collection.
     * Returns null if no sorting is used or the natural sort order should be used.
     *
     * @return the comparator that should be used for sorting in the element collection
     * @since 1.6.0
     */
    public Comparator<?> getElementCollectionComparator();

    /**
     * Returns the comparator class that should be used for sorting in the element collection.
     * Returns null if no sorting is used or the natural sort order should be used.
     *
     * @return the comparator class that should be used for sorting in the element collection
     * @since 1.6.0
     */
    public Class<Comparator<?>> getElementCollectionComparatorClass();

    /**
     * The different collection types.
     */
    public static enum CollectionType {

        /**
         * Collection-valued attribute.
         */
        COLLECTION,
        /**
         * List-valued attribute.
         */
        LIST,
        /**
         * Map-valued attribute.
         */
        MAP,
        /**
         * Set-valued attribute.
         */
        SET;
    }

    /**
     * The different element collection types.
     * @since 1.5.0
     */
    public static enum ElementCollectionType {

        /**
         * Collection type.
         */
        COLLECTION,
        /**
         * List type.
         */
        LIST,
        /**
         * Set type.
         */
        SET,
        /**
         * SortedSet type.
         */
        SORTED_SET;
    }
}
